/*
 * @(#)MaachangConnect.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect ;

import java.net.InetAddress;

import org.maachang.connect.table.CompletionTable;
import org.maachang.connect.table.ConnectTable;
import org.maachang.connect.table.MaachangConnectTableFactory;
import org.maachang.connect.table.SendReceiveTable;
import org.maachang.connect.table.SendTable;
import org.maachang.connect.table.UnderReceiveTable;
import org.maachang.connect.thread.MaachangConnectThreadFactory;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.BaseUdpProtocol;
import org.maachang.commons.net.UdpProtocol;
import org.maachang.commons.resource.ResourceType;


/**
 * UDPプロトコルに再送機能を持たせたプロトコル(MaachangConnect).
 * <BR><BR>
 * このプロトコルは、UDPの特性を持ちつつ、TCP/IPのような
 * 再送機能を持ちます.
 * <BR>
 * MaachangConnectを利用する想定としては、１対複数との関係で接続する場合に
 * 利用することです.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class MaachangConnect extends AbstractMaachangConnect implements MaachangConnectInterface {
    
    /**
     * コンストラクタ.
     */
    public MaachangConnect() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @param port オープンポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public MaachangConnect( Cb32Bean bean,int port )
        throws InputException,AccessException {
        this.open( bean,port ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @param port オープンポート番号を設定します.
     * @param resType 対象のリソースタイプを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public MaachangConnect( Cb32Bean bean,int port,ResourceType resType )
        throws InputException,AccessException {
        this.open( bean,port,resType ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @param port オープンポート番号を設定します.
     * @param buffer 送受信バッファ長を設定します.
     * @param resType 対象のリソースタイプを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public MaachangConnect( Cb32Bean bean,int port,int buffer,ResourceType resType )
        throws InputException,AccessException {
        this.open( bean,port,buffer,resType ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @param port オープンポート番号を設定します.
     * @param bindAddress バインド先アドレスを設定します.
     * @param buffer 送受信バッファ長を設定します.
     * @param resType 対象のリソースタイプを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public MaachangConnect( Cb32Bean bean,int port,InetAddress bindAddress,int buffer,ResourceType resType )
        throws InputException,AccessException {
        this.open( bean,port,bindAddress,buffer,resType ) ;
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @exception AccessException アクセス例外.
     */
    public void open( Cb32Bean bean )
        throws AccessException {
        this.open( bean,-1,null,-1,null ) ;
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @param port オープンポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void open( Cb32Bean bean,int port )
        throws InputException,AccessException {
        this.open( bean,port,null,-1,null ) ;
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @param port オープンポート番号を設定します.
     * @param resType 対象のリソースタイプを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void open( Cb32Bean bean,int port,ResourceType resType )
        throws InputException,AccessException {
        this.open( bean,port,null,-1,resType ) ;
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @param port オープンポート番号を設定します.
     * @param buffer 送受信バッファ長を設定します.
     * @param resType 対象のリソースタイプを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void open( Cb32Bean bean,int port,int buffer,ResourceType resType )
        throws InputException,AccessException {
        this.open( bean,port,null,buffer,resType ) ;
    }
    
    /**
     * オープン処理.
     * <BR><BR>
     * MaachangConnectプロトコルをオープンします.
     * <BR>
     * @param bean 暗号条件を設定します.
     * @param port オープンポート番号を設定します.
     * @param bindAddress バインド先アドレスを設定します.
     * @param buffer 送受信バッファ長を設定します.
     * @param resType 対象のリソースタイプを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void open( Cb32Bean bean,int port,InetAddress bindAddress,int buffer,ResourceType resType )
        throws InputException,AccessException {
        
        if( port != -1 && ( port < 0 || port > 65535 ) ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        sync.create() ;
        
        if( buffer == -1 ) {
            buffer = DEFAULT_BUFFER ;
        }
        else if( buffer <= MIN_BUFFER ) {
            buffer = MIN_BUFFER ;
        }
        else if( buffer >= MAX_BUFFER ) {
            buffer = MAX_BUFFER ;
        }
        
        try {
            
            BaseUdpProtocol udp = new UdpProtocol() ;
            if( port == -1 ) {
                 udp.open( buffer ) ;
            }
            else if( bindAddress != null ) {
                udp.open( buffer,port,bindAddress ) ;
            }
            else {
                udp.open( buffer,port ) ;
            }
            
            SendReceiveTable sendReceiveTable = new SendReceiveTable() ;
            ConnectTable connectTable = new ConnectTable() ;
            SendTable sendTable = new SendTable() ;
            UnderReceiveTable underReceiveTable = null ;
            if( resType != null ) {
                underReceiveTable = new UnderReceiveTable( resType ) ;
            }
            else {
                underReceiveTable = new UnderReceiveTable() ;
            }
            CompletionTable completionTable = new CompletionTable() ;
            
            boolean cb32 = false ;
            String[] words = null ;
            if( bean != null ) {
                cb32 = bean.isCb32() ;
                words = bean.getWords() ;
            }
            
            MaachangConnectTableFactory tableFactory = new MaachangConnectTableFactory(
                cb32,words,sendReceiveTable,connectTable,sendTable,
                underReceiveTable,completionTable ) ;
            
            MaachangConnectThreadFactory factory = new MaachangConnectThreadFactory(
                udp,tableFactory ) ;
            
            synchronized( sync.get() ) {
                this.factory = factory ;
            }
            
        } catch( AccessException ae ) {
            throw ae ;
        } catch( Exception e ) {
            throw new AccessException( e ) ;
        }
        
    }
    
}

