/*
 * @(#)MaachangConnectTableFactory.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.table ;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;

/**
 * MaachangConnectテーブルファクトリ.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class MaachangConnectTableFactory {
    
    /**
     * 送受信テーブル.
     */
    private SendReceiveTable sendReceiveTable = null ;
    
    /**
     * コネクションテーブル.
     */
    private ConnectTable connectTable = null ;
    
    /**
     * 送信テーブル.
     */
    private SendTable sendTable = null ;
    
    /**
     * 受信中テーブル.
     */
    private UnderReceiveTable underReceiveTable = null ;
    
    /**
     * 受信完了テーブル.
     */
    private CompletionTable completionTable = null ;
    
    /**
     * スレッド終了通知.
     */
    private boolean exitThreadFlag = true ;
    
    /**
     * 暗号フラグ.
     */
    private boolean cb32Flag = false ;
    
    /**
     * 暗号プライベートコード.
     */
    private Cb32Table cb32Table = null ;
    
    /**
     * スレッド終了用フラグ.
     */
    private final Synchronized exitThreadSync = new Synchronized() ;
    
    /**
     * CB32定義用フラグ.
     */
    private final Synchronized cb32Sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    private MaachangConnectTableFactory() {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * テーブルオブジェクトを設定して、ファクトリオブジェクトを生成します.
     * <BR>
     * @param cb32 暗号フラグを設定します.
     * @param words 暗号プライベートワード群を設定します.
     * @param sendReceiveTable 対象の送受信テーブルを設定します.
     * @param connectTable コネクションテーブルを設定します.
     * @param sendTable 送信テーブルを設定します.
     * @param underReceiveTable 受信中テーブルを設定します.
     * @param completionTable 受信完了テーブルを設定します.
     * @exception InputException 入力例外.
     */
    public MaachangConnectTableFactory( boolean cb32,String[] words,
        SendReceiveTable sendReceiveTable,ConnectTable connectTable,
        SendTable sendTable,UnderReceiveTable underReceiveTable,CompletionTable completionTable )
        throws InputException {
        
        if( connectTable == null || sendTable == null ||
            underReceiveTable == null || completionTable == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        exitThreadSync.create() ;
        cb32Sync.create() ;
        
        this.sendReceiveTable = sendReceiveTable ;
        this.connectTable = connectTable ;
        this.sendTable = sendTable ;
        this.underReceiveTable = underReceiveTable ;
        this.completionTable = completionTable ;
        this.exitThreadFlag = false ;
        
        this.cb32Flag = cb32 ;
        if( cb32 == true ) {
            this.cb32Table = new Cb32Table( words ) ;
        }
        else {
            this.cb32Table = null ;
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクト破棄時の処理.
     */
    protected void finalize() throws Exception {
        this.sendReceiveTable = null ;
        this.connectTable = null ;
        this.sendTable = null ;
        this.underReceiveTable = null ;
        this.completionTable = null ;
        this.exitThreadFlag = true ;
        this.cb32Flag = false ;
        this.cb32Table = null ;
        
        exitThreadSync.clear() ;
        cb32Sync.clear() ;
    }

    /**
     * completionTable を取得.
     * <BR><BR>
     * @return completionTable が返されます.
     */
    public CompletionTable getCompletionTable() {
        return completionTable;
    }

    /**
     * connectTable を取得.
     * <BR><BR>
     * @return connectTable が返されます.
     */
    public ConnectTable getConnectTable() {
        return connectTable;
    }

    /**
     * sendTable を取得.
     * <BR><BR>
     * @return sendTable が返されます.
     */
    public SendTable getSendTable() {
        return sendTable;
    }

    /**
     * underReceiveTable を取得.
     * <BR><BR>
     * @return underReceiveTable が返されます.
     */
    public UnderReceiveTable getUnderReceiveTable() {
        return underReceiveTable;
    }

    /**
     * sendReceiveTable を取得.
     * <BR><BR>
     * @return sendReceiveTable が返されます.
     */
    public SendReceiveTable getSendReceiveTable() {
        return sendReceiveTable;
    }

    /**
     * exitThreadFlag を取得.
     * <BR><BR>
     * @return exitThreadFlag が返されます.
     */
    public boolean isExitThreadFlag() {
        synchronized( exitThreadSync.get() ) {
            return exitThreadFlag;
        }
    }

    /**
     * exitThreadFlag を設定.
     * <BR><BR>
     * @param exitThreadFlag exitThreadFlag を設定します.
     */
    public void setExitThreadFlag(boolean exitThreadFlag) {
        synchronized( exitThreadSync.get() ) {
            this.exitThreadFlag = exitThreadFlag;
        }
    }

    /**
     * cb32Flag を取得.
     * <BR><BR>
     * @return cb32Flag が返されます.
     */
    public boolean isCb32Flag() {
        synchronized( cb32Sync.get() ) {
            return cb32Flag;
        }
    }

    /**
     * cb32Table を取得.
     * <BR><BR>
     * @return cb32Table が返されます.
     */
    public Cb32Table getCb32Table() {
        synchronized( cb32Sync.get() ) {
            return cb32Table;
        }
    }

    /**
     * cb32Table を設定.
     * <BR><BR>
     * @param cb32Table を設定します.
     */
    public synchronized void setCb32Table( Cb32Table cb32Table ) {
        synchronized( cb32Sync.get() ) {
            if( cb32Table != null ) {
                this.cb32Flag = true ;
                this.cb32Table = cb32Table ;
            }
            else {
                this.cb32Flag = false ;
                this.cb32Table = null ;
            }
        }
    }

}

