/*
 * @(#)SendReceiveTable.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.table ;

import java.net.InetAddress;

import org.maachang.commons.util.RollObject;

/**
 * 送受信データ管理テーブル.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class SendReceiveTable {
    
    /**
     * デフォルトロール管理サイズ.
     */
    private static final int DEF_ROLL_LENGTH = 8192 ;
    
    /**
     * 最小ロール管理サイズ.
     */
    private static final int MIN_ROLL_LENGTH = 128 ;
    
    /**
     * 最大ロール管理サイズ.
     */
    private static final int MAX_ROLL_LENGTH = 32767 ;
    
    /**
     * 送信用データテーブル.
     */
    private RollObject sendRoll = null ;
    
    /**
     * 受信用データテーブル.
     */
    private RollObject receiveRoll = null ;
    
    
    /**
     * コンストラクタ.
     */
    public SendReceiveTable() {
        this( DEF_ROLL_LENGTH ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ロール長を設定して、オブジェクトを生成します.
     * <BR>
     * @param size 生成対象のロール長を設定します.<BR>
     *              設定可能な最小値は[128]です.<BR>
     *              設定可能な最大値は[32767]です.
     */
    public SendReceiveTable( int size ) {
        if( size == -1 ) {
            size = DEF_ROLL_LENGTH ;
        }
        if( size <= MIN_ROLL_LENGTH ) {
            size = MIN_ROLL_LENGTH ;
        }
        else if( size >= MAX_ROLL_LENGTH ) {
            size = MAX_ROLL_LENGTH ;
        }
        
        sendRoll = new RollObject( size ) ;
        receiveRoll = new RollObject( size ) ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理を実施.
     */
    protected void finalize() throws Exception {
        sendRoll = null ;
        receiveRoll = null ;
    }
    
    /**
     * 送信データ追加.
     * <BR><BR>
     * 送信データを追加します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param telegram 対象の電文を設定します.
     * @return boolean 追加結果が返されます.<BR>
     *                  [true]が返された場合、追加に成功しました.<BR>
     *                  [false]が返された場合、追加に失敗しました.
     */
    public boolean addSend( InetAddress addr,int port,byte[] telegram ) {
        boolean ret = true ;
        synchronized( sendRoll ) {
            if( sendRoll.isAdd() == false ) {
                ret = false ;
            }
            sendRoll.add( new SendReceiveBean( addr,port,telegram ) ) ;
        }
        return ret ;
    }
    
    /**
     * 送信データ追加.
     * <BR><BR>
     * 送信データを追加します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param telegram 対象の電文を設定します.
     * @param sendCount 送信回数を設定します.
     * @return boolean 追加結果が返されます.<BR>
     *                  [true]が返された場合、追加に成功しました.<BR>
     *                  [false]が返された場合、追加に失敗しました.
     */
    public boolean addSend( InetAddress addr,int port,byte[] telegram,int sendCount ) {
        boolean ret = true ;
        synchronized( sendRoll ) {
            if( sendRoll.isAdd() == false ) {
                ret = false ;
            }
            sendRoll.add( new SendReceiveBean( addr,port,telegram,sendCount ) ) ;
        }
        return ret ;
    }
    
    /**
     * 受信データ追加.
     * <BR><BR>
     * 受信データを追加します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param telegram 対象の電文を設定します.
     * @return boolean 追加結果が返されます.<BR>
     *                  [true]が返された場合、追加に成功しました.<BR>
     *                  [false]が返された場合、追加に失敗しました.
     */
    public boolean addReceive( InetAddress addr,int port,byte[] telegram ) {
        boolean ret = true ;
        synchronized( receiveRoll ) {
            if( receiveRoll.isAdd() == false ) {
                ret = false ;
            }
            else {
                receiveRoll.add( new SendReceiveBean( addr,port,telegram ) ) ;
            }
        }
        return ret ;
    }
    
    /**
     * 送信データを取得.
     * <BR><BR>
     * 送信データを取得します.
     * <BR>
     * @return SendReceiveBean 送信データが返されます.<BR>
     *                          [null]が返された場合、情報は存在しません.
     */
    public SendReceiveBean getSend() {
        SendReceiveBean ret = null ;
        synchronized( sendRoll ) {
            ret = ( SendReceiveBean )sendRoll.get() ;
        }
        return ret ;
    }
    
    /**
     * 受信データを取得.
     * <BR><BR>
     * 受信データを取得します.
     * <BR>
     * @return SendReceiveBean 受信データが返されます.<BR>
     *                          [null]が返された場合、情報は存在しません.
     */
    public SendReceiveBean getReceive() {
        SendReceiveBean ret = null ;
        synchronized( receiveRoll ) {
            ret = ( SendReceiveBean )receiveRoll.get() ;
        }
        return ret ;
    }
    
    /**
     * 格納されている送信データ長を取得.
     * <BR><BR>
     * 格納されている送信データ長を取得します.
     * <BR>
     * @return int 送信データ数が返されます.
     */
    public synchronized int getSendSize() {
        int ret = -1 ;
        synchronized( sendRoll ) {
            ret = sendRoll.getSize() ;
        }
        return ret ;
    }
    
}

