/*
 * @(#)SendTable.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.table ;

import org.maachang.commons.util.BigTable;
import org.maachang.commons.util.array.ObjectArray;

/**
 * コネクションが確立した後に設定される送信準備テーブルです.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class SendTable extends BaseTable {
    
    /**
     * デフォルトコネクションタイムアウト.
     * 15sec.
     */
    private static final long DEF_TIMEOUT = 15000L ;
    
    /**
     * 最小コネクションタイムアウト.
     * 5sec.
     */
    private static final long MIN_TIMEOUT = 5000L ;
    
    /**
     * 最大コネクションタイムアウト.
     * 60sec.
     */
    private static final long MAX_TIMEOUT = 60000L ;
    
    /**
     * インデックステーブル.
     */
    private BigTable indexTable = null ;
    
    /**
     * データタイムアウト.
     */
    private long timeout = -1L ;
    
    /**
     * コンストラクタ.
     */
    public SendTable() {
        this( DEF_TIMEOUT ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * タイムアウト値を設定して、送信テーブルを生成します.
     * <BR>
     * @param timeout コネクションタイムアウト値を設定します.<BR>
     *                 設定可能な最小値は[5000]ミリ秒です.<BR>
     *                 設定可能な最大値は[60000]ミリ秒です.
     */
    public SendTable( long timeout ) {
        
        if( timeout == -1L ) {
            timeout = DEF_TIMEOUT ;
        }
        
        if( timeout <= MIN_TIMEOUT ) {
            timeout = MIN_TIMEOUT ;
        }
        else if( timeout >= MAX_TIMEOUT ) {
            timeout = MAX_TIMEOUT ;
        }
        this.table = new ObjectArray() ;
        this.indexTable = new BigTable() ;
        this.timeout = timeout ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理です.
     */
    protected void finalize() throws Exception {
        this.table = null ;
        this.indexTable = null ;
        this.timeout = -1L ;
    }
    
    /**
     * 送信データを設定.
     * <BR><BR>
     * コネクションが確立された送信データを設定します.
     * <BR>
     * @param connectBean 対象の送信データを設定します.
     */
    public synchronized void putSendData( ConnectBean connectBean ) {
        if( connectBean != null ) {
            connectBean.updateTime() ;
            connectBean.setStartSendFlag( true ) ;
            table.add( connectBean ) ;
            indexTable.add( connectBean.getId(),connectBean ) ;
        }
    }
    
    /**
     * 送信データを削除.
     * <BR><BR>
     * 送信データをテーブルから削除します.<BR>
     * 主な利用としては、データ送信完了時において、このメソッドを
     * 呼び出します.<BR>
     * そのため、送信処理を１度でも行わない限り(isStartSendFlag=true)
     * 削除できません.
     * <BR>
     * @param id 削除対象のコネクションIDを設定します.
     * @return ConnectBean 削除されたコネクションBeanが返されます.<BR>
     *                     [null]が返された場合、情報は存在しません.
     */
    public synchronized ConnectBean removeSendData( long id ) {
        int num = this.getNumber( id ) ;
        if( num != -1 ) {
            ConnectBean bean = ( ConnectBean )table.remove( num ) ;
            if( bean != null ) {
                indexTable.remove( id ) ;
                if( bean.isStartSendFlag() == true ) {
                    return bean ;
                }
            }
        }
        return null ;
    }
    
    /**
     * 指定条件がタイムアウトしている場合、送信データを削除.
     * <BR><BR>
     * 指定条件がタイムアウトしている場合、送信データを削除します.<BR>
     * または、送信完了フラグがONの場合も同様に削除します.
     * <BR>
     * @param num 項番を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                  [true]が返された場合、タイムアウトしています.
     *                  [false]が返された場合、タイムアウトしていません.
     */
    public synchronized boolean removeSendDataByTimeout( int num ) {
        if( num < 0 || table.size() <= num ) {
            return false ;
        }
        
        ConnectBean bean = ( ConnectBean )table.get( num ) ;
        if( bean != null && bean.isStartSendFlag() == true ) {
            if( bean.isEndSendFlag() == true ||
                bean.getCreateTime() + timeout <= System.currentTimeMillis() ) {
                table.remove( num ) ;
                indexTable.remove( bean.getId() ) ;
                return true ;
            }
        }
        
        return false ;
    }
    
    /**
     * 指定コネクションBeanを送信開始に設定.
     * <BR><BR>
     * 指定コネクションBeanを送信開始に設定します.
     * <BR>
     * @param connectBean 送信開始に設定するコネクションBeanを設定します.
     */
    public synchronized void setStartSend( ConnectBean connectBean ) {
        if( connectBean == null && connectBean.isStartSendFlag() == true ) {
            return ;
        }
        if( this.getNumber( connectBean.getId() ) != -1 ) {
            connectBean.setStartSendFlag( true ) ;
            connectBean.updateTime() ;
        }
    }
    
    /**
     * 送信待ちタイムアウト値を取得.
     * <BR><BR>
     * 指定されている送信待ちタイムアウト値を取得します.
     * <BR>
     * @return long 送信待ちタイムアウト値が返されます.
     */
    public synchronized long getTimeout() {
        return timeout ;
    }
    
    /**
     * コネクションIDを取得.
     * <BR><BR>
     * 項番を設定して、コネクションIDを取得します.
     * <BR>
     * @param num 項番を設定します.
     * @return long コネクションIDが返されます.<BR>
     *               [-1L]が返された場合、情報は存在しません.
     */
    public synchronized long getConnectId( int num ) {
        if( num < 0 || table.size() <= num ) {
            return -1L ;
        }
        
        ConnectBean bean = ( ConnectBean )table.get( num ) ;
        if( bean != null ) {
            return bean.getId() ;
        }
        
        return -1L ;
    }
    
    /**
     * コネクションBeanを取得.
     * <BR><BR>
     * コネクションIDを設定して、コネクションBeanを取得します.
     * <BR>
     * @param id コネクションIDを設定します.
     * @return ConnectBean コネクションBeanが返されます.<BR>
     *                      [null]が返された場合、情報は存在しません.
     */
    public synchronized ConnectBean getConnectBean( long id ) {
        return ( ConnectBean )indexTable.get( id ) ;
        //return this.getConnectBean( this.getNumber( id ) ) ;
    }
    
    /**
     * コネクションBeanを取得.
     * <BR><BR>
     * 項番を設定して、コネクションBeanを取得します.
     * <BR>
     * @param num 項番を設定します.
     * @return ConnectBean コネクションBeanが返されます.<BR>
     *                      [null]が返された場合、情報は存在しません.
     */
    public synchronized ConnectBean getConnectBean( int num ) {
        if( num < 0 || table.size() <= num ) {
            return null ;
        }
        
        return ( ConnectBean )table.get( num ) ;
    }
    
    /**
     * 情報数を取得.
     * <BR><BR>
     * 格納されている情報数を取得します.
     * <BR>
     * @return int 情報数が返されます.
     */
    public synchronized int size() {
        return table.size() ;
    }
    
    /**
     * 格納コネクションID群を取得.
     * <BR><BR>
     * 格納コネクションID群を取得します.
     * <BR>
     * @return long[] 格納されているコネクションID群が返されます.
     */
    public synchronized long[] getConnectIds() {
        return indexTable.getNumbers() ;
    }
    
    /**
     * 対象コネクションIDが存在するかチェック.
     * <BR><BR>
     * 指定したコネクションIDが存在するかチェックします.
     * <BR>
     * @param id 対象のコネクションIDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                  [true]が返された場合、情報は存在します.<BR>
     *                  [false]が返された場合、情報は存在しません.
     */
    public synchronized boolean isConnectId( long id ) {
        return indexTable.isData( id ) ;
        //int num = this.getNumber( id ) ;
        //if( num != -1 ) {
        //    return true ;
        //}
        //return false ;
    }
}

