/*
 * @(#)MaachangConnectBaseThread.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.thread ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.connect.table.MaachangConnectSync;
import org.maachang.connect.table.MaachangConnectTableFactory;

import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.net.BaseUdpProtocol;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;

/**
 * MaachangConnectプロトコル基本スレッド.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class MaachangConnectBaseThread extends ExecutionThread {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( MaachangConnectBaseThread.class ) ;
    
    
    /**
     * スレッド項番.
     */
    private int threadNum = -1 ;
    
    /**
     * UDPプロトコル.
     */
    private BaseUdpProtocol udp = null ;
    
    /**
     * テーブルファクトリ.
     */
    private MaachangConnectTableFactory tableFactory = null ;
    
    /**
     * 処理同期オブジェクト.
     */
    private MaachangConnectSync execSynchronized = null ;
    
    /**
     * 実行処理.
     */
    private MaachangConnectBaseExecution execution = null ;
    
    /**
     * コネクションアドレス.
     */
    private ConnectAddress connectAddress = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    /**
     * 暗号解析時のコード.
     */
    private final String[] cb32Word = new String[ 1 ] ;
    
    
    /**
     * コンストラクタ.
     */
    private MaachangConnectBaseThread(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * タイムアウト監視対象のテーブルFactoryを設定します.
     * <BR>
     * @param threadNum スレッド項番を設定されます.
     * @param execution 実行オブジェクトを設定します.
     * @param udp 対象のUDPプロトコルを設定されます.
     * @param tableFactory テーブルFactoryを設定されます.
     * @param execSyncronized 同期オブジェクトが設定されます.
     */
    public MaachangConnectBaseThread( int threadNum,MaachangConnectBaseExecution execution,BaseUdpProtocol udp,
        MaachangConnectTableFactory tableFactory,MaachangConnectSync execSyncronized ) {
        
        sync.create() ;
        
        try{
            
            this.threadNum = threadNum ;
            this.udp = udp ;
            this.tableFactory = tableFactory ;
            this.execSynchronized = execSyncronized ;
            this.execution = execution ;
            this.connectAddress = new ConnectAddress() ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        sync.clear() ;
        thread.clear() ;
        this.threadNum = -1 ;
        this.udp = null ;
        this.tableFactory = null ;
        this.execSynchronized = null ;
        this.execution = null ;
        this.connectAddress = null ;
    }
    
    /**
     * スレッド項番を取得.
     * <BR><BR>
     * スレッド項番を取得します.
     * <BR>
     * @return int スレッド項番が返されます.
     */
    public int getThreadNum() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = threadNum ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        // スレッド終了に合わせて、全体終了フラグをONに設定.
        tableFactory.setExitThreadFlag( true ) ;
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int no ;
        BaseUdpProtocol protocol = null ;
        MaachangConnectTableFactory factory = null ;
        MaachangConnectSync execSync = null ;
        MaachangConnectBaseExecution exec = null ;
        ConnectAddress connAddr = null ;
        
        try{
            
            synchronized( sync.get() ) {
                no = this.threadNum ;
                protocol = this.udp ;
                factory = this.tableFactory ;
                execSync = this.execSynchronized ;
                exec = this.execution ;
                connAddr = this.connectAddress ;
            }
            
            // スレッド終了通知がONの場合.
            if( factory.isExitThreadFlag() == true ) {
                LOG.error( "スレッド終了通知を検知しました" ) ;
                throw new ExecutionException( "スレッド終了通知を検知",
                    ExecutionException.LEVEL_STOP ) ;
            }
            
            // 実行処理.
            exec.execution( no,protocol,factory,execSync,connAddr,cb32Word ) ;
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
        }
        
    }
    
}

