/*
 * @(#)MqConfigImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.config.service ;

import java.net.InetAddress;

import org.maachang.commons.conf.BaseConfig;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.ReadSyntaxException;
import org.maachang.commons.io.FileAccessException;
import org.maachang.commons.io.IODef;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.config.MqConfig;
import org.maachang.queue.config.MqDefine;

/**
 * MaachangQコンフィグファイル読み込む処理.
 *
 * @version 2006/12/31
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
class MqConfigImple extends BaseConfig implements MqConfig {
    
    /**
     * デフォルトアドレス名.
     */
    private static final String DEFAULT_ADDRESS = "default" ;
    
    /**
     * コンフィグセクション : MaachangQ定義.
     */
    private static final String SECTION_FAST_MQ = "maachangq" ;
    
    /**
     * MaachangQ定義 : キャッシュディレクトリ.
     */
    private static final String CACHE_DIRECTORY = "cache-directory" ;
    
    /**
     * MaachangQ定義 : 構成情報セーブAndロードディレクトリ.
     */
    private static final String SAVE_AND_LOAD_DIRECTORY = "save-and-load-directory" ;
    
    /**
     * MaachangQ定義 : オートセーブ間隔.
     */
    private static final String AUTH_SAVE_TIMING = "auto-save-timing" ;
    
    
    /**
     * コンフィグセクション : MaachangQAccessと接続する定義.
     */
    private static final String SECTION_CONNECT = "connect" ;
    
    /**
     * MaachangQAccessと接続する定義 : 受付バインドポート.
     */
    private static final String BIND_PORT = "bind-port" ;
    
    /**
     * MaachangQAccessと接続する定義 : 受付バインドアドレス.
     */
    private static final String BIND_ADDRESS = "bind-address" ;
    
    /**
     * MaachangQAccessと接続する定義 : 受付SSLバインドポート.
     */
    private static final String SSL_BIND_PORT = "ssl-bind-port" ;
    
    /**
     * MaachangQAccessと接続する定義 : 受付SSLバインドアドレス.
     */
    private static final String SSL_BIND_ADDRESS = "ssl-bind-address" ;
    
    /**
     * MaachangQAccessと接続する定義 : 受付コネクション最大数.
     */
    private static final String MAX_ACCESS = "max-access" ;
    
    /**
     * MaachangQAccessと接続する定義 : アドミニ用バインドポート.
     */
    private static final String ADMIN_PORT = "admin-port" ;
    
    /**
     * MaachangQAccessと接続する定義 : アドミニ用バインドアドレス.
     */
    private static final String ADMIN_ADDRESS = "admin-address" ;
    
    /**
     * MaachangQAccessと接続する定義 : SSLアドミニ用バインドポート.
     */
    private static final String SSL_ADMIN_PORT = "ssl-admin-port" ;
    
    /**
     * MaachangQAccessと接続する定義 : SSLアドミニ用バインドアドレス.
     */
    private static final String SSL_ADMIN_ADDRESS = "ssl-admin-address" ;
    
    
    /**
     * コンフィグセクション : 送信チャネル定義.
     */
    private static final String SECTION_SEND_CHANNEL = "send-channel" ;
    
    /**
     * 送信チャネル定義 : 送信スレッド数.
     */
    private static final String SEND_THREAD = "thread" ;
    
    /**
     * 送信チャネル定義 : ハートビート間隔.
     */
    private static final String HEART_BEAT_TIMING = "heartbeat-timing" ;
    
    /**
     * 送信チャネル定義 : ハートビートコネクション不能判定値.
     */
    private static final String HEART_BEAT_CLOSE_COUNT = "heartbeat-close-count" ;
    
    
    /**
     * コンフィグセクション : 受信チャネル定義.
     */
    private static final String SECTION_RECEIVE_CHANNEL = "receive-channel" ;
    
    /**
     * 受信チャネル定義 : 受信コアスレッド数.
     */
    private static final String CORE_THREAD = "core-thread" ;
    
    /**
     * 受信チャネル定義 : 受信電文判別スレッド数.
     */
    private static final String SWITCH_THREAD = "switch-thread" ;
    
    /**
     * 受信チャネル定義 : 受信データ解析スレッド数.
     */
    private static final String DATA_ANALYSIS_THREAD = "data-analysis-thread" ;
    
    /**
     * 受信チャネル定義 : 受信ハートビート解析スレッド数.
     */
    private static final String HEART_BEAT_ANALYSIS_THREAD = "heartbeat-analysis-thread" ;
    
    /**
     * 受信チャネル定義 : 受信正常結果解析スレッド数.
     */
    private static final String SUCCESS_ANALYSIS_THREAD = "success-analysis-thread" ;
    
    /**
     * 受信チャネル定義 : 受信エラー結果解析スレッド数.
     */
    private static final String ERROR_ANALYSIS_THREAD = "error-analysis-thread" ;
    
    
    
    /**
     * コンストラクタ.
     */
    public MqConfigImple() {
        super() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * MQ定義ファイルを指定して生成.
     * <BR><BR>
     * MQ定義ファイルを指定して生成します.
     * <BR>
     * @param name 生成対象のファイル名を設定します.
     * @exception ExecutionException 実行例外.
     */
    public void create( String name )
        throws ExecutionException {
        
        try{
            
            super.open( name ) ;
            
        }catch( BaseException be ){
            this.clear() ;
            throw new ExecutionException( be ) ;
        }catch( Exception e ){
            this.clear() ;
            throw new ExecutionException( e ) ;
        }finally{
            name = null ;
        }
        
    }
    
    /**
     * オブジェクトクリア.
     * <BR><BR>
     * オブジェクト情報をクリアします.
     */
    public void clear() {
        
        super.close() ;
    }
    
    /**
     * キャッシュディレクトリを取得.
     * <BR><BR>
     * キャッシュディレクトリを取得します.<BR>
     * この情報はセクション名[maachangq]以下に
     * 設定されている[cache-directory]情報を取得します.
     * <BR>
     * @return String キャッシュディレクトリ名が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public String getCacheDirectory() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                MqConfigImple.SECTION_FAST_MQ,
                MqConfigImple.CACHE_DIRECTORY,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        if( ret == null ) {
            
            ret = IODef.trimFileName(
                IODef.convertEnvironmentByName(
                    new StringBuffer().
                    append( MqDefine.EMV_NAME ).
                    append( MqDefine.CACHE_DIR ).
                    toString() ) ) ;
            
        }
        
        return ret ;
    }
    
    /**
     * 構成情報セーブAndロードディレクトリ名を取得.
     * <BR><BR>
     * 構成情報セーブAndロードディレクトリ名を取得します.<BR>
     * この情報はセクション名[maachangq]以下に
     * 設定されている[save-and-load-directory]情報を取得します.
     * <BR>
     * @return String 構成情報セーブAndロードディレクトリ名が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public String getSaveAndLoadDirectory() {
        
        String ret = null ;
        
        try{
            
            ret = this.getString(
                MqConfigImple.SECTION_FAST_MQ,
                MqConfigImple.SAVE_AND_LOAD_DIRECTORY,
                0
            ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        if( ret == null ) {
            
            ret = IODef.trimFileName(
                IODef.convertEnvironmentByName(
                    new StringBuffer().
                    append( MqDefine.EMV_NAME ).
                    append( MqDefine.SAVE_DIR ).
                    toString() ) ) ;
            
        }
        
        return ret ;
    }
    
    /**
     * オートセーブ間隔を取得.
     * <BR><BR>
     * オートセーブ間隔を取得します.<BR>
     * この情報はセクション名[maachangq]以下に
     * 設定されている[auto-save-timing]情報を取得します.
     * <BR>
     * @return long オートセーブ間隔が返されます.<BR>
     *              情報が存在しない場合[-1]が返されます.
     */
    public long getAutoSaveTiming() {
        
        long ret = -1L ;
        
        try{
            
            ret = this.getBigNumber(
                    MqConfigImple.SECTION_FAST_MQ,
                    MqConfigImple.AUTH_SAVE_TIMING,
                    0 ) ;
            
        }catch( Exception t ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 通信バインドポートを取得.
     * <BR><BR>
     * 通信バインドポートを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[bind-port]情報を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return int バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getBindPort( int no ) {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.BIND_PORT,
                no
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        if( ret == -1 ) {
            ret = MaachangQAccessDefine.TELEGRAM_PORT ;
        }
        
        return ret ;
    }
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[bind-address]情報を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return InetAddress バインドアドレスが返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public InetAddress getBindAddress( int no ) {
        
        InetAddress ret = null ;
        
        if( no < 0 || no >= this.getBindSize() ) {
            return null ;
        }
        
        try{
            
            String name = this.getString(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.BIND_ADDRESS,
                no
            ) ;
            
            if( name != null && name.length() > 0 ) {
                if( DEFAULT_ADDRESS.equals( name.trim().toLowerCase() ) == true ) {
                    ret = null ;
                }
                else {
                    ret = InetAddress.getByName( name ) ;
                }
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * SSL通信バインドポートを取得.
     * <BR><BR>
     * SSL通信バインドポートを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[ssl-bind-port]情報を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return int バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSSLBindPort( int no ) {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.SSL_BIND_PORT,
                no
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        if( ret == -1 ) {
            ret = MaachangQAccessDefine.SSL_TELEGRAM_PORT ;
        }
        
        return ret ;
        
    }
    
    /**
     * SSLバインドアドレスを取得.
     * <BR><BR>
     * SSLバインドアドレスを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[ssl-bind-address]情報を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return InetAddress バインドアドレスが返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public InetAddress getSSLBindAddress( int no ) {
        
        InetAddress ret = null ;
        
        if( no < 0 || no >= this.getBindSize() ) {
            return null ;
        }
        
        try{
            
            String name = this.getString(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.SSL_BIND_ADDRESS,
                no
            ) ;
            
            if( name != null && name.length() > 0 ) {
                if( DEFAULT_ADDRESS.equals( name.trim().toLowerCase() ) == true ) {
                    ret = null ;
                }
                else {
                    ret = InetAddress.getByName( name ) ;
                }
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * バインド情報数を取得.
     * <BR><BR>
     * 設定されているバインド情報数が返されます.
     * <BR>
     * @return int バインド情報数が返されます.
     */
    public int getBindSize() {
        
        int ret = 0 ;
        
        try {
            
            ret = super.getLength(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.BIND_PORT ) ;
            
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 受付コネクション最大数を取得.
     * <BR><BR>
     * 受付コネクション最大数を取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[max-access]情報を取得します.
     * <BR>
     * @return int 受付コネクション最大数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getMaxAccess() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.MAX_ACCESS,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        if( ret == -1 ) {
            ret = MaachangQAccessDefine.DEFAULT_ACCESS_TELEGRAM_PORT ;
        }
        else if( ret >= MaachangQAccessDefine.MAX_ACCESS_TELEGRAM_PORT ) {
            ret = MaachangQAccessDefine.MAX_ACCESS_TELEGRAM_PORT ;
        }
        
        return ret ;
    }
    
    /**
     * アドミニ用バインドポートを取得.
     * <BR><BR>
     * アドミニ用バインドポートを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[admin-port]情報を取得します.
     * <BR>
     * @return int アドミニ用バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getAdminPort() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.ADMIN_PORT,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        if( ret == -1 ) {
            ret = MaachangQAccessDefine.ADMIN_PORT ;
        }
        
        return ret ;
    }
    
    /**
     * アドミニ用アドレスを取得.
     * <BR><BR>
     * アドミニ用アドレスを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[admin-address]情報を取得します.
     * <BR>
     * @return InetAddress アドミニ用アドレスが返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public InetAddress getAdminAddress() {
        
        InetAddress ret = null ;
        
        try{
            
            String name = this.getString(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.ADMIN_ADDRESS,
                0
            ) ;
            
            if( name != null && name.length() > 0 ) {
                if( DEFAULT_ADDRESS.equals( name.trim().toLowerCase() ) == true ) {
                    ret = null ;
                }
                else {
                    ret = InetAddress.getByName( name ) ;
                }
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * SSLアドミニ用バインドポートを取得.
     * <BR><BR>
     * SSLアドミニ用バインドポートを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[ssl-admin-port]情報を取得します.
     * <BR>
     * @return int アドミニ用バインドポートが返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSSLAdminPort() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.SSL_ADMIN_PORT,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        if( ret == -1 ) {
            ret = MaachangQAccessDefine.SSL_ADMIN_PORT ;
        }
        
        return ret ;
        
    }
    
    /**
     * SSLアドミニ用アドレスを取得.
     * <BR><BR>
     * SSLアドミニ用アドレスを取得します.<BR>
     * この情報はセクション名[connect]以下に
     * 設定されている[ssl-admin-address]情報を取得します.
     * <BR>
     * @return InetAddress アドミニ用アドレスが返されます.<BR>
     *                     情報が存在しない場合[null]が返されます.
     */
    public InetAddress getSSLAdminAddress() {
        
        InetAddress ret = null ;
        
        try{
            
            String name = this.getString(
                MqConfigImple.SECTION_CONNECT,
                MqConfigImple.SSL_ADMIN_ADDRESS,
                0
            ) ;
            
            if( name != null && name.length() > 0 ) {
                if( DEFAULT_ADDRESS.equals( name.trim().toLowerCase() ) == true ) {
                    ret = null ;
                }
                else {
                    ret = InetAddress.getByName( name ) ;
                }
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 送信スレッド数を取得.
     * <BR><BR>
     * 送信スレッド数を取得します.<BR>
     * この情報はセクション名[send-channel]以下に
     * 設定されている[thread]情報を取得します.
     * <BR>
     * @return int 送信スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSendThread() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_SEND_CHANNEL,
                MqConfigImple.SEND_THREAD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * ハートビート間隔を取得.
     * <BR><BR>
     * ハートビート間隔を取得します.<BR>
     * この情報はセクション名[send-channel]以下に
     * 設定されている[heartbeat-timing]情報を取得します.
     * <BR>
     * @return long ハートビート間隔が返されます.<BR>
     *              情報が存在しない場合[-1]が返されます.
     */
    public long getHearbeatTiming() {
        
        long ret = -1L ;
        
        try{
            
            ret = this.getBigNumber(
                MqConfigImple.SECTION_SEND_CHANNEL,
                MqConfigImple.HEART_BEAT_TIMING,
                0 ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * ハートビートコネクション不能判定値を取得.
     * <BR><BR>
     * ハートビートコネクション不能判定値を取得します.<BR>
     * この情報はセクション名[send-channel]以下に
     * 設定されている[heartbeat-close-count]情報を取得します.
     * <BR>
     * @return int ハートビートコネクション不能判定値が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getHeartBeatCloseCount() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_SEND_CHANNEL,
                MqConfigImple.HEART_BEAT_CLOSE_COUNT,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信コアスレッド数を取得.
     * <BR><BR>
     * 受信コアスレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[core-thread]情報を取得します.
     * <BR>
     * @return int 受信コアスレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getCoreThread() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_RECEIVE_CHANNEL,
                MqConfigImple.CORE_THREAD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信電文判別スレッド数を取得.
     * <BR><BR>
     * 受信電文判別スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[switch-thread]情報を取得します.
     * <BR>
     * @return int 受信電文判別スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSwitchThread() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_RECEIVE_CHANNEL,
                MqConfigImple.SWITCH_THREAD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信データ解析スレッド数を取得.
     * <BR><BR>
     * 受信データ解析スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[data-analysis-thread]情報を取得します.
     * <BR>
     * @return int 受信データ解析スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getDataAnalysisThread() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_RECEIVE_CHANNEL,
                MqConfigImple.DATA_ANALYSIS_THREAD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信ハートビート解析スレッド数を取得.
     * <BR><BR>
     * 受信ハートビート解析スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[heartbeat-analysis-thread]情報を取得します.
     * <BR>
     * @return int 受信ハートビート解析スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getHeartBeatAnalysisThread() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_RECEIVE_CHANNEL,
                MqConfigImple.HEART_BEAT_ANALYSIS_THREAD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信正常結果解析スレッド数を取得.
     * <BR><BR>
     * 受信正常結果解析スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[success-analysis-thread]情報を取得します.
     * <BR>
     * @return int 受信正常結果解析スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getSuccessAnalysisThread() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_RECEIVE_CHANNEL,
                MqConfigImple.SUCCESS_ANALYSIS_THREAD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 受信エラー結果解析スレッド数を取得.
     * <BR><BR>
     * 受信エラー結果解析スレッド数を取得します.<BR>
     * この情報はセクション名[receive-channel]以下に
     * 設定されている[error-analysis-thread]情報を取得します.
     * <BR>
     * @return int 受信エラー結果解析スレッド数が返されます.<BR>
     *             情報が存在しない場合[-1]が返されます.
     */
    public int getErrorAnalysisThread() {
        
        int ret = -1 ;
        
        try{
            
            ret = this.getNumber(
                MqConfigImple.SECTION_RECEIVE_CHANNEL,
                MqConfigImple.ERROR_ANALYSIS_THREAD,
                0
            ) ;
            
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の情報を文字列で取得.
     * <BR><BR>
     * 対象の情報を文字列で取得します.
     * <BR>
     * @param section 対象のセクション名を設定します.
     * @param key 対象のキー名を設定します.
     * @param no 対象のキー名に対する項番を設定します.
     * @return String 対象の条件をStringで取得した結果が返されます.
     * @exception InputException 入力例外.
     */
    public String get( String section,String key,int no ) {
        
        String ret = null ;
        
        try{
            
            ret = this.getString( section,key,no ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象条件数を取得.
     * <BR><BR>
     * 対象の条件の定義数を取得します.
     * <BR>
     * @return int 定義数が返されます.
     */
    public int size( String section,String key ) {
        return this.getLength( section,key ) ;
    }
    
    /**
     * 定義ファイルで必要なセクション名群が有効であるかチェック.
     * <BR><BR>
     * 定義ファイルで必要なセクション名群が有効であるかチェックします.
     * <BR>
     * @return boolean 有効であるかチェックします.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、有効ではありません.
     */
    public boolean isUseSection() {
        return (
            super.isSection( MqConfigImple.SECTION_FAST_MQ ) == false ||
            super.isSection( MqConfigImple.SECTION_CONNECT ) == false ||
            super.isSection( MqConfigImple.SECTION_SEND_CHANNEL ) == false ||
            super.isSection( MqConfigImple.SECTION_RECEIVE_CHANNEL ) == false
        ) ? false : true ;
    }
    
    /**
     * 対象セクション名存在チェック.
     * <BR><BR>
     * 対象のセクション名が存在するかチェックします.
     * @param section 対象のセクション名を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、セクション名は存在します.<BR>
     *                 [false]が返された場合、セクション名は存在しません.
     */
    public boolean isSection( String section ) {
        return super.isSection( section ) ;
    }
    
    
    
    
    /**
     * 対象コンフィグ情報を取得.
     * <BR><BR>
     * 対象のコンフィグ情報を取得します.
     * <BR>
     * @param name 対象のコンフィグ名を設定します.
     * @exception InputException 入力例外.
     * @exception FileAccessException ファイルアクセス例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    protected final void newInit( String name )
        throws InputException,FileAccessException,ReadSyntaxException {
        
        try{
            synchronized( m_sync.get() ){
                m_conf.create( name ) ;
                m_name = name ;
                m_isRead = true ;
            }
        }catch( InputException in ){
            this.clear() ;
            throw in ;
        }catch( FileAccessException fa ){
            this.clear() ;
            throw fa ;
        }catch( ReadSyntaxException rs ){
            this.clear() ;
            throw rs ;
        }
    }
    
}
