/*
 * @(#)ExecutionAdminQueueManager.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.admin ;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.admin.AdminQueueManagerBean;
import org.maachang.queue.access.protocol.admin.AdminQueueManagerProtocol;
import org.maachang.queue.access.status.QueueManagerStatus;
import org.maachang.queue.connect.ExecutionAdmin;
import org.maachang.queue.connect.admin.login.LoginSession;
import org.maachang.queue.connect.common.Telegram;
import org.maachang.queue.main.cache.MqCache;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.service.SaveAndLoadService;


/**
 * キューマネージャ電文処理.
 *
 * @version 2007/01/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionAdminQueueManager
    extends AbstractExecutionAdmin
    implements ExecutionAdmin {
    
    /**
     * コンストラクタ.
     */
    public ExecutionAdminQueueManager() {
        
    }
    
    /**
     * 管理者実行処理IDを取得.
     * <BR><BR>
     * 設定されている管理者実行処理IDを取得します.
     * <BR>
     * @return int 管理者実行処理IDが返されます.
     */
    public int getId() {
        return AdminQueueManagerProtocol.CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * 各管理者用実行処理を実装するためのメソッド.
     * <BR>
     * @param telegramType 処理対象の実行タイプを格納します.
     * @param connect コネクションオブジェクトが設定されます.
     * @param session 対象のログインセッションが設定されます.
     * @param id 対象の電文IDが設定されます.
     * @param telegram 受信された電文が設定されます.
     * @return boolean 処理結果が返されます.
     * @exception Exception 例外.
     */
    public boolean execution( int[] telegramType,LoginSession session,
        ConnectObject connect,int id,Telegram telegram )
        throws Exception {
        
        byte[] binary = telegram.getTelegram() ;
        telegram.clear() ;
        
        // 受信バイナリを解析.
        AdminQueueManagerBean bean = new AdminQueueManagerBean() ;
        AdminQueueManagerProtocol.analysisProtocol( bean,binary ) ;
        telegramType[ 0 ] = bean.getType() ;
        bean.setId( id ) ;
        binary = null ;
        
        switch( bean.getType() ) {
            case AdminQueueManagerProtocol.TYPE_ADD_QMANAGER :
                this.addQueueManager( session,connect,bean ) ;
                return true ;
            case AdminQueueManagerProtocol.TYPE_REMOVE_QMANAGER :
                this.removeQueueManager( session,connect,bean ) ;
                return true ;
            case AdminQueueManagerProtocol.TYPE_STOP_QMANAGER :
                this.stopQueueManager( session,connect,bean ) ;
                return true ;
            case AdminQueueManagerProtocol.TYPE_START_QMANAGER :
                this.startQueueManager( session,connect,bean ) ;
                return true ;
            case AdminQueueManagerProtocol.TYPE_GET_QMANAGER :
                this.getQueueManager( session,connect,bean ) ;
                return true ;
            case AdminQueueManagerProtocol.TYPE_GET_QMANAGER_NAMES :
                this.getQueueManagerNames( session,connect,bean ) ;
                return true ;
            case AdminQueueManagerProtocol.TYPE_GET_QMANAGER_SIZE :
                this.getQueueManagerSize( session,connect,bean ) ;
                return true ;
            case AdminQueueManagerProtocol.TYPE_IS_QMANAGER :
                this.isQueueManager( session,connect,bean ) ;
                return true ;
                
        }
        
        return false ;
        
    }
    
    /**
     * キューマネージャ追加.
     */
    public final void addQueueManager(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( QueueManagerFactory.getSync() ) {
            
            // キューマネージャが存在する場合.
            if( QueueManagerFactory.get( bean.getName() ) != null ) {
                throw new MaachangQException(
                    "キューマネージャ(" + bean.getName() +
                    ")は既に存在します",
                    MaachangQErrorCode.ERROR_USE_QUEUE_MANAGER ) ;
            }
            
            try {
                // キューマネージャ生成.
                QueueManagerFactory.create(
                    bean.getName(),bean.getMaxSector() ) ;
                
                // 状態セーブ.
                SaveAndLoadService.save(
                    SaveAndLoadService.SAVE_TYPE_MANAGER ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キューマネージャ(" + bean.getName() +
                    ")の追加に失敗",
                    MaachangQErrorCode.ERROR_ADD_QUEUE_MANAGER ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キューマネージャ追加" ) ;
        
    }
    
    /**
     * キューマネージャ削除.
     */
    public final void removeQueueManager(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( QueueManagerFactory.getSync() ) {
            
            this.getManager( bean ) ;
            
            try {
                // キューマネージャ削除.
                QueueManagerFactory.remove( true,bean.getName() ) ;
                
                // 状態セーブ.
                SaveAndLoadService.save(
                    SaveAndLoadService.SAVE_TYPE_MANAGER ) ;
                SaveAndLoadService.save(
                    SaveAndLoadService.SAVE_TYPE_QUEUE ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キューマネージャ(" + bean.getName() +
                    ")の削除に失敗",
                    MaachangQErrorCode.ERROR_REMOVE_QUEUE_MANAGER ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キューマネージャ削除" ) ;
        
    }
    
    /**
     * キューマネージャ停止.
     */
    public final void stopQueueManager(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( QueueManagerFactory.getSync() ) {
            
            QueueManager man = this.getManager( bean ) ;
            
            // キューマネージャステータスを計画停止.
            if( man.getState() == QueueManagerStatus.STATE_SUCCESS ) {
                try {
                    man.setState( QueueManagerStatus.STATE_PLANNED_STOP ) ;
                } catch( Exception e ) {
                    LOG.error( "error",e ) ;
                    throw new MaachangQException(
                        "キューマネージャ(" + bean.getName() +
                        ")の停止に失敗",
                        MaachangQErrorCode.ERROR_STOP_QUEUE_MANAGER ) ;
                }
            }
            else {
                throw new MaachangQException(
                    "キューマネージャの停止に失敗しました(" +
                    QueueManagerStatus.getStateByString( man.getState() ) +
                    ")",
                    MaachangQErrorCode.ERROR_NOT_STOP_QUEUE_MANAGER ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キューマネージャ停止" ) ;
        
    }
    
    /**
     * キューマネージャ停止から復元.
     */
    public final void startQueueManager(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( QueueManagerFactory.getSync() ) {
            
            QueueManager man = this.getManager( bean ) ;
            
            // キューマネージャステータスを計画停止から復元.
            if( man.getState() == QueueManagerStatus.STATE_PLANNED_STOP ) {
                try {
                    man.setState( QueueManagerStatus.STATE_STARTUP ) ;
                } catch( Exception e ) {
                    LOG.error( "error",e ) ;
                    throw new MaachangQException(
                        "キューマネージャ(" + bean.getName() +
                        ")の開始に失敗",
                        MaachangQErrorCode.ERROR_RESTART_QUEUE_MANAGER ) ;
                }
            }
            else {
                throw new MaachangQException(
                    "キューマネージャの開始に失敗しました(" +
                    QueueManagerStatus.getStateByString( man.getState() ) +
                    ")",
                    MaachangQErrorCode.ERROR_NOT_RESTART_QUEUE_MANAGER ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キューマネージャ開始" ) ;
        
    }
    
    /**
     * キューマネージャ情報取得.
     */
    public final void getQueueManager(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( QueueManagerFactory.getSync() ) {
            
            QueueManager man = this.getManager( bean ) ;
            
            try {
                // キューマネージャ情報を取得.
                MqCache cache = man.getMqCache() ;
                if( cache == null ) {
                    bean.setMaxSector( -1 ) ;
                    bean.setUseSector( -1 ) ;
                }
                else {
                    bean.setMaxSector( cache.getCacheSize() ) ;
                    bean.setUseSector( cache.getUseCacheSize() ) ;
                }
                
                bean.setState( man.getState() ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キューマネージャ(" + bean.getName() +
                    ")の取得に失敗",
                    MaachangQErrorCode.ERROR_GET_QUEUE_MANAGER ) ;
            }
            
        }
        
        this.sendQueueManager( session,connect,bean ) ;
        
    }
    
    /**
     * キューマネージャ名群取得.
     */
    public final void getQueueManagerNames(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( QueueManagerFactory.getSync() ) {
            
            try {
                String[] names = QueueManagerFactory.getNames() ;
                bean.setParams( names ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キューマネージャ名群取得に失敗",
                    MaachangQErrorCode.ERROR_GET_LIST_QUEUE_MANAGER ) ;
            }
            
        }
        
        this.sendQueueManager( session,connect,bean ) ;
        
    }
    
    /**
     * キューマネージャ数取得.
     */
    public final void getQueueManagerSize(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( QueueManagerFactory.getSync() ) {
            
            try {
                int size = QueueManagerFactory.size() ;
                bean.setSize( size ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キューマネージャ数取得に失敗",
                    MaachangQErrorCode.ERROR_GET_SIZE_QUEUE_MANAGER ) ;
            }
            
        }
        
        this.sendQueueManager( session,connect,bean ) ;
        
    }
    
    /**
     * キューマネージャ存在チェック.
     */
    public final void isQueueManager(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( QueueManagerFactory.getSync() ) {
            
            try {
                this.getManager( bean ) ;
            } catch( MaachangQException fa ) {
                throw fa ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キューマネージャ(" + bean.getName() +
                    ")の存在確認に失敗",
                    MaachangQErrorCode.ERROR_CHECK_QUEUE_MANAGER ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キューマネージャ存在" ) ;
    }
    
    /**
     * 戻り電文送信.
     */
    private final void sendQueueManager(
        LoginSession session,ConnectObject connect,AdminQueueManagerBean bean )
        throws Exception {
        
        byte[] binary = AdminQueueManagerProtocol.createProtocol(
            bean.getId(),session.isOwner(),bean ) ;
        
        connect.send( binary ) ;
        
    }
    
    /**
     * キューマネージャ取得.
     */
    private final QueueManager getManager( AdminQueueManagerBean bean )
        throws Exception {
        QueueManager man = QueueManagerFactory.get( bean.getName() ) ;
        
        // キューマネージャが存在しない場合.
        if( man == null ) {
            throw new MaachangQException(
                "キューマネージャ(" + bean.getName() +
                ")は存在しません",
                MaachangQErrorCode.ERROR_NOT_QUEUE_MANAGER ) ;
        }
        
        return man ;
    }
}

