/*
 * @(#)ServerConnectThread.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.common ;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.array.ObjectArray;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;

/**
 * サーバコネクションスレッド.
 *
 * @version 2007/01/14
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ServerConnectThread extends ExecutionThread {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ServerConnectThread.class ) ;
    
    /**
     * アクセプトタイムアウト.
     */
    private static final int ACCEPT_TIMEOUT = 500 ;
    
    /**
     * デフォルト未操作タイムアウト値.
     */
    private static final long DEFAULT_NO_WORK_TIMEOUT = 300000L ;
    
    /**
     * 最小未操作タイムアウト値.
     */
    private static final long MIN_NO_WORK_TIMEOUT = 60000L ;
    
    /**
     * 最大未操作タイムアウト値.
     */
    private static final long MAX_NO_WORK_TIMEOUT = 21600000L ;
    
    
    /**
     * 未操作タイムアウト.
     */
    private long noWorkTimeout = -1L ;
    
    /**
     * サーバソケット.
     */
    private ServerSocket server = null ;
    
    /**
     * クライアント接続処理.
     */
    private ExecutionReceiveClient client = null ;
    
    /**
     * クライアント管理.
     */
    private ObjectArray clientManager = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    private ServerConnectThread(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * サーバソケットを設定してオブジェクトを生成します.
     * <BR>
     * @param server 対象のサーバソケットを設定します.
     * @param client 対象のクライアントコネクション実行オブジェクトを設定します.
     */
    public ServerConnectThread( ServerSocket server,ExecutionReceiveClient client ) {
        this( DEFAULT_NO_WORK_TIMEOUT,server,client ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * サーバソケットを設定してオブジェクトを生成します.
     * <BR>
     * @param noWorkTimeout 未操作時のタイムアウト値を設定します.<BR>
     *                      設定可能な最小値は[60000](１分)です.<BR>
     *                      設定可能な最小値は[21600000](６時間)です.
     * @param server 対象のサーバソケットを設定します.
     * @param client 対象のクライアントコネクション実行オブジェクトを設定します.
     */
    public ServerConnectThread(
        long noWorkTimeout,ServerSocket server,ExecutionReceiveClient client ) {
        
        if( server == null || server.isClosed() == true || client == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( noWorkTimeout <= MIN_NO_WORK_TIMEOUT ) {
            noWorkTimeout = MIN_NO_WORK_TIMEOUT ;
        }
        if( noWorkTimeout >= MAX_NO_WORK_TIMEOUT ) {
            noWorkTimeout = MAX_NO_WORK_TIMEOUT ;
        }
        
        sync.create() ;
        
        try{
            
            server.setSoTimeout( ACCEPT_TIMEOUT ) ;
            this.noWorkTimeout = noWorkTimeout ;
            this.server = server ;
            this.client = client ;
            this.clientManager = new ObjectArray() ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        this.destroyClient() ;
        sync.clear() ;
        thread.clear() ;
        server = null ;
        client = null ;
        clientManager = null ;
    }
    
    /**
     * クライアント接続を全て破棄.
     * <BR><BR>
     * クライアント接続を全て破棄します.
     */
    public final void destroyClient() {
        if( clientManager != null ) {
            ClientConnectThread th = null ;
            int len = clientManager.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                th = ( ClientConnectThread )clientManager.get( i ) ;
                if( th != null ) {
                    th.destroy() ;
                }
                th = null ;
            }
            clientManager.clear() ;
        }
    }
    
    /**
     * 子スレッドオブジェクトを管理から破棄.
     * <BR><BR>
     * 子スレッドオブジェクトを管理から破棄します.
     * <BR>
     * @param obj 対象の子スレッドオブジェクトィを設定します.
     */
    protected final void remove( Object obj ) {
        if( obj != null ) {
            try {
                synchronized( sync.get() ) {
                    if( clientManager != null ) {
                        int len = clientManager.size() ;
                        for( int i = 0 ; i < len ; i ++ ) {
                            if( clientManager.get( i ) == obj ) {
                                clientManager.remove( i ) ;
                                break ;
                            }
                        }
                    }
                }
            } catch( Exception e ) {
            }
        }
    }
    
    /**
     * エラー発生時にその内容を接続先に返す.
     * <BR><BR>
     * エラー発生時にその内容を接続先に返します.
     * <BR>
     * @param id 対象のプロトコルIDを設定します.
     * @param telegramType 対象の電文タイプを設定します.
     * @param queueManager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param conn 対象のコネクションオブジェクトを設定します.
     * @param ex 対象の例外オブジェクトを設定します.
     */
    public static final void resultError( int id,int telegramType,
         String queueManager,String queue,int queueType,
         ConnectObject conn,Exception ex ) {
        
        try {
            
            int errorCode = -1 ;
            
            if( ex instanceof MaachangQException ) {
                errorCode = ( ( MaachangQException )ex ).getErrorCode() ;
            }
            else {
                errorCode = MaachangQErrorCode.UNKNOWN_ERROR ;
            }
            
            resultError( id,telegramType,queueManager,queue,queueType,
                conn,ex.getMessage(),errorCode ) ;
            
        } catch( Exception e ) {
            LOG.error( "エラーメッセージ送信中にエラーが発生",e ) ;
        }
        
    }
    
    /**
     * エラー発生時にその内容を接続先に返す.
     * <BR><BR>
     * エラー発生時にその内容を接続先に返します.
     * <BR>
     * @param id 対象のプロトコルIDを設定します.
     * @param telegramType 対象の電文タイプを設定します.
     * @param queueManager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param conn 対象のコネクションオブジェクトを設定します.
     * @param message 対象のメッセージ情報を設定します.
     * @param resultCode 対象の処理結果コードを設定します.
     */
    public static final void resultError( int id,int telegramType,
        String queueManager,String queue,int queueType,
        ConnectObject conn,String message,int resultCode ) {
        
        try {
            
            if( message == null ||
                ( message = message.trim() ).length() <= 0 ) {
                message = "no-message" ;
            }
            
            ServerConnectThread.sendRsult( id,ResultProtocol.TYPE_ERROR,
                telegramType,queueManager,queue,queueType,
                conn,message,resultCode ) ;
            
        } catch( Exception e ) {
            LOG.error( "エラーメッセージ送信中にエラーが発生",e ) ;
        }
        
    }
    
    /**
     * 処理結果をクライアントに送信.
     * <BR><BR>
     * 処理結果をクライアントに送信します.
     * <BR>
     * @param id 対象のプロトコルIDを設定します.
     * @param resultType 処理結果タイプを設定します.
     * @param telegramType 対象の電文タイプを設定します.
     * @param conn 対象のコネクションオブジェクトを設定します.
     * @param message 対象のメッセージ情報を設定します.
     * @param resultCode 対象の処理結果コードを設定します.
     */
    public static final void sendRsult( int id,int resultType,int telegramType,
        ConnectObject conn,String message,int resultCode ) {
        ServerConnectThread.sendRsult( id,resultType,
            telegramType,null,null,-1,
            conn,message,resultCode ) ;
    }
    
    /**
     * 処理結果をクライアントに送信.
     * <BR><BR>
     * 処理結果をクライアントに送信します.
     * <BR>
     * @param id 対象のプロトコルIDを設定します.
     * @param resultType 処理結果タイプを設定します.
     * @param telegramType 対象の電文タイプを設定します.
     * @param queueManager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param conn 対象のコネクションオブジェクトを設定します.
     * @param message 対象のメッセージ情報を設定します.
     * @param resultCode 対象の処理結果コードを設定します.
     */
    public static final void sendRsult( int id,int resultType,
        int telegramType,String queueManager,String queue,int queueType,
        ConnectObject conn,String message,int resultCode ) {
        
        try {
            
            if( queueManager == null || queueManager.length() <= 0 ) {
                queueManager = CommonProtocol.DUMMY ;
            }
            if( queue == null || queue.length() <= 0 ) {
                queue = CommonProtocol.DUMMY ;
                queueType = MaachangQAccessDefine.TYPE_SEND ;
            }
            if( message == null || ( message = message.trim() ).length() <= 0 ) {
                message = null ;
            }
            
            byte[] result = ResultProtocol.createProtocol(
                id,resultType,queueManager,queue,queueType,
                telegramType,resultCode,message ) ;
            
            conn.send( result ) ;
            
        } catch( Exception e ) {
            LOG.error( "処理結果送信中にエラーが発生",e ) ;
        }
        
    }
    
    /**
     * 現在接続数を取得.
     * <BR><BR>
     * 現在のクライアント接続数が返されます.
     * <BR>
     * @return int 接続数が返されます.
     */
    public final int getConnectSize() {
        int ret = 0 ;
        
        try {
            synchronized( sync.get() ) {
                ret = clientManager.size() ;
            }
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 平均通信速度を取得.
     * <BR><BR>
     * 平均通信速度を取得します.
     * <BR>
     * @return int 平均通信速度が返されます.
     */
    public final int getSpeed() {
        int ret = 0 ;
        
        try {
            synchronized( sync.get() ) {
                int len = clientManager.size() ;
                if( len > 0 ) {
                    long sp = 0L ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        sp += (
                            ( ClientConnectThread )clientManager.get( i )
                        ).getSpeed() ;
                    }
                    ret = ( int )( sp / ( long )len ) ;
                } 
                
            }
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        this.destroyClient() ;
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        long noWorkTimeout = -1L ;
        ServerSocket server = null ;
        ExecutionReceiveClient client = null ;
        ObjectArray clientManager = null ;
        Socket socket = null ;
        
        try{
            
            //UtilCom.idleTime() ;
            
            synchronized( sync.get() ) {
                noWorkTimeout = this.noWorkTimeout ;
                server = this.server ;
                client = this.client ;
                clientManager = this.clientManager ;
            }
            
            try {
                socket = server.accept() ;
            } catch( SocketTimeoutException tm ) {
                UtilCom.idleTime() ;
                return ;
            } catch( IOException io ) {
                throw new ExecutionException(
                    io,ExecutionException.LEVEL_STOP
                ) ;
            }
            
            if( socket != null && socket.isClosed() == false ) {
                ClientConnectThread th = new ClientConnectThread(
                    noWorkTimeout,this,socket,client ) ;
                clientManager.add( th ) ;
            }
            
        }catch( OutOfMemoryError mem ) {
            LOG.error( "OutOfMemoryError",mem ) ;
        }catch( NullPointerException nul ){
            LOG.error( "スレッドストップを検知",nul ) ;
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ) {
            LOG.error( "スレッドストップを検知",ee ) ;
            throw ee ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
        }
        
    }
    
}

