/*
 * @(#)ChannelFactory.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel ;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.sys.NamingManager;
import org.maachang.commons.util.NumberTable;
import org.maachang.commons.util.array.ObjectArray;
import org.maachang.queue.main.connect.ConnectFactory;

/**
 * チャネルファクトリ.
 *  
 * @version 2006/08/30
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class ChannelFactory
{
    
    /**
     * ネーミングマネージャ登録拡張子.
     */
    public static final String NAMING_PLUS = "@maachangq.channel.factory" ;
    
    /**
     * ネーミングマネージャ登録拡張子(チャネルID管理).
     */
    public static final String NAMING_CHANNEL_TABLE_PLUS = "CHANNEL_FACTORY@maachangq.channel.table" ;
    
    /**
     * ネーミングマネージャ登録拡張子(チャネル項番管理).
     */
    public static final String NAMING_CHANNEL_ARRAY_PLUS = "CHANNEL_FACTORY@maachangq.channel.array" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public ChannelFactory() {
        
    }
    
    /**
     * コネクション情報を全て破棄.
     * <BR><BR>
     * コネクション情報を全て破棄します.
     */
    public final void destroy() {
        
        synchronized( SYNC ) {
            
            int len ;
            ChannelKey[] keys = ChannelFactory.getChannekKeys() ;
            if( keys != null && ( len = keys.length ) > 0 ) {
                for( int i = 0 ; i < len ; i ++ ) {
                    try {
                        ChannelFactory.remove(
                            true,keys[ i ].getChannelType(),keys[ i ].getName() ) ;
                    } catch( Exception e ) {
                    }
                }
            }
            
        }
    }
    
    /**
     * 新しいチャネルを生成.
     * <BR><BR>
     * 新しいチャネルを生成します.
     * <BR>
     * @param bean 対象のチャネルBeanを設定します.
     * @exception Exception 例外.
     */
    public static final void create( ChannelBean bean )
        throws Exception {
        
        if( bean == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        // Bean内の条件は、受信チャネルの場合.
        if( bean.getType() == Channel.TYPE_RECEIVE ) {
            ChannelFactory.createReveiveChannel( bean.getChannelName() ) ;
        }
        // Bean内の条件は、送信チャネルの場合.
        else if( bean.getType() == Channel.TYPE_SEND ) {
            
            InetAddress addr = null ;
            if( bean.getConnectAddress() != null ) {
                try {
                    addr = InetAddress.getByName( bean.getConnectAddress() ) ;
                } catch( Exception e ) {
                    throw new InputException( e ) ;
                }
            }
            
            ChannelFactory.createSendChannel(
                bean.getChannelName(),bean.getQueueManager(),addr,
                bean.getConnectPort(),bean.getConnectObjectName(),
                bean.getUseCb32Word()
            ) ;
            
        }
        else {
            throw new InputException(
                "指定Beanのタイプは不正です(" + bean.getType() + ")" ) ;
        }
        
    }
    
    /**
     * 新しい受信チャネルを生成.
     * <BR><BR>
     * 新しい受信チャネルを生成します.
     * <BR>
     * @param naming チャネル名を設定します.
     * @exception Exception 例外.
     */
    public static final void createReveiveChannel( String naming )
        throws Exception {
        
        synchronized( SYNC ) {
            
            String channelName = null ;
            if(
                naming == null || ( naming = naming.trim().toLowerCase() ).length() <= 0 ||
                ( channelName = ChannelFactory.createNaming( Channel.TYPE_RECEIVE,naming ) ) == null ||
                NamingManager.get( channelName ) != null
            ) {
                if( naming == null || naming.length() <= 0 || channelName == null ) {
                    throw new InputException( "引数は不正です" ) ;
                }
                throw new InputException( "対象の受信チャネル名[" + naming +
                    "]は既に登録されています" ) ;
            }
            
            ChannelImple imple = new ChannelImple( naming ) ;
            NamingManager.add( channelName,imple ) ;
            
            NumberTable tbl = null ;
            ObjectArray ary = null ;
            
            if( ( tbl = ( NumberTable )NamingManager.get( NAMING_CHANNEL_TABLE_PLUS ) ) == null ) {
                tbl = new NumberTable() ;
                NamingManager.add( NAMING_CHANNEL_TABLE_PLUS,tbl ) ;
            }
            
            if( ( ary = ( ObjectArray )NamingManager.get( NAMING_CHANNEL_ARRAY_PLUS ) ) == null ) {
                ary = new ObjectArray() ;
                NamingManager.add( NAMING_CHANNEL_ARRAY_PLUS,ary ) ;
            }
            
            tbl.add( imple.getId(),imple ) ;
            ary.add( imple ) ;
            
        }
            
    }
    
    /**
     * 新しい送信チャネルを生成.
     * <BR><BR>
     * 新しい送信チャネルを生成します.
     * <BR>
     * @param naming チャネル名を設定します.
     * @param manager 対象の接続先キューマネージャ名を設定します.
     * @param addr 接続先アドレスを設定します.
     * @param port 接続先ポート番号を設定します.
     * @param connectName 通信処理で利用する通信オブジェクト名を設定します.
     * @param cb32Word 対象の利用暗号ワードを設定します.
     * @exception Exception 入力例外.
     */
    public static final void createSendChannel(
        String naming,String manager,InetAddress addr,int port,
        String connectName,String useCb32Word )
        throws Exception {
        
        synchronized( SYNC ) {
            
            String channelName = null ;
            
            if(
                naming == null || ( naming = naming.trim().toLowerCase() ).length() <= 0 ||
                ( channelName = ChannelFactory.createNaming( Channel.TYPE_SEND,naming ) ) == null ||
                NamingManager.get( channelName ) != null
            ) {
                if( naming == null || naming.length() <= 0 || channelName == null ) {
                    throw new InputException( "引数は不正です" ) ;
                }
                throw new InputException( "対象の受信チャネル名[" + naming +
                    "]は既に登録されています" ) ;
            }
            else if( ConnectFactory.get( connectName ) == null ) {
                throw new InputException( "指定された通信オブジェクト[" + connectName +
                    "]は存在しません" ) ;
            }
            
            SendChannelImple imple = new SendChannelImple(
                naming,manager,addr,port,connectName,useCb32Word ) ;
            NamingManager.add( channelName,imple ) ;
            NumberTable tbl = null ;
            ObjectArray ary = null ;
            
            if( ( tbl = ( NumberTable )NamingManager.get( NAMING_CHANNEL_TABLE_PLUS ) ) == null ) {
                tbl = new NumberTable() ;
                NamingManager.add( NAMING_CHANNEL_TABLE_PLUS,tbl ) ;
            }
            
            if( ( ary = ( ObjectArray )NamingManager.get( NAMING_CHANNEL_ARRAY_PLUS ) ) == null ) {
                ary = new ObjectArray() ;
                NamingManager.add( NAMING_CHANNEL_ARRAY_PLUS,ary ) ;
            }
            
            tbl.add( imple.getId(),imple ) ;
            ary.add( imple ) ;
            
        }
        
    }
    
    /**
     * チャネル削除.
     * <BR><BR>
     * チャネルオブジェクトを削除します.
     * <BR>
     * @param type チャネルタイプを設定します.
     * @param naming 削除対象のネーミングを設定します.
     */
    public static final void remove( int type,String naming ) {
        ChannelFactory.remove( false,type,naming ) ;
    }
    
    /**
     * チャネル削除.
     */
    private static final void remove( boolean mode,int type,String naming ) {
        
        int i ;
        int len ;
        
        ObjectArray ary = null ;
        NumberTable tbl = null ;
        Object o = null ;
        
        if( ( naming = ChannelFactory.createNaming( type,naming ) ) == null ) {
            return ;
        }
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( naming ) ;
            
            if( o != null && o instanceof ChannelImple ) {
                
                String name = ( ( ChannelImple )o ).getName() ;
                if( name != null && name.length() > 0 ) {
                    
                    // 送信チャネルの削除処理の場合.
                    //if( mode == false && type == Channel.TYPE_SEND ) {
                    //    
                    //    // 対象送信キューから、削除対象のチャネル情報を削除.
                    //    len = BaseQueueFactory.getQueueTypeByElementLength( QueueDef.TYPE_SEND ) ;
                    //    if( len > 0 ) {
                    //        for( i = len-1 ; i >= 0 ; i -- ) {
                    //            try {
                    //                BaseQueue queue = BaseQueueFactory.get( QueueDef.TYPE_SEND,i ) ;
                    //                if( queue != null ) {
                    //                    synchronized( queue.getSynchronized().get() ) {
                    //                        SendMqOption opt = ( SendMqOption )queue.getOption() ;
                    //                        if( opt != null ) {
                    //                            opt.remove( name ) ;
                    //                        }
                    //                    }
                    //                }
                    //            } catch( Exception e ) {
                    //            }
                    //        }
                    //    }
                    //    
                    //}
                    
                    // ネーミング管理テーブルから、対象条件を削除.
                    int id = ( ( ChannelImple )o ).getId() ;
                    NamingManager.remove( naming ) ;
                    tbl = ( NumberTable )NamingManager.get( NAMING_CHANNEL_TABLE_PLUS ) ;
                    
                    if( tbl != null && ( len = tbl.size() ) > 0 ) {
                        
                        tbl.remove( id ) ;
                        
                    }
                    
                    // ID管理テーブルから、対象条件を削除.
                    ary = ( ObjectArray )NamingManager.get( NAMING_CHANNEL_ARRAY_PLUS ) ;
                    
                    if( ary != null && ( len = ary.size() ) > 0 ) {
                        for( i = 0 ; i < len ; i ++ ) {
                            if( o == ary.get( i ) ) {
                                ary.remove( i ) ;
                                break ;
                            }
                        }
                    }
                    
                }
                
            }
        }
        
    }
    
    /**
     * チャネル取得.
     * <BR><BR>
     * チャネルオブジェクトを取得します.
     * <BR>
     * @param type チャネルタイプを設定します.
     * @param naming 取得対象のネーミングを設定します.
     * @return Channel チャネル情報が返されます.
     */
    public static final Channel get( int type,String naming ) {
        
        Object o = null ;
        Channel ret = null ;
        
        if( ( naming = ChannelFactory.createNaming( type,naming ) ) == null ) {
            return null ;
        }
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( naming ) ;
            if( o != null && o instanceof Channel ) {
                ret = ( Channel )o ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * チャネル取得.
     * <BR><BR>
     * チャネルオブジェクトを取得します.
     * <BR>
     * @param id 取得対象のチャネルIDを設定します.
     * @return Channel チャネル情報が返されます.
     */
    public static final Channel getToID( int id ) {
        
        Channel ret = null ;
        
        try {
            synchronized( SYNC ) {
                
                ret = ( Channel )(
                    ( NumberTable )NamingManager.get(
                        NAMING_CHANNEL_TABLE_PLUS )
                ).get( id ) ;
                
            }
        } catch( Exception e ) {
        }
        
        return ret ;
        
    }
    
    /**
     * チャネル取得.
     * <BR><BR>
     * チャネルオブジェクトを取得します.
     * <BR>
     * @param no 取得対象のチャネル項番を設定します.
     * @return Channel チャネル情報が返されます.
     */
    public static final Channel getToNumber( int no ) {
        
        Channel ret = null ;
        
        try {
            synchronized( SYNC ) {
                
                ret = ( Channel )(
                    ( ObjectArray )NamingManager.get(
                        NAMING_CHANNEL_ARRAY_PLUS )
                ).get( no ) ;
                
            }
        } catch( Exception e ) {
        }
        
        return ret ;
        
    }
    
    /**
     * 登録されているチャネルKey一覧を取得.
     * <BR><BR>
     * 登録されているチャネルKey一覧を取得します.
     * <BR>
     * @return ChannelKey[] チャネルKey一覧が返されます.
     */
    public static final ChannelKey[] getChannekKeys() {
        
        int i ;
        int len ;
        
        ObjectArray ary = null ;
        ChannelKey[] ret = null ;
        
        try {
            synchronized( SYNC ) {
                
                ary = ( ObjectArray )NamingManager.get(
                    NAMING_CHANNEL_ARRAY_PLUS ) ;
                if( ary != null && ( len = ary.size() ) > 0 ) {
                    ret = new ChannelKey[ len ] ;
                    Object obj = null ;
                    for( i = 0 ; i < len ; i ++ ) {
                        if( ( obj = ary.get( i ) ) != null ) {
                            ret[ i ] = ( ( ChannelImple )obj ).getChannelKey() ;
                        }
                    }
                }
                
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 登録されているチャネルID一覧を取得.
     * <BR><BR>
     * 登録されているチャネルID一覧を取得します.
     * <BR>
     * @return int[] チャネルID一覧が返されます.
     */
    public static final int[] getIDs() {
        
        int[] ret = null ;
        
        try {
            synchronized( SYNC ) {
                
                ret = ( ( NumberTable )NamingManager.get(
                    NAMING_CHANNEL_TABLE_PLUS ) ).getNumbers() ;
                    
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 登録チャネル数を取得.
     * <BR><BR>
     * 登録されているチャネル数を取得します.
     * <BR>
     * @return int 登録されているチャネル数が返されます.
     */
    public static final int size() {
        
        int ret = 0 ;
        
        try {
            synchronized( SYNC ) {
                
                ret = ( ( ObjectArray )NamingManager.get(
                     NAMING_CHANNEL_ARRAY_PLUS ) ).size() ;
                
            }
        } catch( Exception e ) {
            ret = 0 ;
        }
        
        return ret ;
        
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Object 同期オブジェクトが返されます.
     */
    public static final Object getSync() {
        return SYNC ;
    }
    
    
    
    /**
     * チャネルファクトリ名生成.
     */
    private static final String createNaming( int type,String name ) {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        
        return new StringBuffer().
            append( "%" ).append( name ).append( "%" ).
            append( type ).append( NAMING_PLUS ).
            toString() ;
        
    }
    
}

