/*
 * @(#)ChannelImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel ;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.queue.access.status.ChannelStatus;

/**
 * 受信チャネルオブジェクト.
 * 
 * @version 2006/09/02
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class ChannelImple implements Channel {
    
    /**
     * チャネル名.
     */
    protected String name = null ;
    
    /**
     * チャネルタイプ.
     */
    protected int type = -1 ;
    
    /**
     * チャネルID.
     */
    protected int id = -1 ;
    
    /**
     * ステータス.
     */
    protected int state = ChannelStatus.STATE_SHUTDOWN ;
    
    /**
     * チャネルKey.
     */
    protected ChannelKey channelKey = null ;
    
    /**
     * 最終受信ID管理.
     */
    protected LastReceiveIdManage lastReceiveIdManage = null ;
    
    /**
     * 同期オブジェクト.
     */
    protected final Synchronized sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    protected ChannelImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を設定して、オブジェクトを生成します.
     * <BR>
     * @param name チャネル名を設定します.
     * @exception Exception 例外.
     */
    public ChannelImple( String name )
        throws Exception {
            
        if(  name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        sync.create() ;
        
        try {
            int id = ( ( ( name.hashCode() & 0x3fffffff ) * 2 ) & 0x3fffffff ) ;
            synchronized( ChannelFactory.getSync() ) {
                for( ;; ) {
                    if( id != -1 && ChannelFactory.getToID( id ) != null ) {
                        id ++ ;
                    }
                    else {
                        break ;
                    }
                }
                synchronized( sync.get() ) {
                    this.name = name ;
                    this.id = id ;
                    this.type = Channel.TYPE_RECEIVE ;
                    this.state = ChannelStatus.STATE_SUCCESS ;
                    this.channelKey = new ChannelKey( Channel.TYPE_RECEIVE,name ) ;
                    this.lastReceiveIdManage = new LastReceiveIdManage() ;
                }
            }
        } catch( Exception e ) {
            this.destroy() ;
            throw e ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * チャネルオブジェクトを破棄します.
     */
    public void destroy() {
        
        try {
            synchronized( sync.get() ) {
                name = null ;
                id = -1 ;
                state = ChannelStatus.STATE_DOWN ;
                channelKey = null ;
                lastReceiveIdManage = null ;
            }
        } catch( Exception e ) {
        }
        
        name = null ;
        id = -1 ;
        state = ChannelStatus.STATE_DOWN ;
        channelKey = null ;
        lastReceiveIdManage = null ;
        
        sync.clear() ;
    }
    
    /**
     * チャネル名を取得.
     * <BR><BR>
     * チャネル名を取得します.
     * <BR>
     * @return String チャネル名が返されます.
     */
    public String getName() {
        
        String ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = name ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * チャネルタイプを取得.
     * <BR><BR>
     * チャネルタイプを取得します.
     * <BR>
     * @return int チャネルタイプが返されます.
     */
    public int getType() {
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = type ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * チャネルIDを設定.
     * <BR><BR>
     * チャネルIDを設定します.
     * <BR>
     * @param id チャネルIDを設定します.
     */
    public void setId( int id ) {
        
        try {
            synchronized( sync.get() ){
                this.id = id ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * チャネルIDを取得.
     * <BR><BR>
     * チャネルIDを取得します.
     * <BR>
     * @return int チャネルIDが返されます.
     */
    public int getId() {
        
        int ret = -1 ;
        
        try {
            synchronized( sync.get() ) {
                ret = id ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
        
    }
    
    /**
     * ステータスを設定.
     * <BR><BR>
     * チャネルステータスを設定します.
     * <BR>
     * @param state 対象のチャネルステータスを設定します.
     */
    public void setState( int state ) {
        /*
        if(
            state != ChannelStatus.STATE_SUCCESS &&
            state != ChannelStatus.STATE_DOWN &&
            state != ChannelStatus.STATE_STARTUP &&
            state != ChannelStatus.STATE_SHUTDOWN &&
            state != ChannelStatus.STATE_ERROR
        ) {
            return ;
        }
        
        try {
            synchronized( sync.get() ) {
                if( this.state != state ) {
                    LOG.info(
                        new StringBuffer().
                        append( "受信チャネル(" ).
                        append( name ).
                        append( ")のステータスは " ).
                        append( ChannelStatus.getStateByString( this.state ) ).
                        append( " から " ).
                        append( ChannelStatus.getStateByString( state ) ).
                        append( " に変更されました" ).
                        toString() ) ;
                    this.state = state ;
                }
            }
        } catch( Exception e ) {
        }
        */
        
        // 受信チャネルは、ステータスが変動しない.
        try {
            synchronized( sync.get() ) {
                this.state = ChannelStatus.STATE_SUCCESS ;
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * ステータスを取得.
     * <BR><BR>
     * 現在のチャネルステータスが返されます.
     * <BR>
     * @return int チャネルステータスが返されます.<BR>
     *             [STATE_SUCCESS]が返された場合、「正常」です.<BR>
     *             [STATE_DOWN]が返された場合、「停止」です.<BR>
     *             [STATE_STARTUP]が返された場合、「起動中」です.<BR>
     *             [STATE_SHUTDOWN]が返された場合、「停止」です.<BR>
     *             [STATE_ERROR]が返された場合、「異常」です.<BR>
     *             [STATE_NOT_HEART_BEAT]が返された場合、「接続未確立による停止」です.<BR>
     *             [STATE_NOT_CONNECT_CHANNEL]が返された場合、「接続先チャネル
     *             非存在による停止」です.
     */
    public int getState() {
        
        int ret = ChannelStatus.STATE_DOWN ;
        
        try {
            synchronized( sync.get() ) {
                ret = state ;
            }
        } catch( Exception e ) {
            ret = ChannelStatus.STATE_DOWN ;
        }
        
        return ret ;
        
    }
    
    /**
     * チャネルBeanオブジェクトを取得.
     * <BR><BR>
     * チャネルBeanオブジェクトを取得します.
     * <BR>
     * @return ChannelBean チャネルBeanオブジェクトが返されます.
     */
    public ChannelBean getChannelBean() {
        
        ChannelBean ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = new ChannelBean() ;
                ret.setType( this.type ) ;
                ret.setChannelName( this.name ) ;
                ret.setConnectAddress( null ) ;
                ret.setConnectObjectName( null ) ;
                ret.setConnectPort( -1 ) ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * チャネルBeanオブジェクトを取得.
     * <BR><BR>
     * チャネルBeanオブジェクトを取得します.
     * <BR>
     * @param bean 設定対象のBeanを設定します.
     */
    public void getBean( ChannelBean bean ) {
        try {
            synchronized( sync.get() ) {
                bean.setType( this.type ) ;
                bean.setChannelName( this.name ) ;
                bean.setConnectAddress( null ) ;
                bean.setConnectObjectName( null ) ;
                bean.setConnectPort( -1 ) ;
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * チャネルKeyを取得.
     * <BR><BR>
     * 対象のチャネルKeyを取得します.
     * <BR>
     * @return ChannelKey チャネルKeyが返されます.
     */
    public ChannelKey getChannelKey() {
        
        ChannelKey ret = null ;
        
        try {
            synchronized( sync.get() ) {
                ret = channelKey ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 最終受信ID管理オブジェクトを取得.
     * <BR><BR>
     * 最終受信ID管理オブジェクトを取得します.
     * <BR>
     * @return LastReceiveIdManage 最終受信ID管理オブジェクトが返されます.
     */
    public LastReceiveIdManage getLastReceiveIdManage() {
        return lastReceiveIdManage ;
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    public Synchronized getSynchronized() {
        return sync ;
    }
    
    /**
     * チャネルオブジェクトが有効であるか取得.
     * <BR><BR>
     * チャネルオブジェクトが有効であるか取得します.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public boolean isChannel() {
        
        boolean ret = false ;
        
        try {
            synchronized( sync.get() ) {
                ret = ( name != null && name.length() > 0 ) ?
                    true : false ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
}

