/*
 * @(#)ProtocolData.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.protocol ;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.NetDef;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.Resource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.queue.main.queue.QueueMessage;
import org.maachang.queue.main.queue.QueueUtil;
import org.maachang.queue.main.queue.base.core.QArrayChild;

/**
 * チャネル通信データプロトコル.
 * 
 * @version 2006/08/31
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class ProtocolData implements ProtocolObject {
    
    /**
     * 電文ID.
     */
    private long id = -1L ;
    
    /**
     * 処理名.
     */
    private String name = null ;
    
    /**
     * キューマネージャ名.
     */
    private String queueManagerName = null ;
    
    /**
     * 送信元のキューマネージャ名.
     */
    private String srcQueueManagerName = null ;
    
    /**
     * チャネル名.
     */
    private String channelName = null ;
    
    /**
     * データ.
     */
    private Object data = null ;
    
    /**
     * データ長.
     */
    private int length = 0 ;
    
    /**
     * キーコード.
     */
    private String keyCode = null ;
    
    /**
     * プロセスID.
     */
    private String processId = null ;
    
    /**
     * プライオリティ.
     */
    private int priority = 0 ;
    
    /**
     * Expire.
     */
    private int expire = -1 ;
    
    /**
     * 受信ポート番号.
     */
    private int port = -1 ;
    
    /**
     * チャネルID.
     */
    private int channelId = -1 ;
    
    /**
     * マックアドレス.
     */
    private byte[] macAddress = null ;
    
    
    /**
     * コンストラクタ.
     */
    public ProtocolData() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * キューデータを設定して、オブジェクトを生成します.
     * <BR>
     * @param id 対象の送信IDを設定します.
     * @param qman 対象のキューマネージャ名を設定します.
     * @param srcQman 対象の送信元キューマネージャ名を設定します.
     * @param qname 対象のキュー名を設定します.
     * @param channelName 対象のチャネル名を設定します.
     * @param qdata キューデータを設定してオブジェクトを生成します.
     * @param port 対象のポート番号を設定します.
     * @param channelId 対象のチャネルIDを設定します.
     * @param macAddress 対象のマックアドレスを設定します.
     * @exception InputException 入力例外.
     */
    public void create( long id,String qman,String srcQman,String qname,
        String channelName,QArrayChild qdata,int port,int channelId,byte[] macAddress )
        throws InputException {
        
        if( qdata == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        // Expire値が設定されている場合、残りのExpire値を設定.
        if( qdata.getExpire() != -1L &&
            qdata.getExpire() != Long.MAX_VALUE ) {
            long exp = ( qdata.getExpire() + qdata.getTime() ) -
                System.currentTimeMillis() ;
            if( exp <= 0 ) {
                exp = 100L ;
            }
            this.setExpire( ( int )( exp & 0x00000000ffffffffL ) ) ;
        }
        // Expire値が設定されていない場合、未設定値を設定.
        else {
            this.setExpire( -1 ) ;
        }
        
        this.setId( id ) ;
        this.setData( qdata.getValue() ) ;
        this.setName( qname ) ;
        this.setQueueManagerName( qman ) ;
        this.setSrcQueueManagerName( srcQman ) ;
        this.setChannelName( channelName ) ;
        this.setPriority( qdata.getPriority() ) ;
        this.setKeyCode( qdata.getKey() ) ;
        this.setProcessId( qdata.getProcessId() ) ;
        this.setReceivePort( port ) ;
        this.setChannelId( channelId ) ;
        this.setMacAddress( macAddress ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear() {
        id = -1L ;
        name = null ;
        queueManagerName = null ;
        srcQueueManagerName = null ;
        channelName = null ;
        data = null ;
        length = 0 ;
        keyCode = null ;
        processId = null ;
        priority = 0 ;
        expire = -1 ;
        port = -1 ;
        channelId = -1 ;
        macAddress = null ;
    }
    
    
    /**
     * データオブジェクトを取得.
     * <BR><BR>
     * データオブジェクトを取得します.
     * <BR>
     * @return Object データオブジェクトを取得します.<BR>
     *                 取得されるデータオブジェクトは[byte[]]か[BinResource]です.
     */
    public Object getData() {
        return data;
    }
    
    /**
     * データオブジェクトをBinResourceに変換して取得.
     * <BR><BR>
     * データオブジェクトをBinResourceに変換して取得します.
     * <BR>
     * @return BinResource BinResourceに変換されたデータオブジェクトが返されます.
     */
    public BinResource getDataByBinResource() {
        return this.getDataByBinResource( null ) ;
    }
    
    /**
     * データオブジェクトをBinResourceに変換して取得.
     * <BR><BR>
     * データオブジェクトをBinResourceに変換して取得します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     * @return BinResource BinResourceに変換されたデータオブジェクトが返されます.
     */
    public BinResource getDataByBinResource( ResourceType resType ) {
        
        Object o = null ;
        BinResource ret = null ;
        
        try {
            if(
                data != null &&
                (
                    ( data instanceof byte[] ) == true ||
                    ( data instanceof BinResource ) == true ||
                    ( data instanceof QueueMessage ) == true
                )
            ) {
                
                if( data instanceof QueueMessage ) {
                    o = QueueUtil.convertQueueMessageByMessage(
                            resType,( QueueMessage )data
                        ) ;
                }
                else {
                    o = data ;
                }
                
                if( o != null ) {
                    if( o instanceof byte[] ) {
                        ret = Resource.createBinResource( resType,( byte[] )o ) ;
                    }
                    else if( o instanceof BinResource ) {
                        ret = ( BinResource )o ;
                    }
                }
                else {
                    ret = null ;
                }
                
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * データ情報を設定.
     * <BR><BR>
     * データ情報を設定します.
     * <BR>
     * @param data 設定対象のデータを設定します.<BR>
     *             設定可能なオブジェクトは以下の通りです.<BR>
     *             [byte[]]<BR>
     *             [BinResource]<BR>
     *             [QueueMessage].
     */
    public void setData(Object data) {
        
        if(
            data != null &&
            (
                ( data instanceof byte[] ) == true ||
                ( data instanceof BinResource ) == true ||
                ( data instanceof QueueMessage ) == true
            )
        ) {
            if( data instanceof byte[] ) {
                this.length = ( ( byte[] )data ).length ;
            }
            else if( data instanceof BinResource ) {
                this.length = ( ( BinResource )data ).size() ;
            }
            else {
                this.length = ( ( QueueMessage )data ).getLength() ;
            }
            this.data = data;
        }
        
    }
    
    /**
     * 電文ID情報を取得.
     * <BR><BR>
     * 電文ID情報を取得します.
     * <BR>
     * @return long 電文ID情報が返されます.
     */
    public long getId() {
        return id;
    }
    
    /**
     * 電文ID情報を設定.
     * <BR><BR>
     * 電文ID情報を設定します.
     * <BR>
     * @param id 対象の電文IDを設定します.
     */
    public void setId(long id) {
        this.id = id;
    }
    
    /**
     * データ長を取得.
     * <BR><BR>
     * 設定されているデータ長を取得します.
     * <BR>
     * @return int データ長が返されます.
     */
    public int getLength() {
        return length;
    }
    
    /**
     * 処理名を取得.
     * <BR><BR>
     * 処理名を取得します.
     * <BR>
     * @return String 処理名が返されます.
     */
    public String getName() {
        return name ;
    }
    
    /**
     * 処理名を設定.
     * <BR><BR>
     * 処理名を設定します.
     * <BR>
     * @param name キュー名を設定します.
     */
    public void setName(String name) {
        this.name = name ;
    }
    
    /**
     * キューマネージャ名を取得.
     * <BR><BR>
     * キューマネージャ名を取得します.
     * <BR>
     * @return String キューマネージャ名が返されます.
     */
    public String getQueueManagerName() {
        return queueManagerName ;
    }
    
    /**
     * キューマネージャ名を設定.
     * <BR><BR>
     * キューマネージャ名を設定します.
     * <BR>
     * @param queueManagerName キューマネージャ名を設定します.
     */
    public void setQueueManagerName( String queueManagerName ) {
        this.queueManagerName = queueManagerName ;
    }
    
    /**
     * 送信元キューマネージャ名を取得.
     * <BR><BR>
     * 送信元キューマネージャ名を取得します.
     * <BR>
     * @return String 送信元キューマネージャ名が返されます.
     */
    public String getSrcQueueManagerName() {
        return srcQueueManagerName ;
    }
    
    /**
     * 送信元キューマネージャ名を設定.
     * <BR><BR>
     * 送信元キューマネージャ名を設定します.
     * <BR>
     * @param srcQueueManagerName 送信元キューマネージャ名を設定します.
     */
    public void setSrcQueueManagerName( String srcQueueManagerName ) {
        this.srcQueueManagerName = srcQueueManagerName ;
    }
    
    /**
     * チャネル名を設定.
     * <BR><BR>
     * チャネル名を設定します.
     * <BR>
     * @param channelName 対象のチャネル名を設定します.
     */
    public void setChannelName( String channelName ) {
        this.channelName = channelName ;
    }
    
    /**
     * チャネル名を取得.
     * <BR><BR>
     * チャネル名を取得します.
     * <BR>
     * @return String チャネル名が返されます.
     */
    public String getChannelName() {
       return this.channelName ;
    }
    
    /**
     * Expireを取得.
     * <BR><BR>
     * Exoireを取得します.
     * <BR>
     * @return int Expireを取得します.
     */
    public int getExpire() {
        return expire;
    }
    
    /**
     * Expireを設定.
     * <BR><BR>
     * Expireを設定します.
     * <BR>
     * @param expire Expireを設定します.
     */
    public void setExpire(int expire) {
        this.expire = expire;
    }
    
    /**
     * Expireを設定.
     * <BR><BR>
     * Expireを設定します.
     * <BR>
     * @param expire Expireを設定します.
     */
    public void setExpire(long expire) {
        if( expire == Long.MAX_VALUE ) {
            this.expire = -1 ;
        }
        else {
            this.setExpire(
                ( int )( expire &
                0x000000007fffffffL ) ) ;
        }
    }
    
    /**
     * Keyコードを取得.
     * <BR><BR>
     * Keyコードを取得します.
     * <BR>
     * @return String Keyコードが返されます.
     */
    public String getKeyCode() {
        return keyCode;
    }
    
    /**
     * Keyコードを設定.
     * <BR><BR>
     * Keyコードを設定します.
     * <BR>
     * @param keyCode Keyコードを設定します.
     */
    public void setKeyCode(String keyCode) {
        this.keyCode = keyCode;
    }
    
    /**
     * プロセスIDを取得.
     * <BR><BR>
     * プロセスIDを取得します.
     * <BR>
     * @return String プロセスIDが返されます.
     */
    public String getProcessId() {
        return processId ;
    }
    
    /**
     * プロセスIDを設定.
     * <BR><BR>
     * プロセスIDを設定します.
     * <BR>
     * @param processId プロセスIDを設定します.
     */
    public void setProcessId( String processId ) {
        this.processId = processId ;
    }
    
    /**
     * プライオリティを取得.
     * <BR><BR>
     * プライオリティを取得します.
     * <BR>
     * @return int プライオリティが返されます.
     */
    public int getPriority() {
        return priority;
    }
    
    /**
     * プライオリティを設定.
     * <BR><BR>
     * プライオリティを設定します.
     * <BR>
     * @param priority プライオリティを設定します.
     */
    public void setPriority(int priority) {
        this.priority = priority;
    }
    
    /**
     * 受信ポート番号を取得.
     * <BR><BR>
     * 受信ポート番号を取得します.
     * <BR>
     * @return int 受信ポート番号が返されます.
     */
    public int getReceivePort() {
        return port ;
    }
    
    /**
     * 受信ポート番号を設定.
     * <BR><BR>
     * 受信ポート番号を設定します.
     * <BR>
     * @param port 受信ポート番号を設定します.
     */
    public void setReceivePort( int port ) {
        this.port = port ;
    }
    
    /**
     * チャネルIDを取得
     * <BR><BR>
     * チャネルIDを取得します.
     * <BR>
     * @return int チャネルIDが返されます.
     */
    public int getChannelId() {
        return channelId ;
    }
    
    /**
     * チャネルIDを設定.
     * <BR><RB>
     * チャネルIDを設定します.
     * <BR>
     * @param channelId 対象のチャネルIDを設定します.
     */
    public void setChannelId( int channelId ) {
        this.channelId = channelId ;
    }
    
    /**
     * マックアドレスを取得.
     * <BR><BR>
     * マックアドレスを取得します.
     * <BR>
     * @return byte[] マックアドレスが返されます.
     */
    public byte[] getMacAddress() {
        return macAddress ;
    }
    
    /**
     * マックアドレスをlong値で取得.
     * <BR><BR>
     * マックアドレスをlong値で取得します.
     * <BR>
     * @return long マックアドレスをlong値で変換されたものが返されます.
     */
    public long getMacAddressByLong() {
        if( this.macAddress == null ||
            this.macAddress.length != 6 ) {
            return 0L ;
        }
        
        return ( long )(
            ( ( long )this.macAddress[ 0 ] & 0x00000000000000ffL ) |
            ( ( ( long )this.macAddress[ 1 ] & 0x00000000000000ffL ) << 8L ) |
            ( ( ( long )this.macAddress[ 2 ] & 0x00000000000000ffL ) << 16L ) |
            ( ( ( long )this.macAddress[ 3 ] & 0x00000000000000ffL ) << 24L ) |
            ( ( ( long )this.macAddress[ 4 ] & 0x00000000000000ffL ) << 32L ) |
            ( ( ( long )this.macAddress[ 5 ] & 0x00000000000000ffL ) << 40L ) ) ;
    }
    
    /**
     * マックアドレスを設定.
     * <BR><BR>
     * マックアドレスを設定します.
     * <BR>
     * @param macAddress マックアドレスを設定します.
     */
    public void setMacAddress( byte[] macAddress ) {
        if( macAddress != null &&
            macAddress.length != NetDef.MAC_ADDRESS_LENGTH ) {
            return ;
        }
        this.macAddress = macAddress ;
    }
    
    /**
     * プロトコルタイプを取得.
     * <BR><BR>
     * プロトコルタイプを取得します.
     * <BR>
     * @return int プロトコルタイプが返されます.<BR>
     *             [ProtocolObject.PROTOCOL_DATA]が返された場合、データオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_HEARTBEAT]が返された場合、ハートビートオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_SUCCESS]が返された場合、正常オブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_ERROR]が返された場合ｍエラーオブジェクトです.
     */
    public int getType() {
        return ProtocolObject.PROTOCOL_DATA ;
    }
    
    /**
     * プロトコル戻りタイプを取得.
     * <BR><BR>
     * プロトコル戻りタイプを取得します.
     * <BR>
     * @return int プロトコル戻りタイプが返されます.
     */
    public int getReturnType() {
        return ProtocolObject.PROTOCOL_DATA ;
    }
    
    /**
     * オブジェクトを文字列に変換.
     * <BR><BR>
     * オブジェクトを文字列に変換します.
     * <BR>
     * @return String 文字列に変換された情報が返されます.
     */
    public String toString() {
        
        String ret = null ;
        
        ret = new StringBuffer().
            append( "[PROTOCOL_DATA]" ).
            append( " id:" ).
            append( id ).
            append( " name:" ).
            append( name ).
            append( " queueManagerName:" ).
            append( queueManagerName ).
            append( " srcQueueManagerName:" ).
            append( srcQueueManagerName ).
            append( " channelName:" ).
            append( channelName ).
            append( " dataLenght:" ).
            append( length ).
            append( " keyCode:" ).
            append( keyCode ).
            append( " processId:" ).
            append( processId ).
            append( " priority:" ).
            append( priority ).
            append( " expire:" ).
            append( expire ).
            append( " port:" ).
            append( port ).
            append( " channelId:" ).
            append( channelId ).
            toString() ;
            
        return ret ;
    }
    
}

