/*
 * @(#)ReceiveService.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.receive ;

import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.sys.NamingManager;
import org.maachang.queue.main.channel.service.receive.core.CoreReceiveTable;
import org.maachang.queue.main.channel.service.receive.core.CoreReceiveThread;
import org.maachang.queue.main.channel.service.receive.core.SwitchReceiveThread;

/**
 * 受信サービス.
 *  
 * @version 2006/12/21
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class ReceiveService {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ReceiveService.class ) ;
    
    /**
     * ネーミングマネージャ登録名.
     */
    public static final String NAMING_MANAGER = "SERVICE@maachangq.main.channel.service.receive" ;
    
    /**
     * スレッド最小数.
     */
    private static final int MIN_THREAD = 1 ;
    
    /**
     * スレッド最大数.
     */
    private static final int MAX_THREAD = 6 ;
    
    /**
     * コアスレッドデフォルト値.
     */
    private static final int DEF_CORE_THREAD = 4 ;
    
    /**
     * スイッチスレッドデフォルト値.
     */
    private static final int DEF_SWITCH_THREAD = 3 ;
    
    /**
     * データスレッドデフォルト値.
     */
    private static final int DEF_DATA_THREAD = 4 ;
    
    /**
     * ハートビートスレッドデフォルト値.
     */
    private static final int DEF_HEART_BEAT_THREAD = 1 ;
    
    /**
     * 正常戻りスレッドデフォルト値.
     */
    private static final int DEF_RESULT_SUCCESS_THREAD = 4 ;
    
    /**
     * 異常戻りスレッドデフォルト値.
     */
    private static final int DEF_RESULT_ERROR_THREAD = 1 ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public ReceiveService() {
        
    }
    
    /**
     * 受信スイッチオブジェクト群初期化処理.
     * <BR><BR>
     * @param coreThread コア受信スレッド数を設定します.
     * @param switchThread スイッチ受信スレッド数を設定します.
     * @param dataThread データ受信スレッド数を設定します.
     * @param heartBeatThread データ受信スレッド数を設定します.
     * @param resultErrorThread データ受信スレッド数を設定します.
     * @param resultSuccessThread データ受信スレッド数を設定します.
     */
    public final void init(
        int coreThread,int switchThread,
        int dataThread,int heartBeatThread,int resultErrorThread,int resultSuccessThread ) {
        
        int i ;
        Vector vector = null ;
        ExecutionSwitch exSw = null ;
        
        if( coreThread == -1 ) {
            coreThread = DEF_CORE_THREAD ;
        }
        
        if( switchThread == -1 ) {
            switchThread = DEF_SWITCH_THREAD ;
        }
        
        if( dataThread == -1 ) {
            dataThread = DEF_DATA_THREAD ;
        }
        
        if( heartBeatThread == -1 ) {
            heartBeatThread = DEF_HEART_BEAT_THREAD ;
        }
        
        if( resultErrorThread == -1 ) {
            resultErrorThread = DEF_RESULT_ERROR_THREAD ;
        }
        
        if( resultSuccessThread == -1 ) {
            resultSuccessThread = DEF_RESULT_SUCCESS_THREAD ;
        }
        
        coreThread = ( coreThread <= MIN_THREAD ) ? MIN_THREAD : coreThread ;
        coreThread = ( coreThread >= MAX_THREAD ) ? MAX_THREAD : coreThread ;
        
        switchThread = ( switchThread <= MIN_THREAD ) ? MIN_THREAD : switchThread ;
        switchThread = ( switchThread >= MAX_THREAD ) ? MAX_THREAD : switchThread ;
        
        dataThread = ( dataThread <= MIN_THREAD ) ? MIN_THREAD : dataThread ;
        dataThread = ( dataThread >= MAX_THREAD ) ? MAX_THREAD : dataThread ;
        
        heartBeatThread = ( heartBeatThread <= MIN_THREAD ) ? MIN_THREAD : heartBeatThread ;
        heartBeatThread = ( heartBeatThread >= MAX_THREAD ) ? MAX_THREAD : heartBeatThread ;
        
        resultErrorThread = ( resultErrorThread <= MIN_THREAD ) ? MIN_THREAD : resultErrorThread ;
        resultErrorThread = ( resultErrorThread >= MAX_THREAD ) ? MAX_THREAD : resultErrorThread ;
        
        resultSuccessThread = ( resultSuccessThread <= MIN_THREAD ) ? MIN_THREAD : resultSuccessThread ;
        resultSuccessThread = ( resultSuccessThread >= MAX_THREAD ) ? MAX_THREAD : resultSuccessThread ;
        
        LOG.info( "#### 受信サービス生成処理 ####" ) ;
        LOG.info( "# コアスレッド数:" + coreThread ) ;
        LOG.info( "# スイッチスレッド数:" + switchThread ) ;
        LOG.info( "# 受信データスレッド数:" + dataThread ) ;
        LOG.info( "# 受信ハートビートスレッド数:" + heartBeatThread ) ;
        LOG.info( "# 受信戻りエラースレッド数:" + resultErrorThread ) ;
        LOG.info( "# 受信戻り正常スレッド数:" + resultSuccessThread ) ;
        
        // コア受信テーブルを生成.
        CoreReceiveTable coreReceiveTable = new CoreReceiveTable() ;
        
        // コア受信スレッドを生成.
        CoreReceiveThread[] coreReceiveThread = new CoreReceiveThread[ coreThread ] ;
        for( i = 0 ; i < coreThread ; i ++ ) {
            coreReceiveThread[ i ] = new CoreReceiveThread( i,coreReceiveTable ) ;
        }
        
        // スイッチ受信スレッドを生成.
        SwitchReceiveThread[] switchReceiveThread = new SwitchReceiveThread[ coreThread ] ;
        for( i = 0 ; i < switchThread ; i ++ ) {
            switchReceiveThread[ i ] = new SwitchReceiveThread( i,coreReceiveTable ) ;
        }
        
        // データ電文処理群生成.
        ReceiveSwitchImple[] receiveDataSwitch = new ReceiveSwitchImple[ dataThread ] ;
        vector = new Vector() ;
        exSw = new ExecutionDataSwitch() ;
        for( i = 0 ; i < dataThread ; i ++ ) {
            receiveDataSwitch[ i ] = new ReceiveSwitchImple( exSw,i,vector ) ;
        }
        
        // ハートビート電文処理群生成.
        ReceiveSwitchImple[] receiveHeartBeatSwitch = new ReceiveSwitchImple[ heartBeatThread ] ;
        vector = new Vector() ;
        exSw = new ExecutionHeartBeatSwitch() ;
        for( i = 0 ; i < heartBeatThread ; i ++ ) {
            receiveHeartBeatSwitch[ i ] = new ReceiveSwitchImple( exSw,i,vector ) ;
        }
        
        // 戻りエラー電文処理群生成.
        ReceiveSwitchImple[] receiveResultErrorSwitch = new ReceiveSwitchImple[ resultErrorThread ] ;
        vector = new Vector() ;
        exSw = new ExecutionResultErrorSwitch() ;
        for( i = 0 ; i < resultErrorThread ; i ++ ) {
            receiveResultErrorSwitch[ i ] = new ReceiveSwitchImple( exSw,i,vector ) ;
        }
        
        // 戻り正常電文処理群生成.
        ReceiveSwitchImple[] receiveResultSuccessSwitch = new ReceiveSwitchImple[ resultSuccessThread ] ;
        vector = new Vector() ;
        exSw = new ExecutionResultSuccessSwitch() ;
        for( i = 0 ; i < resultSuccessThread ; i ++ ) {
            receiveResultSuccessSwitch[ i ] = new ReceiveSwitchImple( exSw,i,vector ) ;
        }
        
        // 生成オブジェクトを１つのオブジェクトに集約.
        ReceiveServiceBean swObjs = null ;
        swObjs = new ReceiveServiceBean() ;
        swObjs.setCoreReceiveTable( coreReceiveTable ) ;
        swObjs.setCoreReceiveThread( coreReceiveThread ) ;
        swObjs.setSwitchReceiveThread( switchReceiveThread ) ;
        swObjs.setReceiveDataSwitch( receiveDataSwitch ) ;
        swObjs.setReceiveHeartBeatSwitch( receiveHeartBeatSwitch ) ;
        swObjs.setReceiveResultErrorSwitch( receiveResultErrorSwitch ) ;
        swObjs.setReceiveResultSuccessSwitch( receiveResultSuccessSwitch ) ;
        
        this.destroy( false ) ;
        
        // ネーミングマネージャに登録.
        synchronized( SYNC ) {
            
            NamingManager.add( NAMING_MANAGER,swObjs ) ;
            
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy() {
        this.destroy( true ) ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @param mode ログ表示モードを設定します.
     */
    public final void destroy( boolean mode ) {
        
        if( mode == true ) {
            LOG.info( "#### 受信サービス破棄処理 ####" ) ;
        }
        
        synchronized( SYNC ) {
            
            NamingManager.remove( NAMING_MANAGER ) ;
            
        }
    }
    
    /**
     * コア受信テーブルを取得.
     * <BR><BR>
     * コア受信テーブルを取得します.
     * <BR>
     * @return CoreReceiveTable コア受信テーブルが返されます.
     */
    public static final CoreReceiveTable getCoreReceiveTable() {
        
        synchronized( SYNC ) {
            
            ReceiveServiceBean sw = ReceiveService.getReceiveServiceBean() ;
            if( sw != null ) {
                return sw.getCoreReceiveTable() ;
            }
            
        }
        
        return null ;
    }
    
    /**
     * 受信データスイッチスレッドを取得.
     * <BR><BR>
     * 受信データスイッチスレッドを取得します.
     * <BR>
     * @return ReceiveSwitch 受信スイッチ情報が返されます.
     */
    public static final ReceiveSwitch getReceiveDataSwitch() {
        
        synchronized( SYNC ) {
            
            ReceiveServiceBean sw = ReceiveService.getReceiveServiceBean() ;
            if( sw != null ) {
                return sw.getReceiveDataSwitch()[ sw.getReceiveDataPosition() ] ;
            }
            
        }
        
        return null ;
        
    }
    
    /**
     * 受信ハートビートスイッチスレッドを取得.
     * <BR><BR>
     * 受信ハートビートスイッチスレッドを取得します.
     * <BR>
     * @return ReceiveSwitch 受信スイッチ情報が返されます.
     */
    public static final ReceiveSwitch getReceiveHeartBeatSwitch() {
        
        synchronized( SYNC ) {
            
            ReceiveServiceBean sw = ReceiveService.getReceiveServiceBean() ;
            if( sw != null ) {
                return sw.getReceiveHeartBeatSwitch()[ sw.getReceiveHeartBeatPosition() ] ;
            }
            
        }
        
        return null ;
        
    }
    
    /**
     * 受信戻りエラースイッチスレッドを取得.
     * <BR><BR>
     * 受信戻りエラースイッチスレッドを取得します.
     * <BR>
     * @return ReceiveSwitch 受信スイッチ情報が返されます.
     */
    public static final ReceiveSwitch getReceiveResultErrorSwitch() {
        
        synchronized( SYNC ) {
            
            ReceiveServiceBean sw = ReceiveService.getReceiveServiceBean() ;
            if( sw != null ) {
                return sw.getReceiveResultErrorSwitch()[ sw.getReceiveResultErrorPosition() ] ;
            }
            
        }
        
        return null ;
        
    }
    
    /**
     * 受信戻り正常スイッチスレッドを取得.
     * <BR><BR>
     * 受信戻り正常スイッチスレッドを取得します.
     * <BR>
     * @return ReceiveSwitch 受信スイッチ情報が返されます.
     */
    public static final ReceiveSwitch getReceiveResultSuccessSwitch() {
        
        synchronized( SYNC ) {
            
            ReceiveServiceBean sw = ReceiveService.getReceiveServiceBean() ;
            if( sw != null ) {
                return sw.getReceiveResultSuccessSwitch()[ sw.getReceiveResultSuccessPosition() ] ;
            }
            
        }
        
        return null ;
        
    }
    
    /**
     * オブジェクトが有効であるかチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オブジェクトは有効です.<BR>
     *                 [false]が返された場合、オブジェクトは無効です.
     */
    public static final boolean isUse() {
        synchronized( SYNC ) {
            
            ReceiveServiceBean sw = ReceiveService.getReceiveServiceBean() ;
            if( sw != null ) {
                
                int i ;
                int len ;
                
                // コア受信オブジェクトチェック.
                if( sw.getCoreReceiveThread() == null ||
                    ( len = sw.getCoreReceiveThread().length ) <= 0 ) {
                    return false ;
                }
                for( i = 0 ; i < len ; i ++ ) {
                    if( sw.getCoreReceiveThread()[ i ] == null ||
                        sw.getCoreReceiveThread()[ i ].isThread() == false ) {
                        return false ;
                    }
                }
                
                // スイッチ受信オブジェクトチェック.
                if( sw.getSwitchReceiveThread() == null ||
                    ( len = sw.getSwitchReceiveThread().length ) <= 0 ) {
                    return false ;
                }
                for( i = 0 ; i < len ; i ++ ) {
                    if( sw.getSwitchReceiveThread()[ i ] == null ||
                        sw.getSwitchReceiveThread()[ i ].isThread() == false ) {
                        return false ;
                    }
                }
                
                // データ電文処理スレッドオブジェクトチェック.
                if( sw.getReceiveDataSwitch() == null ||
                    ( len = sw.getReceiveDataSwitch().length ) <= 0 ) {
                    return false ;
                }
                for( i = 0 ; i < len ; i ++ ) {
                    if( sw.getReceiveDataSwitch()[ i ] == null ||
                        sw.getReceiveDataSwitch()[ i ].isThread() == false ) {
                        return false ;
                    }
                }
                
                // ハートビート電文処理スレッドオブジェクトチェック.
                if( sw.getReceiveHeartBeatSwitch() == null ||
                    ( len = sw.getReceiveHeartBeatSwitch().length ) <= 0 ) {
                    return false ;
                }
                for( i = 0 ; i < len ; i ++ ) {
                    if( sw.getReceiveHeartBeatSwitch()[ i ] == null ||
                        sw.getReceiveHeartBeatSwitch()[ i ].isThread() == false ) {
                        return false ;
                    }
                }
                
                // 戻りエラー電文処理スレッドオブジェクトチェック.
                if( sw.getReceiveResultErrorSwitch() == null ||
                    ( len = sw.getReceiveResultErrorSwitch().length ) <= 0 ) {
                    return false ;
                }
                for( i = 0 ; i < len ; i ++ ) {
                    if( sw.getReceiveResultErrorSwitch()[ i ] == null ||
                        sw.getReceiveResultErrorSwitch()[ i ].isThread() == false ) {
                        return false ;
                    }
                }
                
                // 戻り正常電文処理スレッドオブジェクトチェック.
                if( sw.getReceiveResultSuccessSwitch() == null ||
                    ( len = sw.getReceiveResultSuccessSwitch().length ) <= 0 ) {
                    return false ;
                }
                for( i = 0 ; i < len ; i ++ ) {
                    if( sw.getReceiveResultSuccessSwitch()[ i ] == null ||
                        sw.getReceiveResultSuccessSwitch()[ i ].isThread() == false ) {
                        return false ;
                    }
                }
                
                // 全て動作確認OK.
                return true ;
                
            }
            
        }
        
        return false ;
    }
    
    /**
     * このオブジェクトが破棄されたかチェック.
     * <BR><BR>
     * このオブジェクトが破棄されたかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.
     */
    public static final boolean isDestroy() {
        synchronized( SYNC ) {
            return ( NamingManager.get( NAMING_MANAGER ) == null ) ? false : true ;
        }
    }
    
    /**
     * ReceiveServiceBeanオブジェクトを取得.
     */
    private static final ReceiveServiceBean getReceiveServiceBean() {
        Object o = NamingManager.get( NAMING_MANAGER ) ;
        if( o != null && o instanceof ReceiveServiceBean ) {
            return ( ReceiveServiceBean )o ;
        }
        return null ;
    }
    
}

/**
 * 受信サービスBeanオブジェクト.
 */
class ReceiveServiceBean {
    
    /**
     * コア受信データテーブル.
     */
    private CoreReceiveTable coreReceiveTable = null ;
    
    /**
     * コア受信スレッド.
     */
    private CoreReceiveThread[] coreReceiveThread = null ;
    
    /**
     * スイッチ受信スレッド.
     */
    private SwitchReceiveThread[] switchReceiveThread = null ;
    
    
    
    /**
     * データ電文オブジェクト群.
     */
    private ReceiveSwitchImple[] receiveDataSwitch = null ;
    
    /**
     * ハートビート電文オブジェクト群.
     */
    private ReceiveSwitchImple[] receiveHeartBeatSwitch = null ;
    
    /**
     * 戻りエラー電文オブジェクト群.
     */
    private ReceiveSwitchImple[] receiveResultErrorSwitch = null ;
    
    /**
     * 戻り正常電文オブジェクト群.
     */
    private ReceiveSwitchImple[] receiveResultSuccessSwitch = null ;
    
    /**
     * データ電文オブジェクトポジション.
     */
    private int receiveDataPosition = 0 ;
    
    /**
     * ハートビート電文オブジェクトポジション.
     */
    private int receiveHeartBeatPosition = 0 ;
    
    /**
     * 戻りエラー電文オブジェクトポジション.
     */
    private int receiveResultErrorPosition = 0 ;
    
    /**
     * 戻り正常電文オブジェクトポジション.
     */
    private int receiveResultSuccessPosition = 0 ;
    
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクトクリア.
     * <BR><BR>
     * オブジェクトをクリアします.
     */
    public void clear() {
        coreReceiveTable = null ;
        coreReceiveThread = null ;
        switchReceiveThread = null ;
        receiveDataSwitch = null ;
        receiveHeartBeatSwitch = null ;
        receiveResultErrorSwitch = null ;
        receiveResultSuccessSwitch = null ;
        receiveDataPosition = 0 ;
        receiveHeartBeatPosition = 0 ;
        receiveResultErrorPosition = 0 ;
        receiveResultSuccessPosition = 0 ;
    }
    
    

    /**
     * coreReceiveTableを取得.
     * <BR><BR>
     * @return coreReceiveTable が返されます。
     */
    public CoreReceiveTable getCoreReceiveTable() {
        return coreReceiveTable;
    }

    /**
     * coreReceiveTableを設定.
     * <BR><BR>
     * @param coreReceiveTable 対象のcoreReceiveTableを設定.
     */
    public void setCoreReceiveTable(CoreReceiveTable coreReceiveTable) {
        this.coreReceiveTable = coreReceiveTable;
    }

    /**
     * coreReceiveThreadを取得.
     * <BR><BR>
     * @return coreReceiveThread が返されます。
     */
    public CoreReceiveThread[] getCoreReceiveThread() {
        return coreReceiveThread;
    }

    /**
     * coreReceiveThreadを設定.
     * <BR><BR>
     * @param coreReceiveThread 対象のcoreReceiveThreadを設定.
     */
    public void setCoreReceiveThread(CoreReceiveThread[] coreReceiveThread) {
        this.coreReceiveThread = coreReceiveThread;
    }

    /**
     * switchReceiveThreadを取得.
     * <BR><BR>
     * @return switchReceiveThread が返されます。
     */
    public SwitchReceiveThread[] getSwitchReceiveThread() {
        return switchReceiveThread;
    }

    /**
     * switchReceiveThreadを設定.
     * <BR><BR>
     * @param switchReceiveThread 対象のswitchReceiveThreadを設定.
     */
    public void setSwitchReceiveThread(SwitchReceiveThread[] switchReceiveThread) {
        this.switchReceiveThread = switchReceiveThread;
    }
    
    /**
     * receiveDataSwitchを取得.
     * <BR><BR>
     * @return receiveDataSwitch が返されます。
     */
    public ReceiveSwitchImple[] getReceiveDataSwitch() {
        return receiveDataSwitch;
    }

    /**
     * receiveDataSwitchを設定.
     * <BR><BR>
     * @param receiveDataSwitch 対象のreceiveDataSwitchを設定.
     */
    public void setReceiveDataSwitch(ReceiveSwitchImple[] receiveDataSwitch) {
        this.receiveDataSwitch = receiveDataSwitch;
    }

    /**
     * receiveHeartBeatSwitchを取得.
     * <BR><BR>
     * @return receiveHeartBeatSwitch が返されます。
     */
    public ReceiveSwitchImple[] getReceiveHeartBeatSwitch() {
        return receiveHeartBeatSwitch;
    }

    /**
     * receiveHeartBeatSwitchを設定.
     * <BR><BR>
     * @param receiveHeartBeatSwitch 対象のreceiveHeartBeatSwitchを設定.
     */
    public void setReceiveHeartBeatSwitch(
            ReceiveSwitchImple[] receiveHeartBeatSwitch) {
        this.receiveHeartBeatSwitch = receiveHeartBeatSwitch;
    }

    /**
     * receiveResultErrorSwitchを取得.
     * <BR><BR>
     * @return receiveResultErrorSwitch が返されます。
     */
    public ReceiveSwitchImple[] getReceiveResultErrorSwitch() {
        return receiveResultErrorSwitch;
    }

    /**
     * receiveResultErrorSwitchを設定.
     * <BR><BR>
     * @param receiveResultErrorSwitch 対象のreceiveResultErrorSwitchを設定.
     */
    public void setReceiveResultErrorSwitch(
            ReceiveSwitchImple[] receiveResultErrorSwitch) {
        this.receiveResultErrorSwitch = receiveResultErrorSwitch;
    }

    /**
     * receiveResultSuccessSwitchを取得.
     * <BR><BR>
     * @return receiveResultSuccessSwitch が返されます。
     */
    public ReceiveSwitchImple[] getReceiveResultSuccessSwitch() {
        return receiveResultSuccessSwitch;
    }

    /**
     * receiveResultSuccessSwitchを設定.
     * <BR><BR>
     * @param receiveResultSuccessSwitch 対象のreceiveResultSuccessSwitchを設定.
     */
    public void setReceiveResultSuccessSwitch(
            ReceiveSwitchImple[] receiveResultSuccessSwitch) {
        this.receiveResultSuccessSwitch = receiveResultSuccessSwitch;
    }

    /**
     * receiveDataPositionを取得.
     * <BR><BR>
     * @return receiveDataPosition が返されます。
     */
    public int getReceiveDataPosition() {
        receiveDataPosition ++ ;
        if( receiveDataSwitch != null &&
            receiveDataSwitch.length <= receiveDataPosition ) {
            receiveDataPosition = 0 ;
        }
        return receiveDataPosition;
    }

    /**
     * receiveHeartBeatPositionを取得.
     * <BR><BR>
     * @return receiveHeartBeatPosition が返されます。
     */
    public int getReceiveHeartBeatPosition() {
        receiveHeartBeatPosition ++ ;
        if( receiveHeartBeatSwitch != null &&
            receiveHeartBeatSwitch.length <= receiveHeartBeatPosition ) {
            receiveHeartBeatPosition = 0 ;
        }
        return receiveHeartBeatPosition;
    }

    /**
     * receiveResultErrorPositionを取得.
     * <BR><BR>
     * @return receiveResultErrorPosition が返されます。
     */
    public int getReceiveResultErrorPosition() {
        receiveResultErrorPosition ++ ;
        if( receiveResultErrorSwitch != null &&
            receiveResultErrorSwitch.length <= receiveResultErrorPosition ) {
            receiveResultErrorPosition = 0 ;
        }
        return receiveResultErrorPosition;
    }

    /**
     * receiveResultSuccessPositionを取得.
     * <BR><BR>
     * @return receiveResultSuccessPosition が返されます。
     */
    public int getReceiveResultSuccessPosition() {
        receiveResultSuccessPosition ++ ;
        if( receiveResultSuccessSwitch != null &&
            receiveResultSuccessSwitch.length <= receiveResultSuccessPosition ) {
            receiveResultSuccessPosition = 0 ;
        }
        return receiveResultSuccessPosition;
    }
    
}

