/*
 * @(#)SwitchReceiveThread.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.receive.core ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.main.channel.protocol.ChannelProtocol;
import org.maachang.queue.main.channel.service.receive.ReceiveService;
import org.maachang.queue.main.channel.service.receive.ReceiveSwitch;

/**
 * スイッチ受信スレッド.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class SwitchReceiveThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( SwitchReceiveThread.class ) ;
    
    /**
     * スレッド項番.
     */
    private int threadNum = 0 ;
    
    /**
     * コア受信テーブル.
     */
    private CoreReceiveTable coreReceiveTable = null ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    private SwitchReceiveThread(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 処理分別を行うためのコアスレッド処理を生成します.
     * <BR>
     * @param threadNum スレッド項番を設定します.
     * @param coreReceiveTable テーブルオブジェクトを設定します.
     */
    public SwitchReceiveThread( int threadNum,CoreReceiveTable coreReceiveTable ){
        
        sync.create() ;
        
        try{
            
            this.coreReceiveTable = coreReceiveTable ;
            this.threadNum = threadNum ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        sync.clear() ;
        thread.clear() ;
        threadNum = 0 ;
        coreReceiveTable = null ;
    }
    
    /**
     * テーブルオブジェクトを取得.
     * <BR><BR>
     * テーブルオブジェクトを取得します.
     * <BR>
     * @return CoreReceiveTable テーブルオブジェクトが返されます.
     */
    public final CoreReceiveTable getCoreReceiveTable() {
        return coreReceiveTable ;
    }
    
    /**
     * スレッド項番を取得.
     * <BR><BR>
     * スレッド項番を取得します.
     * <BR>
     * @return int スレッド項番が返されます.
     */
    public final int getThreadNum() {
        return this.threadNum ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        CoreReceiveTable table = null ;
        
        try{
            
            //UtilCom.idleTime() ;
            
            table = this.coreReceiveTable ;
            
            // テーブルにデータが存在する場合.
            if( table.size() > 0 ) {
                
                // 受信情報を取得.
                CoreReceiveChild child = table.get() ;
                
                // 情報が存在する場合は、区分に合わせて処理.
                if( child != null && child.getBinResource() != null ) {
                    
                    ReceiveSwitch sw = null ;
                    
                    // プロトコルタイプによって、処理を分ける.
                    switch( ChannelProtocol.getTelegramType( child.getBinResource() ) ) {
                        case ChannelProtocol.PROTOCOL_DATA :      // データ処理.
                            sw = ReceiveService.getReceiveDataSwitch() ;
                            break ;
                        case ChannelProtocol.PROTOCOL_HEARTBEAT : // ハートビート処理.
                            sw = ReceiveService.getReceiveHeartBeatSwitch() ;
                            break ;
                        case ChannelProtocol.PROTOCOL_SUCCESS :   // 正常戻り値.
                            sw = ReceiveService.getReceiveResultSuccessSwitch() ;
                            break ;
                        case ChannelProtocol.PROTOCOL_ERROR :     // 異常戻り値.
                            sw = ReceiveService.getReceiveResultErrorSwitch() ;
                            break ;
                    }
                    
                    // 条件が見つかった場合は、対象オブジェクトをセット.
                    if( sw != null ) {
                        sw.setCoreReceiveChild( child ) ;
                    }
                    
                }
                
            }
            // テーブルにデータが存在しない場合.
            else {
                // 待機.
                UtilCom.idleTime() ;
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました[" + this.threadNum + "]", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました[" + this.threadNum + "]", e ) ;
        }finally{
            
        }
        
    }
    
}

