/*
 * @(#)LockSendQueue.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.send ;

import java.util.HashMap;

import org.maachang.commons.exception.InputException;
import org.maachang.queue.main.queue.Queue;

/**
 * 送信キューの、処理ロックを管理するオブジェクト.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class LockSendQueue {
    
    /**
     * 区切り文字.
     */
    private static final String CUT_NAME = "@" ;
    
    /**
     * キューロック管理.
     */
    private HashMap table = null ;
    
    /**
     * コンストラクタ.
     */
    public LockSendQueue() {
        table = new HashMap() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public synchronized final void destroy() {
        table = null ;
    }
    
    /**
     * 対象キュー条件をロック.
     * <BR><BR>
     * 指定送信キューのロック条件を設定します.
     * <BR>
     * @param queue 対象のキューオブジェクトを設定します.
     * @param threadNum 処理対象のスレッド項番を設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合は、既にロック条件として、設定されています.<BR>
     *                 [false]が返された場合、新しくロック条件として、設定されました.
     */
    public synchronized boolean lock( Queue queue,int threadNum ) {
        
        String name = this.getQueueByString( queue ) ;
        
        if( table.get( name ) != null ) {
            return true ;
        }
        Object[] objs = new Object[ 2 ] ;
        objs[ 0 ] = new Integer( threadNum ) ;
        objs[ 1 ] = new Long( System.currentTimeMillis() ) ;
        table.put( name,objs ) ;
        return false ;
    }
    
    /**
     * 対象キュー条件をアンロック.
     * <BR><BR>
     * 対象のキュー条件をアンロックします.
     * <BR>
     * @param queue 対象のキュー条件を設定します.
     */
    public synchronized void unlock( Queue queue ) {
        
        String name = this.getQueueByString( queue ) ;
        table.remove( name ) ;
        
    }
    
    /**
     * 指定キューがロックしている、スレッド項番を取得.
     * <BR><BR>
     * 指定キューがロックしている、スレッド項番を取得します.
     * <BR>
     * @param queue 対象のキューオブジェクトを設定します.
     * @return int ロックしているスレッド項番が返されます.<BR>
     *             [-1]が返された場合、ロックしている対象は存在しません.
     */
    public synchronized int getLockThreadNum( Queue queue ) {
        
        Object[] objs = null ;
        String name = this.getQueueByString( queue ) ;
        
        if( ( objs = ( Object[] )table.get( name ) ) != null  && objs.length == 2 ) {
            
            return ( ( Integer )objs[ 0 ] ).intValue() ;
        }
        
        return -1 ;
        
    }
    
    /**
     * 指定キューがロックしている、ロック開始時間を取得.
     * <BR><BR>
     * 指定キューがロックしている、ロック開始時間を取得します.
     * <BR>
     * @param queue 対象のキューオブジェクトを設定します.
     * @return long ロック開始時間が返されます.<BR>
     *             [-1L]が返された場合、ロックしている対象は存在しません.
     */
    public synchronized long getLockTime( Queue queue ) {
        
        Object[] objs = null ;
        String name = this.getQueueByString( queue ) ;
        
        if( ( objs = ( Object[] )table.get( name ) ) != null  && objs.length == 2 ) {
            
            return ( ( Long )objs[ 1 ] ).longValue() ;
        }
        
        return -1L ;
        
    }
    
    /**
     * 対象キュー条件がロックされているかチェック.
     * <BR><BR>
     * 対象キュー条件がロックされているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合は、既にロック条件として、設定されています.<BR>
     *                 [false]が返された場合、新しくロック条件として、設定されました.
     */
    public synchronized boolean isLock( Queue queue ) {
        
        String name = this.getQueueByString( queue ) ;
        
        if( table.get( name ) != null ) {
            return true ;
        }
        
        return false ;
    }
    
    
    
    /**
     * キューオブジェクトから、文字列を取得.
     */
    private final String getQueueByString( Queue queue ) {
        
        if( queue == null || queue.isQueue() == false ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        return new StringBuffer().
            append( queue.getState().getName() ).
            append( CUT_NAME ).
            append( queue.getState().getParentManagerName() ).
            toString() ;
        
    }
    
}

