/*
 * @(#)SendHeartBeatThread.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.send ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.channel.SendChannel;
import org.maachang.queue.main.channel.protocol.ChannelProtocol;
import org.maachang.queue.main.channel.protocol.ProtocolHeartBeat;
import org.maachang.queue.main.connect.Connect;
import org.maachang.queue.main.connect.ConnectFactory;

/**
 * ハートビート送信用スレッド.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class SendHeartBeatThread extends ExecutionThread
{
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( SendHeartBeatThread.class ) ;
    
    /**
     * 処理間隔 : 最小値.
     */
    public static final long MIN_TIME = 5000L ;
    
    /**
     * 処理間隔 : 最大値.
     */
    public static final long MAX_TIME = 60000L ;
    
    /**
     * コネクション不能までのカウント : 最小値.
     */
    public static final int MIN_CLOSE_COUNT = 3 ;
    
    /**
     * コネクション不能までのカウント : 最大値.
     */
    public static final int MAX_CLOSE_COUNT = 12 ;
    
    
    
    /**
     * 処理間隔.
     */
    private long time = -1L ;
    
    /**
     * コネクション不能までのカウント.
     */
    private int closeCount = -1 ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    private SendHeartBeatThread(){
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ハートビート処理間隔を設定して、オブジェクトを設定します.
     * <BR>
     * @param time ハートビート処理間隔を設定します.
     * @param closeCount コネクション不能までのカウント値を設定します.
     */
    public SendHeartBeatThread( long time,int closeCount ){
        
        sync.create() ;
        
        time = ( time <= MIN_TIME ) ? MIN_TIME : time ;
        time = ( time >= MAX_TIME ) ? MAX_TIME : time ;
        closeCount = ( closeCount <= MIN_CLOSE_COUNT ) ? MIN_CLOSE_COUNT : closeCount ;
        closeCount = ( closeCount >= MAX_CLOSE_COUNT ) ? MAX_CLOSE_COUNT : closeCount ;
        
        try{
            
            this.time = time ;
            this.closeCount = closeCount ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        sync.clear() ;
        thread.clear() ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        int i ;
        int len ;
        
        long time ;
        int closeCount ;
        
        try{
            
            //UtilCom.idleTime() ;
            
            synchronized( sync.get() ) {
                time = this.time ;
                closeCount = this.closeCount ;
            }
            
            len = ChannelFactory.size() ;
            
            if( len > 0 ) {
                
                // チャネル数分処理.
                for( i = len-1 ; i >= 0 ; i -- ) {
                    
                    // 待機処理.
                    UtilCom.idleTime() ;
                    
                    // １つのチャネルオブジェクトを取得.
                    Channel channel = ChannelFactory.getToNumber( i ) ;
                    if( channel == null ||
                        ( channel instanceof SendChannel ) == false ||
                        channel.isChannel() == false ) {
                        
                        continue ;
                    }
                    
                    // 送信チャネルを取得.
                    SendChannel sendChannel = ( SendChannel )channel ;
                    
                    // 対象チャネルステータスが「シャットダウン中」および
                    // 「シャットダウン」および「計画停止」の場合は処理しない.
                    if( channel.getState() == ChannelStatus.STATE_DOWN ||
                        channel.getState() == ChannelStatus.STATE_SHUTDOWN ||
                        channel.getState() == ChannelStatus.STATE_PLANNED_STOP ) {
                        continue ;
                    }
                    
                    try {
                        
                        ChannelOption opt = null ;
                        
                        // チャネルオブジェクトから、オプションを取得.
                        if( ( opt = ( ChannelOption )sendChannel.getOption() ) == null ) {
                            // 存在しない場合は、新規作成.
                            synchronized( channel.getSynchronized().get() ) {
                                opt = new ChannelOption() ;
                                opt.setSendProtocol( new ProtocolHeartBeat() ) ;
                                sendChannel.setOption( opt ) ;
                            }
                        }
                        
                        synchronized( opt ) {
                            
                            // 指定チャネルはハートビート送信条件である場合.
                            if( opt.getStartTime() + time <= System.currentTimeMillis() ) {
                                
                                // 送信プロトコルオブジェクトを取得.
                                ProtocolHeartBeat send = opt.getSendProtocol() ;
                                send.create( channel ) ;
                                
                                // ハートビート送信回数が規定値を上回った場合.
                                if( channel.getState() != ChannelStatus.STATE_NOT_CONNECT_CHANNEL &&
                                    opt.getSendCount() >= closeCount ) {
                                    channel.setState( ChannelStatus.STATE_NOT_HEART_BEAT ) ;
                                }
                                
                                // チャネルにセットされているコネクションオブジェクトを取得.
                                Connect conn = ConnectFactory.get( sendChannel.getConnectName() ) ;
                                
                                // コネクションオブジェクトが存在しない場合は処理しない.
                                if( channel.getState() != ChannelStatus.STATE_NOT_CONNECT_CHANNEL &&
                                    conn == null ) {
                                    channel.setState( ChannelStatus.STATE_NOT_HEART_BEAT ) ;
                                    continue ;
                                }
                                
                                // ハートビート送信テーブルに対して、コネクションポート番号を設定.
                                send.setReceivePort( conn.getPort() ) ;
                                
                                // 送信用プロトコルデータを生成.
                                BinResource bin = ChannelProtocol.getProtocolHeartBeatByTelegram( send ) ;
                                
                                // 次の条件を準備.
                                opt.addSendCount() ;
                                opt.setStartTime( System.currentTimeMillis() ) ;
                                
                                // 送信処理.
                                conn.sendByResource(
                                    sendChannel.getUseCb32Word(),
                                    sendChannel.getInetAddress(),
                                    sendChannel.getPort(),bin ) ;
                                
                            }
                            
                        }
                        
                    } catch( Exception e ) {
                        LOG.warn( "## チャネル[" + channel.getName() + "]のHeartBeat処理に失敗",e ) ;
                    } finally {
                        channel = null ;
                    }
                    
                }
            }
            else {
                UtilCom.idleTime() ;
            }
            
        } catch( OutOfMemoryError me ) {
            LOG.error( "OutOfMemoryError",me ) ;
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
        }
        
    }
    
}

