/*
 * @(#)AbstractMq.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.status.QueueManagerStatus;
import org.maachang.queue.access.status.QueueStatus;
import org.maachang.queue.access.status.StatusDefine;
import org.maachang.queue.main.manager.BaseQueueManager;
import org.maachang.queue.main.manager.DeleteQueueManagerException;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.manager.QueueManagerStateNotSuccessException;
import org.maachang.queue.main.queue.base.BaseQueue;
import org.maachang.queue.main.queue.base.BaseQueueBean;
import org.maachang.queue.main.queue.base.BaseQueueFactory;

/**
 * MQを表すオブジェクト.
 * <BR><BR>
 * 1つのMQを表すオブジェクトです.
 *  
 * @version 2006/09/04
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
abstract class AbstractMq implements Mq {
    
    /**
     * キューマネージャオブジェクト.
     */
    protected BaseQueueManager queueManager = null ;
    
    /**
     * キューオブジェクト.
     */
    protected BaseQueue queue = null ;
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクトを破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * また、この処理を行った場合、管理元のキューマネージャから
     * キュー情報を削除しません.
     */
    protected synchronized void destroy() {
        
        this.queueManager = null ;
        this.queue = null ;
        
    }
    
    /**
     * トランザクションを開始.
     * <BR><BR>
     * トランザクションを開始します.
     * <BR>
     * @return int 新しいトランザクションIDが返されます.
     * @return int トランザクションIDが返されます.<BR>
     *             [Queue.TRANSACTION_BY_AUTOCOMMIT]が返された場合、
     *             このキューはオートコミットです.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized int transaction()
        throws MaachangQException {
        
        this.checkState() ;
        
        int nowTran = this.getTransactionId() ;
        if( nowTran > 0 ) {
            throw new AlreadyExistenceException(
                "トランザクションは他のコネクションが保持しています" ) ;
        }
        
        int newTran = this.queue.transaction() ;
        if( newTran == Queue.TRANSACTION_BY_AUTOCOMMIT ) {
            return Queue.TRANSACTION_BY_AUTOCOMMIT ;
        }
        
        if( newTran == Queue.TRANSACTION_BY_ALREADY_EXISTENCE ) {
            throw new AlreadyExistenceException(
                "トランザクションは他の処理が保持しています" ) ;
        }
        
        return newTran ;
    }
    
    /**
     * コミット処理.
     * <BR><BR>
     * トランザクションに対する、コミット処理を実施します.
     * <BR>
     * @param transactionId 対象のトランザクションIDを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void commit( int transactionId )
        throws MaachangQException {
        this.commitOrRollback( transactionId,true ) ;
    }
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * トランザクションに対する、ロールバック処理を実施します.
     * <BR>
     * @param transactionId 対象のトランザクションIDを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void rollback( int transactionId )
        throws MaachangQException {
        this.commitOrRollback( transactionId,false ) ;
    }
    
    /**
     * キューマネージャオブジェクトを取得.
     * <BR><BR>
     * キューマネージャオブジェクトを取得します.
     * <BR>
     * @return QueueManager キューマネージャオブジェクトが返されます.
     */
    public synchronized QueueManager getQueueManager() {
        
        QueueManager ret = null ;
        
        if(
            this.queueManager != null &&
            this.queueManager.isQueueManager() == true
        ) {
            ret = this.queueManager ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューオブジェクトを取得.
     * <BR><BR>
     * キューオブジェクトを取得します.
     * <BR>
     * @return Queue キューオブジェクトが返されます.
     */
    protected synchronized Queue getQueue() {
        
        Queue ret = null ;
        
        if(
            this.queueManager != null &&
            this.queueManager.isQueueManager() == true &&
            this.queue != null &&
            this.queue.isQueue() == true
         ) {
            ret = this.queue ;
        }
        
        return ret ;
    }
    
    /**
     * 対象QueueBean情報を設定.
     * <BR><BR>
     * 対象QueueBean情報を設定します.
     * <BR>
     * @param bean 設定対象のQueueBeanを設定します.
     */
    public synchronized void setQueueBean( QueueBean bean ) {
        
        this.queue.setBean( bean ) ;
    }
    
    /**
     * QueueBeanオブジェクトを取得.
     * <BR><BR>
     * QueueBeanオブジェクトを取得します.
     * <BR>
     * @return AdminQueueBean QueueBeanオブジェクトが返されます.
     */
    public synchronized QueueBean getQueueBean() {
        
        QueueBean ret = null ;
        
        if(
            this.queueManager != null &&
            this.queueManager.isQueueManager() == true &&
            this.queue != null &&
            this.queue.isQueue() == true
        ) {
            ret = this.queue.getQueueBean() ;
        }
        
        return ret ;
    }
    
    /**
     * QueueKeyオブジェクトを取得.
     * <BR><BR>
     * QueueKeyオブジェクトを取得します.
     * <BR>
     * @return QueueKey QueueKeyオブジェクトが返されます.
     */
    public synchronized QueueKey getQueueKey() {
        
        QueueKey ret = null ;
        
        if(
            this.queueManager != null &&
            this.queueManager.isQueueManager() == true &&
            this.queue != null &&
            this.queue.isQueue() == true
         ) {
            ret = this.queue.getKey() ;
        }
        
        return ret ;
        
    }
    
    /**
     * 現在のMQステータスを取得.
     * <BR><BR>
     * 現在のMQステータスを取得します.
     * <BR>
     * @return int MQステータスが返されます.<BR>
     *             [STATE_SUCCESS]が返された場合、「正常」です.<BR>
     *             [STATE_DOWN]が返された場合、「停止」です.<BR>
     *             [STATE_STARTUP]が返された場合、「起動中」です.<BR>
     *             [STATE_SHUTDOWN]が返された場合、「停止」です.<BR>
     *             [STATE_ERROR]が返された場合、「異常」です.<BR>
     *             [STATE_WARNING]が返された場合、「キュー格納数指定量による警告」です.<BR>
     *             [STATE_FULL]が返された場合、「キュー満杯」です.
     */
    public synchronized int getState() {
        
        int ret = QueueStatus.STATE_DOWN ;
        
        if(
            this.queueManager != null &&
            this.queueManager.isQueueManager() == true &&
            this.queue != null &&
            this.queue.isQueue() == true
         ) {
            ret = this.queue.getState().getState() ;
        }
        
        return ret ;
    }
    
    /**
     * 現在の格納電文数を取得.
     * <BR><BR>
     * 現在の格納電文数を取得します.
     * <BR>
     * @return int 格納されている電文数が返されます.
     */
    public synchronized int size() {
        try {
            return this.queue.size() ;
        } catch( Exception e ) {
        }
        
        return 0 ;
    }
    
    /**
     * 現在の最大キュー数を取得.
     * <BR>BR>
     * 現在の最大キュー数を取得します.
     * <BR>
     * @return int 現在の最大キュー数が返されます.
     */
    public synchronized int getMaxSize() {
        try {
            return this.queue.getState().getMaxQueue() ;
        } catch( Exception e ) {
        }
        
        return 0 ;
    }
    
    /**
     * 現在のトランザクションIDを取得.
     * <BR><BR>
     * 現在のトランザクションIDを取得します.
     * <BR>
     * @return int 現在のトランザクションIDが返されます.
     */
    public synchronized int getTransactionId() {
        try {
            return this.queue.getTransactionId() ;
        } catch( Exception e ) {
        }
        
        return -1 ;
    }
    
    /**
     * キュー同期オブジェクトを取得.
     * <BR><BR>
     * キュー同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized キュー同期オブジェクトが返されます.
     */
    public synchronized Synchronized getSynchronized() {
        try {
            return this.queue.getSynchronized() ;
        } catch( Exception e ) {
        }
        return null ;
    }
    
    /**
     * このキューがオートコミットであるかチェック.
     * <BR><BR>
     * このキューがオートコミットであるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オートコミットです.<BR>
     *                 [false]が返された場合、オートコミットではありません.
     */
    public synchronized boolean isAutoCommit() {
        try {
            return this.queue.isAuthCommit() ;
        } catch( Exception e ) {
        }
        
        return false ;
    }
    
    /**
     * MQが有効であるかチェック.
     * <BR><BR>
     * MQが有効であるかチェックします.<BR>
     * またこのオブジェクトの有効性を見るだけでない、
     * Queueが登録されていない場合も、無効扱いとなります.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、MQは有効です.<BR>
     *                 [false]が返された場合、MQは無効です.
     */
    public synchronized boolean isUseMq() {
        
        boolean ret = false ;
        Queue obj = null ;
        QueueManager qman = null ;
        
        try {
            if( 
                queue != null &&
                (
                    qman = QueueManagerFactory.get(
                        queue.getState().getParentManagerName()
                    )
                ) != null &&
                qman.isQueueManager() == true &&
                (
                    obj = BaseQueueFactory.get(
                        queue.getState().getQueueType(),
                        queue.getState().getParentManagerName(),
                        queue.getState().getName()
                    )
                ) != null
            ) {
                ret = obj.isQueue() ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象キューに設定可能かチェック.
     * <BR><BR>
     * 対象キューに設定可能かチェックします.
     * <BR>
     * @param len 対象の電文長を設定します.
     * @exception MessageOverException メッセージオーバー例外.
     * @exception QueueFullException キューフル例外.
     */
    public synchronized void isPut( int len )
        throws MessageOverException,QueueFullException {
        queue.isPut( len ) ;
    }
    
    /**
     * 新しいMQを生成.
     * <BR><BR>
     * 新しいMQを生成します.
     * <BR>
     * @param bean 対象のMQBeanを生成します.
     * @return BaseQueue 生成されたMQオブジェクトが返されます.
     * @exception InputException 入力例外.
     */
    protected synchronized BaseQueue createMq( BaseQueueBean bean )
        throws InputException {
        
        return BaseQueueFactory.create( bean.getType(),bean.isGzip(),bean.getQueueManagerName(),
            bean.getQueueName(),bean.getOption(),bean.getWarning(),bean.getMaxMessageLength(),
            bean.getMaxQueue(),bean.isAutoCommit(),bean.getCacheName() ) ;
        
    }
    
    /**
     * キューマネージャ、キューステータスチェック.
     * <BR><BR>
     * キューマネージャ、キューステータスチェックします.
     * <BR>
     * @exception MaachangQException MaachangQ例外.
     */
    protected void checkState()
        throws MaachangQException {
        
        if( this.queueManager == null || this.queueManager.isQueueManager() == false ) {
            throw new DeleteQueueManagerException(
                "キューマネージャは削除されているか、不明です" ) ;
        }
        if( this.queue == null || this.isUseMq() == false ) {
            throw new DeleteQueueException(
                "キューは削除されているか、不明です" ) ;
        }
        
        int state ;
        state = this.queueManager.getState() ;
        if( ( state & StatusDefine.STATE_MASK_ERROR ) == StatusDefine.STATE_MASK_ERROR ) {
            throw new QueueManagerStateNotSuccessException(
                "対象のキューマネージャ(" + this.queueManager.getName() +
                ")はステータスが異常(" +
                QueueManagerStatus.getStateByString( state ) +
                ")です" ) ;
        }
        
        state = this.queue.getState().getState() ;
        if( ( state & StatusDefine.STATE_MASK_ERROR ) == StatusDefine.STATE_MASK_ERROR ) {
            String sendReceive =
                ( this.queue.getState().getQueueType() ==
                    MaachangQAccessDefine.TYPE_SEND ) ?
                    "送信" : "受信" ;
            throw new QueueStateNotSuccessException(
                "対象の" + sendReceive +
                "キュー(" + this.queue.getState().getName() +
                ")はステータスが異常(" +
                QueueStatus.getStateByString( state ) +
                ")です" ) ;
        }
        
    }
    
    /**
     * コミットor ロールバック.
     */
    private void commitOrRollback( int tranId,boolean mode )
        throws MaachangQException {
        
        this.checkState() ;
        
        if( this.queue.isAuthCommit() == false ) {
            
            int nowTran = this.getTransactionId() ;
            
            if( nowTran == Queue.TRANSACTION_BY_ALREADY_EXISTENCE ||
                nowTran != tranId ) {
                
                if( nowTran == Queue.TRANSACTION_BY_ALREADY_EXISTENCE ) {
                    throw new NotTransactionException(
                        "トランザクションは開始されていません" ) ;
                }
                throw new AlreadyExistenceException(
                    "トランザクションは他のコネクションが保持しています" ) ;
            }
            
            if( mode == true ) {
                this.queue.commit( tranId ) ;
            }
            else {
                this.queue.rollback( tranId ) ;
            }
            
        }
    }
    
}

