/*
 * @(#)QueueUtil.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.BinMemoryResource;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.Resource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.util.zip.GZIPBinary;
import org.maachang.queue.config.MqDefine;

/**
 * キューユーティリティ.
 *  
 * @version 2006/09/04
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class QueueUtil
{
    
    /**
     * コンストラクタ.
     */
    private QueueUtil() {
        
    }
    
    /**
     * 対象オブジェクトからデータ長を取得.
     * <BR><BR>
     * 対象オブジェクトからデータ長を取得します.
     * <BR>
     * @param message 対象のメッセージオブジェクトを設定します.<BR>
     *                このオブジェクト型は以下の内容である必要があります.<BR>
     *                byte[].<BR>
     *                org.maachang.commons.resource.BinResource.<BR>
     *                org.maachang.queue.main.queue.QueueMessage.
     * @return int 対象のメッセージ長が返されます.<BR>
     *             引数条件が不正な場合は[0]が返されます.
     */
    public static final int getMessageLength( Object message ) {
        
        int ret = 0 ;
        
        if(
            message != null &&
            (
                ( message instanceof byte[] ) == true ||
                ( message instanceof BinResource ) == true ||
                ( message instanceof QueueMessage ) == true
            )
        ) {
            if( message instanceof byte[] ) {
                ret = ( ( byte[] )message ).length ;
            }
            else if( message instanceof BinResource ) {
                ret = ( ( BinResource )message ).size() ;
            }
            else {
                ret = ( ( QueueMessage )message ).getLength() ;
            }
        }
        
        return ret ;
    }
    
    /**
     * メッセージ情報をQueueMessageに変換.
     * <BR><BR>
     * 指定メッセージ情報をQueueMessageに変換します.
     * <BR>
     * @param gzipFlag 圧縮フラグを設定します.
     * @param resType リソースタイプを設定します.
     * @param message 対象のメッセージオブジェクトを設定します.<BR>
     *                このオブジェクト型は以下の内容である必要があります.<BR>
     *                byte[].<BR>
     *                org.maachang.commons.resource.BinResource.
     * @param len 対象のメッセージデータ長を設定します.
     * @return QueueMessage キューメッセージが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final QueueMessage convertMessageByQueueMessage(
        boolean gzipFlag,ResourceType resType,Object message,int len )
        throws InputException,AccessException {
        
        boolean gzip = false ;
        boolean type = false ;
        
        Object msg = null ;
        QueueMessage ret = null ;
        
        // リソースタイプが存在しない場合はバイナリでキューにセット.
        // または、電文長が規定値以下の場合は、バイナリでセット.
        if(
            len <= MqDefine.CONVERT_RESOURCE_PAUSE_LENGTH ||
            resType == null
        ) {
            
            byte[] bin = null ;
            
            if( message instanceof byte[] ) {
                bin = ( byte[] )message ;
            }
            else {
                bin = ( ( BinResource )message ).getBinary() ;
            }
            
            // 圧縮条件の場合.
            if(
                gzipFlag == true &&
                len >= MqDefine.CONVERT_GZIP_PAUSE_LENGTH ) {
                
                bin = GZIPBinary.getInstance().
                    convertBinaryByGZIP( bin ) ;
                gzip = true ;
                
            }
            
            msg = bin ;
            type = false ;
            
        }
        // リソースタイプが存在する場合は、対象リソースでキューにセット.
        // または、電文長が規定値以上の場合は、リソースでセット.
        else {
            
            BinResource bin = null ;
            
            if( message instanceof byte[] ) {
                
                // 圧縮条件の場合.
                if(
                    gzipFlag == true &&
                    len >= MqDefine.CONVERT_GZIP_PAUSE_LENGTH ) {
                    
                    bin = Resource.createBinResource(
                        resType,1 ) ;
                    
                    GZIPBinary.getInstance().
                        convertBinaryByGZIP( bin,
                        new BinMemoryResource( ( byte[] )message ) ) ;
                    
                    gzip = true ;
                    
                }
                // 圧縮条件でない場合.
                else {
                    
                    bin = Resource.createBinResource(
                        resType,( byte[] )message ) ;
                        
                }
                
            }
            else {
                
                // 圧縮条件の場合.
                if(
                    gzipFlag == true &&
                    len >= MqDefine.CONVERT_GZIP_PAUSE_LENGTH ) {
                    
                    bin = Resource.createBinResource(
                        resType,1 ) ;
                    
                    GZIPBinary.getInstance().
                        convertBinaryByGZIP( bin,( BinResource )message ) ;
                    
                    gzip = true ;
                    
                }
                // 圧縮条件でない場合.
                else {
                    
                    bin = Resource.createBinResource(
                        resType,len ) ;
                    bin.setBinary( 0,( BinResource )message ) ;
                    
                }
                
            }
            
            msg = bin ;
            type = true ;
            
        }
        
        ret = new QueueMessage() ;
        ret.setType( type ) ;
        ret.setGzipFlag( gzip ) ;
        ret.setLength( len ) ;
        ret.setMessage( msg ) ;
        
        return ret ;
        
    }
    
    /**
     * QueueMessageオブジェクトをメッセージ情報に変換.
     * <BR><BR>
     * QueueMessageオブジェクトをメッセージ情報に変換します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     * @param message 変換対象のQueueMessageオブジェクトを設定します.
     * @return Object 変換されたメッセージ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final Object convertQueueMessageByMessage(
        ResourceType resType,QueueMessage message )
        throws InputException,AccessException {
        
        Object ret = null ;
        
        // 圧縮電文の場合.
        if( message.isGzipFlag() == true ) {
            
            // メッセージタイプがバイナリの場合.
            if( message.isType() == false ) {
                
                byte[] bin = null ;
                
                bin = GZIPBinary.getInstance().convertGZIPByBinary(
                    ( byte[] )message.getMessage() ) ;
                
                ret = bin ;
                
            }
            // メッセージタイプがBinResourceの場合.
            else {
                
                BinResource bin = null ;
                
                bin = Resource.createBinResource( resType,1 ) ;
                GZIPBinary.getInstance().convertGZIPByBinary(
                    bin,( BinResource )message.getMessage() ) ;
                
                ret = bin ;
                
            }
            
        }
        // 圧縮電文でない場合.
        else {
            
            ret = message.getMessage() ;
            
        }
        
        return ret ;
        
    }
    
}

