/*
 * @(#)QueueValue.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue ;

import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.Resource;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.main.queue.base.core.QArrayChild;

/**
 * キューデータ.
 * <BR><BR>
 * キューデータを表すオブジェクト.
 *  
 * @version 2006/08/27
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class QueueValue {
    
    /**
     * 電文ID.
     */
    private long telegramID = -1L ;
    
    /**
     * キュー電文を表す.
     */
    private QArrayChild child = null ;
    
    /**
     * メッセージ.
     */
    private Object message = null ;
    
    /**
     * キー情報.
     */
    private String key = null ;
    
    /**
     * プロセスID.
     */
    private String processId = null ;
    
    /**
     * expire.
     */
    private long expire = -1L ;
    
    /**
     * プライオリティ.
     */
    private int priority = -1 ;
    
    /**
     * トランザクションID.
     */
    private int id = -1 ;
    
    /**
     * コンストラクタ.
     */
    public QueueValue() {
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear() {
        telegramID = -1L ;
        message = null ;
        key = null ;
        processId = null ;
        expire = -1L ;
        priority = -1 ;
        id = -1 ;
    }
    
    /**
     * トランザクションIDを取得.
     * <BR><BR>
     * @return int トランザクションID.
     */
    public int getId() {
        return id;
    }
    
    /**
     * トランザクションIDを設定.
     * <BR><BR>
     * @param id トランザクションID.
     */
    public void setId(int id) {
        this.id = id;
    }
    
    /**
     * 電文IDを取得.
     * <BR><BR>
     * @return long 電文ID.
     */
    public long getTelegramID() {
        return this.telegramID ;
    }
    
    /**
     * 電文IDを設定.
     * <BR><BR>
     * @param telegramID 電文ID.
     */
    public void setTelegramID( long telegramID ) {
        this.telegramID = telegramID ;
    }
    
    /**
     * メッセージ情報を取得.
     * <BR><BR>
     * @return Object メッセージ情報.
     */
    public Object getMessage() {
        return message;
    }
    
    /**
     * メッセージ情報を設定.
     * <BR><BR>
     * @param message メッセージ情報.
     */
    public void setMessage(Object message) {
        if(
            message != null &&
            (
                ( message instanceof byte[] ) == true ||
                ( message instanceof BinResource ) == true
            )
        ) {
            this.message = message;
        }
    }
    
    /**
     * メッセージをBinResourceに変換して取得.
     * <BR><BR>
     * メッセージをBinResourceに変換して取得します.
     * <BR>
     * @return BinResource BinResourceに変換されたメッセージ情報が返されます.
     */
    public BinResource getMessageByBinResource() {
        return this.getMessageByBinResource( null ) ;
    }
    
    /**
     * メッセージをBinResourceに変換して取得.
     * <BR><BR>
     * メッセージをBinResourceに変換して取得します.
     * <BR>
     * @param resType リソースタイプを設定します.
     * @return BinResource BinResourceに変換されたメッセージ情報が返されます.
     */
    public BinResource getMessageByBinResource(
        ResourceType resType ) {
        
        BinResource ret = null ;
        
        try {
            if(
                message != null &&
                (
                    ( message instanceof byte[] ) == true ||
                    ( message instanceof BinResource ) == true
                )
            ) {
                if( message instanceof byte[] ) {
                    ret = Resource.createBinResource( resType,( byte[] )message ) ;
                }
                else {
                    
                    int len ;
                    BinResource res = null ;
                    
                    res = ( BinResource )message ;
                    len = res.size() ;
                    ret = Resource.createBinResource( resType,len ) ;
                    ret.setBinary( 0,res ) ;
                    
                    res = null ;
                    
                }
                
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * メッセージを文字列に変換して取得.
     * <BR><BR>
     * バイナリのメッセージを文字列に変換して取得します.
     * <BR>
     * @param charset 対象のキャラクタセットを設定します.<BR>
     *                [null]を設定した場合、デフォルトのキャラクタセットで
     *                変換します.
     * @return String 変換された文字列が返されます.
     */
    public String getMessageByString( String charset ) {
        
        String ret = null ;
        
        try {
            byte[] bin = this.getBinaryMessage() ;
            if( bin != null && bin.length > 0 ) {
                if( charset == null || ( charset = charset.trim() ).length() <= 0 ) {
                    ret = new String( bin ) ;
                }
                else {
                    ret = new String( bin,charset ) ;
                }
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * メッセージをバイナリで取得.
     * <BR><BR>
     * メッセージをバイナリで取得します.
     * <BR>
     * @return byte[] 変換された文字列が返されます.
     */
    public byte[] getBinaryMessage() {
        
        byte[] ret = null ;
        
        try {
            if(
                message != null &&
                (
                    ( message instanceof byte[] ) == true ||
                    ( message instanceof BinResource ) == true
                )
            ) {
                if( message instanceof byte[] ) {
                    ret = ( byte[] )message ;
                }
                else {
                    ret = ( ( BinResource )message ).getBinary() ;
                }
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * Expire値を取得.
     * <BR><BR>
     * @return long Expire値.
     */
    public long getExpire() {
        return expire;
    }
    
    /**
     * Expire値を設定.
     * <BR><BR>
     * @param expire Expire値.
     */
    public void setExpire(long expire) {
        this.expire = expire;
    }
    
    /**
     * Key情報を取得.
     * <BR><BR>
     * @return String key情報.
     */
    public String getKey() {
        return key;
    }
    
    /**
     * Key情報を設定.
     * <BR><BR>
     * @param key Key情報.
     */
    public void setKey(String key) {
        this.key = key;
    }
    
    /**
     * ProcessId情報を取得.
     * <BR><BR>
     * @return String ProcessId情報.
     */
    public String getProcessId() {
        return processId;
    }
    
    /**
     * ProcessId情報を設定.
     * <BR><BR>
     * @param processId ProcessId情報.
     */
    public void setProcessId(String processId) {
        this.processId = processId;
    }
    
    /**
     * プライオリティを取得.
     * <BR><BR>
     * @return int プライオリティ値.
     */
    public int getPriority() {
        return priority;
    }
    
    /**
     * プライオリティを設定.
     * <BR><BR>
     * @param priority プライオリティ値.
     */
    public void setPriority(int priority) {
        this.priority = priority;
    }
    
    /**
     * キュー配列要素を設定.
     * <BR><BR>
     * @param child キュー配列要素.
     */
    public void setChild( QArrayChild child ) {
        if( child != null ) {
            this.child = child ;
        }
    }
    
    /**
     * キュー配列要素を取得.
     * <BR><BR>
     * @return QArrayChild キュー配列要素.
     */
    public QArrayChild getChild() {
        QArrayChild ret = this.child ;
        this.child = null ;
        return ret ;
    }
    
    /**
     * オブジェクトを文字列に変換.
     * <BR><BR>
     * オブジェクトを文字列に変換します.
     * <BR>
     * @return String 変換された文字列が返されます.
     */
    public String toString() {
        
        int msgLen = 0 ;
        Object o = null ;
        String msg = null ;
        
        if(
            message != null &&
            (
                ( message instanceof byte[] ) == true ||
                ( message instanceof BinResource ) == true
            )
        ) {
            o = message ;
            if( ( message instanceof byte[] ) == true ) {
                msgLen = ( ( byte[] )o ).length ;
                msg = UtilCom.convertBinaryTo16String( ( byte[] )o,',',0,msgLen ) ;
            }
            else {
                msgLen = ( ( BinResource )o ).size() ;
                msg = UtilCom.convertBinaryTo16String( ( BinResource )o,',',0,msgLen ) ;
            }
        }
        
        return new StringBuffer().
            append( "transactionID:" ).
            append( id ).
            append( " telegramID:" ).
            append( telegramID ).
            append( " key:" ).
            append( key ).
            append( " processId;" ).
            append( processId ).
            append( " messageLength:" ).
            append( msgLen ).
            append( " message:" ).
            append( msg ).
            append( " priority:" ).
            append( priority ).
            append( " expire:" ).
            append( expire ).
            toString() ;
    }
    
}

