/*
 * @(#)BaseQueueFactory.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.base ;

import java.io.Serializable;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.sys.NamingManager;
import org.maachang.commons.util.NumberTable;
import org.maachang.commons.util.SearchPortion;
import org.maachang.commons.util.array.ObjectArray;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.status.QueueStatus;
import org.maachang.queue.main.cache.MqCache;
import org.maachang.queue.main.cache.MqCacheFactory;
import org.maachang.queue.main.manager.BaseQueueManager;
import org.maachang.queue.main.manager.QueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.Queue;

/**
 * 基本キューファクトリ.
 * <BR><BR>
 * 基本キューを管理するファクトリです.
 *  
 * @version 2006/08/27
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class BaseQueueFactory
{
    
    /**
     * ネーミングマネージャ登録拡張子.
     */
    public static final String NAMING_PLUS = "@maachangq.queue.factory" ;
    
    /**
     * ネーミングマネージャ登録拡張子(キュータイプ管理).
     */
    public static final String NAMING_QUEUE_TYPE_PLUS = "QUEUE_FACTORY@maachangq.queue.type" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public BaseQueueFactory() {
        
    }
    
    /**
     * キュー情報を全て破棄.
     * <BR><BR>
     * キュー情報を全て破棄します.
     */
    public final void destroy() {
        
        synchronized( SYNC ) {
            
            int len ;
            BaseQueueKey[] keys = BaseQueueFactory.getKeys() ;
            
            if( keys != null && ( len = keys.length ) > 0 ) {
                for( int i = 0 ; i < len ; i ++ ) {
                    try {
                        BaseQueueFactory.delete( keys[ i ] ) ;
                    } catch( Exception e ) {
                    }
                }
            }
            
        }
        
    }
    
    /**
     * 新しいキュー生成.
     * <BR><BR>
     * 新しいキューを生成します.
     * <BR>
     * @param bean 対象のQueueBeanを設定します.
     * @return BaseQueue 生成されたQueueオブジェクトが返されます.
     * @exception InputException 入力例外.
     */
    public static final BaseQueue create( BaseQueueBean bean ) throws InputException {
        
        String queueName = null ;
        String queueManager = null ;
        BaseQueue ret = null ;
        
        if(
            bean == null ||
            ( queueName = bean.getQueueName() ) == null ||
            queueName.length() <= 0 ||
            ( queueManager = bean.getQueueManagerName() ) == null ||
            queueManager.length() < 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            
            synchronized( SYNC ) {
                
                // キュー情報が既に存在する場合.
                if( BaseQueueFactory.get( bean.getType(),queueManager,queueName ) != null ) {
                    throw new InputException(
                        "既に対象条件のキュー(type:" +
                        bean.getType() +
                        " manager:" + queueManager +
                        " queue:" + queueName + ")は存在します" ) ;
                }
                
                // 新しくMQを生成.
                ret = BaseQueueFactory.create( bean.getType(),bean.isGzip(),bean.getQueueManagerName(),
                    bean.getQueueName(),bean.getOption(),bean.getWarning(),bean.getMaxMessageLength(),
                    bean.getMaxQueue(),bean.isAutoCommit(),bean.getCacheName() ) ;
                
                // オプションが存在する場合.
                if( bean.getOption() != null ) {
                    ret.setOption( bean.getOption() ) ;
                }
                
            }
            
        } catch( InputException in ) {
            throw in ;
        } catch( Exception e ) {
            throw new InputException( e ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 新しいキュー生成.
     * <BR><BR>
     * 新しいキューを生成します.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @param gzipFlag 圧縮条件[GZIP圧縮]を設定します.<BR>
     *                 [true]を設定した場合、メッセージ情報は圧縮されます.<BR>
     *                 [false]を設定した場合、メッセージ情報は圧縮されません.
     * @param parentManagerName 管理元のキューマネージャ名を設定します.
     * @param name 対象のキュー名を設定します.
     * @param option キューオプションを設定します.
     * @param warning 対象のキュー警告パーセントを設定します.
     * @param maxDataLength データの最大長を設定します.<BR>
     *                      [0]以下を設定した場合、制限を設けません.
     * @param maxQueue 対象のキュー最大格納数を設定します.
     * @param autoCommit オートコミットを設定します.<BR>
     *                   [true]が返された場合、オートコミットはONになります.<BR>
     *                   [false]が返された場合、オートコミットはOFFになります.
     * @param cacheName 対象のキャッシュ名を設定します.<BR>
     *                   [null]を設定した場合、無効になります.
     * @return BaseQueue 生成されたQueueオブジェクトが返されます.
     * @exception InputException 入力例外.
     */
    public static final BaseQueue create(
        int queueType,boolean gzipFlag,String parentManagerName,String name,Serializable option,
        double warning,int maxDataLength,int maxQueue,boolean autoCommit,String cacheName )
        throws InputException
    {
        
        String key = null ;
        ResourceType resourceType = null ;
        QueueManager man = null ;
        BaseQueueStateImple state = null ;
        BaseQueueImple queue = null ;
        
        if(
            name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ||
            parentManagerName == null ||
            ( parentManagerName = parentManagerName.trim().toLowerCase() ).length() <= 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        else if( ( man = QueueManagerFactory.get( parentManagerName ) ) == null ) {
            throw new InputException(
                "対象のキューマネージャ[" + parentManagerName +
                "]は存在しません" ) ;
        }
        else if( BaseQueueFactory.isQueue( queueType,parentManagerName,name ) == true ) {
            throw new InputException( "対象のキュー名[" + name + "]は既に存在します" ) ;
        }
        
        key = BaseQueueFactory.queueName( queueType,parentManagerName,name ) ;
        MqCache cache = null ;
        if( cacheName == null ||
            ( cacheName = cacheName.trim().toLowerCase() ).length() <= 0 ||
            ( cache = MqCacheFactory.get( cacheName ) ) == null ) {
            
            resourceType = ( ( BaseQueueManager )man ).getResourceType() ;
        }
        else {
            resourceType = cache.getResourceType() ;
        }
        
        maxQueue = ( maxQueue <= 0 ) ? MaachangQAccessDefine.DEFAULT_QUEUE_SIZE : maxQueue ;
        
        synchronized( SYNC ) {
            
            state = new BaseQueueStateImple() ;
            state.create(
                resourceType,queueType,parentManagerName,name,warning,
                maxDataLength,maxQueue,autoCommit,gzipFlag,cacheName ) ;
            queue = new BaseQueueImple( state,option ) ;
            
            NamingManager.add( key,queue ) ;
            BaseQueueFactory.addTypeMan( queue ) ;
            
            ( ( BaseQueueManager )man ).addQueue( queue ) 
            ;
        }
        
        return queue ;
        
    }
    
    /**
     * 指定名のキューオブジェクトを削除.
     * <BR><BR>
     * 指定名のキューオブジェクトを削除します.
     * <BR>
     * @param key 削除対象のキューKey情報を設定します.
     */
    public static final void delete( BaseQueueKey key ) {
        
        if( key == null ) {
            return ;
        }
        
        BaseQueueFactory.delete(
            true,key.getQueueType(),key.getParentManagerName(),
            key.getName() ) ;
        
    }
    
    /**
     * 指定名のキューオブジェクトを削除.
     * <BR><BR>
     * 指定名のキューオブジェクトを削除します.
     * <BR>
     * @param queueType 削除対象のキュータイプを設定します.
     * @param parentManagerName 管理元のキューマネージャ名を設定します.
     * @param name 削除対象のキュー名を設定します.
     */
    public static final void delete( int queueType,String parentManagerName,String name ) {
        
        BaseQueueFactory.delete(
            true,queueType,parentManagerName,name ) ;
        
    }
    
    /**
     * 指定名のキューオブジェクトを削除.
     * <BR><BR>
     * 指定名のキューオブジェクトを削除します.
     * <BR>
     * @param mode 削除時にキューマネージャの管理条件も削除するか設定します.<BR>
     *             [true]を設定した場合、キューマネージャに管理されているキュー情報も削除します.<BR>
     *             [false]を設定した場合、キューマネージャに管理されているキュー情報は削除しません.
     * @param key 削除対象のキューKey情報を設定します.
     */
    public static final void delete( boolean mode,BaseQueueKey key ) {
        
        if( key == null ) {
            return ;
        }
        
        BaseQueueFactory.delete(
            mode,key.getQueueType(),key.getParentManagerName(),
            key.getName() ) ;
        
    }
    
    /**
     * 指定名のキューオブジェクトを削除.
     * <BR><BR>
     * 指定名のキューオブジェクトを削除します.
     * <BR>
     * @param mode 削除時にキューマネージャの管理条件も削除するか設定します.<BR>
     *             [true]を設定した場合、キューマネージャに管理されているキュー情報も削除します.<BR>
     *             [false]を設定した場合、キューマネージャに管理されているキュー情報は削除しません.
     * @param queueType 削除対象のキュータイプを設定します.
     * @param parentManagerName 管理元のキューマネージャ名を設定します.
     * @param name 削除対象のキュー名を設定します.
     */
    public static final void delete( boolean mode,int queueType,String parentManagerName,String name ) {
        
        QueueManager man = null ;
        Object o = null ;
        
        if(
            name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ||
            parentManagerName == null ||
            ( parentManagerName = parentManagerName.trim().toLowerCase() ).length() <= 0 ||
            ( man = QueueManagerFactory.get( parentManagerName ) ) == null
        ) {
            return ;
        }
        
        name = BaseQueueFactory.queueName( queueType,parentManagerName,name ) ;
        
        try{
            synchronized( SYNC ) {
                
                o = NamingManager.get( name ) ;
                if( o != null && ( o instanceof BaseQueue ) == true ) {
                    ( ( BaseQueue )o ).getState().setState( QueueStatus.STATE_SHUTDOWN ) ;
                    NamingManager.remove( name ) ;
                    BaseQueueFactory.removeTypeMan( ( BaseQueue )o ) ;
                }
                
                if( mode == true && man != null ) {
                    try {
                        ( ( BaseQueueManager )man ).removeQueue( false,queueType,name ) ;
                    } catch( Exception e ) {
                    }
                }
                
            }
            
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * 指定名のキューオブジェクトを取得.
     * <BR><BR>
     * 指定名のキューオブジェクトを取得します.
     * <BR>
     * @param key 対象のキューKey情報を設定します.
     * @return BaseQueue 対象の基本キューオブジェクトが返されます.
     */
    public static final BaseQueue get( BaseQueueKey key ) {
        
        if( key == null ) {
            return null ;
        }
        
        return BaseQueueFactory.get( key.getQueueType(),key.getParentManagerName(),key.getName() ) ;
        
    }
    
    /**
     * 指定名のキューオブジェクトを取得.
     * <BR><BR>
     * 指定名のキューオブジェクトを取得します.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @param parentManagerName 管理元のキューマネージャ名を設定します.
     * @param name 対象のキュー名を設定します.
     * @return BaseQueue 対象の基本キューオブジェクトが返されます.
     */
    public static final BaseQueue get( int queueType,String parentManagerName,String name ) {
        
        Object o = null ;
        BaseQueue ret = null ;
        
        if(
            name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ||
            parentManagerName == null ||
            ( parentManagerName = parentManagerName.trim().toLowerCase() ).length() <= 0 ||
            QueueManagerFactory.get( parentManagerName ) == null
        ) {
            return null ;
        }
        
        name = BaseQueueFactory.queueName( queueType,parentManagerName,name ) ;
        
        try{
            
            synchronized( SYNC ) {
                
                o = NamingManager.get( name ) ;
                if( o != null && ( o instanceof BaseQueue ) == true ) {
                    ret = ( BaseQueue )o ;
                }
                
            }
            
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 指定条件のキューオブジェクトを取得.
     * <BR><BR>
     * 指定条件のキューオブジェクトを取得します.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @param no 対象の項番を設定します.
     * @return BaseQueue 対象の基本キューオブジェクトが返されます.
     */
    public static final BaseQueue get( int queueType,int no ) {
        
        BaseQueue ret = null ;
        
        try {
            
            synchronized( SYNC ) {
                
                ret = ( BaseQueue )(
                    (
                        ( ObjectArray )(
                            (
                                ( NumberTable )NamingManager.get( NAMING_QUEUE_TYPE_PLUS )
                            ).get( queueType )
                        )
                    )
                ).get( no ) ;
                
            }
            
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューKey情報群を取得.
     * <BR><BR>
     * キューKey情報群を取得します.
     * <BR>
     * @return BaseQueueKey[] キューKey情報群が返されます.
     */
    public static final BaseQueueKey[] getKeys() {
        
        int i ;
        int len ;
        
        String[] names = null ;
        ObjectArray ary = null ;
        BaseQueueKey[] ret = null ;
        Object o = null ;
        
        try{
            
            synchronized( SYNC ) {
                
                names = SearchPortion.searchString(
                    NamingManager.getNames(),
                    new StringBuffer().append( "*" ).append( NAMING_PLUS ).toString() ) ;
                
                if( names != null && ( len = names.length ) > 0 ) {
                    
                    ary = new ObjectArray() ;
                    
                    for( i = 0 ; i < len ; i ++ ) {
                        
                        o = NamingManager.get( names[ i ] ) ;
                        if( o != null && ( o instanceof Queue ) == true ) {
                            ary.add( new BaseQueueKey(
                                ( ( Queue )o ).getState().getQueueType(),
                                ( ( Queue )o ).getState().getParentManagerName(),
                                ( ( Queue )o ).getState().getName()
                            ) ) ;
                        }
                        
                        o = null ;
                        
                    }
                    
                    if( ( len = ary.size() ) > 0 ) {
                        ret = new BaseQueueKey[ len ] ;
                        System.arraycopy( ary.getObjects(),0,ret,0,len ) ;
                        ary.clear() ;
                    }
                }
                
            }
            
        } catch( Exception e ) {
            ret = null ;
        } finally {
            names = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定キュータイプのキュー数を取得.
     * <BR><BR>
     * 指定したキュータイプに対するキュー数を取得します.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @return int キュータイプに対するキュー数が返されます.
     */
    public static final int getQueueTypeByElementLength( int queueType ) {
        
        int ret ;
        
        NumberTable tbl = null ;
        ObjectArray ary = null ;
        
        synchronized( SYNC ) {
            
            if( ( tbl = ( NumberTable )NamingManager.get( NAMING_QUEUE_TYPE_PLUS ) ) == null ) {
                return 0 ;
            }
            
            if( ( ary = ( ObjectArray )tbl.get( queueType ) ) == null || ( ret = ary.size() ) <= 0 ) {
                return 0 ;
            }
            
        }
        
        return ret ;
        
    }
    
    
    /**
     * 対象情報長を取得.
     * <BR><BR>
     * 対象の情報長を取得します.
     * <BR>
     * @return int 対象の情報長が返されます.
     */
    public static final int size() {
        
        int ret = 0 ;
        
        synchronized( SYNC ) {
            
            ObjectArray ary = null ;
            
            NumberTable tbl = ( NumberTable )NamingManager.get(
                NAMING_QUEUE_TYPE_PLUS ) ;
            
            if( tbl != null ) {
                
                int[] types = tbl.getNumbers() ;
                int len ;
                if( types != null && ( len = types.length ) > 0 ) {
                    for( int i = 0,aryLength = 0 ; i < len ; i ++ ) {
                        
                        if( ( ary = ( ObjectArray )tbl.get( types[ i ] ) ) != null &&
                            ( aryLength = ary.size() ) > 0 ) {
                            ret += aryLength ;
                        }
                        
                    }
                    
                }
                
            }
        }
        
        return ret ;
    }
    
    /**
     * キュータイプ一覧を取得.
     * <BR><BR>
     * キュータイプ一覧を取得します.
     * <BR>
     * @return int[] キュータイプ一覧が返されます.<BR>
     *               キュータイプが存在しない場合[null]が返されます.
     */
    public static final int[] getQueueTypes() {
        
        NumberTable tbl = null ;
        int[] ret = null ;
        
        synchronized( SYNC ) {
            
            if( ( tbl = ( NumberTable )NamingManager.get( NAMING_QUEUE_TYPE_PLUS ) ) == null ) {
                return null ;
            }
            
            ret = tbl.getNumbers() ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 対象キューが存在するかチェック.
     * <BR><BR>
     * 対象のキューが存在するかチェックします.
     * <BR>
     * @param key 対象のキューKey情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象名のキューは存在します.<BR>
     *                 [false]が返された場合、対象名のキューは存在しません.
     */
    public static final boolean isQueue( BaseQueueKey key ) {
        
        if( key == null ) {
            return false ;
        }
        
        return BaseQueueFactory.isQueue( key.getQueueType(),key.getParentManagerName(),key.getName() ) ;
        
    }
    
    /**
     * 対象キューが存在するかチェック.
     * <BR><BR>
     * 対象のキューが存在するかチェックします.
     * <BR>
     * @param queueType 対象のキュータイプを設定します.
     * @param parentManagerName 管理元のキューマネージャ名を設定します.
     * @param name チェック対象のキュー名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、対象名のキューは存在します.<BR>
     *                 [false]が返された場合、対象名のキューは存在しません.
     */
    public static final boolean isQueue( int queueType,String parentManagerName,String name ) {
        
        boolean ret ;
        Object o = null ;
        
        if(
            name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ||
            parentManagerName == null ||
            ( parentManagerName = parentManagerName.trim().toLowerCase() ).length() <= 0 ||
            QueueManagerFactory.get( parentManagerName ) == null
        ) {
            return false ;
        }
        
        name = BaseQueueFactory.queueName( queueType,parentManagerName,name ) ;
        
        try{
            
            synchronized( SYNC ) {
                
                o = NamingManager.get( name ) ;
                if( o != null && ( o instanceof Queue ) == true ) {
                    ret = true ;
                }
                else {
                    ret = false ;
                }
                
            }
            
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * キュー管理名を設定.
     */
    private static final String queueName( int queueType,String parentManagerName,String name ) {
        
        String ret = new StringBuffer().
            append( name ).
            append( "/" ).
            append( parentManagerName ).
            append( "[" ).
            append( queueType ).
            append( "]" ).
            append( NAMING_PLUS ).toString() ;
        
        return ret ;
    }
    
    /**
     * タイプ管理にキュー情報を追加.
     */
    private static final void addTypeMan( BaseQueue queue ) {
        
        int queueType ;
        
        NumberTable tbl = null ;
        ObjectArray ary = null ;
        
        if( queue == null ) {
            return ;
        }
        
        queueType = queue.getState().getQueueType() ;
        
        try {
            
            if( ( tbl = ( NumberTable )NamingManager.get( NAMING_QUEUE_TYPE_PLUS ) ) == null ) {
                tbl = new NumberTable() ;
                NamingManager.add( NAMING_QUEUE_TYPE_PLUS,tbl ) ;
            }
            
            if( ( ary = ( ObjectArray )tbl.get( queueType ) ) == null ) {
                ary = new ObjectArray() ;
                tbl.add( queueType,ary ) ;
            }
            
            ary.add( queue ) ;
            
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * タイプ管理にキュー情報から、指定条件の内容を削除.
     */
    private static final void removeTypeMan( BaseQueue queue ) {
        
        int i ;
        int len ;
        int queueType ;
        
        NumberTable tbl = null ;
        ObjectArray ary = null ;
        
        if( queue == null || queue.isQueue() == false ) {
            return ;
        }
        
        queueType = queue.getState().getQueueType() ;
        
        if( ( tbl = ( NumberTable )NamingManager.get( NAMING_QUEUE_TYPE_PLUS ) ) == null ) {
            return ;
        }
        
        if(
            ( ary = ( ObjectArray )tbl.get( queueType ) ) == null ||
            ( len = ary.size() ) <= 0
        ) {
            return ;
        }
        
        for( i = 0 ; i < len ; i ++ ) {
            if( ary.get( i ) == queue ) {
                ary.remove( i ) ;
                if( ary.size() == 0 ) {
                    tbl.remove( queueType ) ;
                }
                break ;
            }
        }
        
    }
    
}

