/*
 * @(#)MqPool.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.service ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.config.service.ConfigService;

/**
 * MQ構成データ更新スレッド.
 *  
 * @version 2006/12/27
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class SaveThread extends ExecutionThread {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( SaveThread.class ) ;
    
    /**
     * 実行間隔 : デフォルトタイムアウト値.
     * 5分.
     */
    protected static final long DEF_TIME = 300000L ;
    
    /**
     * タイムアウト値 : 最小値.
     * 1分.
     */
    protected static final long MIN_TIME = 60000L ;
    
    /**
     * タイムアウト値 : 最大値.
     * 30分.
     */
    protected static final long MAX_TIME = 1800000L ;
    
    
    
    /**
     * 処理間隔.
     */
    private long time = -1L ;
    
    /**
     * 前回処理時間.
     */
    private long beforeTime = -1L ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public SaveThread(){
        this( ConfigService.getMqConfig().getAutoSaveTiming() ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * データセーブ間隔を設定して処理を生成します.
     * <BR>
     * @param time データセーブ間隔値を設定します.
     */
    public SaveThread( long time ){
        
        sync.create() ;
        
        if( time == -1L ) {
            time = DEF_TIME ;
        }
        
        time = ( time <= MIN_TIME ) ? MIN_TIME : time ;
        time = ( time >= MAX_TIME ) ? MAX_TIME : time ;
        
        try{
            
            this.time = time ;
            this.beforeTime = System.currentTimeMillis() ;
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        sync.clear() ;
        thread.clear() ;
    }
    
    /**
     * 処理間隔を取得.
     * <BR><BR>
     * 処理間隔が返されます.
     * <BR>
     * @return long 処理間隔が返されます.
     */
    public final long getTime() {
        long ret ;
        
        try {
            synchronized( sync.get() ) {
                ret = time ;
            }
        } catch( Exception e ) {
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 処理更新時間を設定.
     * <BR><BR>
     * 処理更新時間を設定します.
     * <BR>
     * @param time 新しい処理更新時間を設定します.
     */
    protected final void setUpdateTime( long time ) {
        this.beforeTime = time ;
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Synchronized 同期オブジェクトが返されます.
     */
    protected final Synchronized getSync() {
        return sync ;
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException
    {
        long execTime ;
        
        try{
            
            UtilCom.idleTime() ;
            
            // 処理タイミングを取得.
            synchronized( sync.get() ) {
                execTime = this.time + this.beforeTime ;
            }
            
            // スレッドオートセーブ規定値の場合.
            if( execTime <= System.currentTimeMillis() ) {
                
                // １度、セーブ＆ロード同期を取得.
                synchronized( SaveAndLoad.getSync() ) {
                    
                    // 再度処理タイミングを取得.
                    synchronized( sync.get() ) {
                        execTime = this.time + this.beforeTime ;
                    }
                    
                    // オートセーブを行う場合.
                    if( execTime <= System.currentTimeMillis() ) {
                        LOG.info( "----- 自動保存開始 -----" )  ;
                        SaveAndLoad.save( this ) ;
                        LOG.info( "----- 自動保存終了 -----" ) ;
                    }
                }
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( ExecutionException ee ) {
            throw ee ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
        }
        
    }
    
}

