/*
 * @(#)ExecutionCache.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CacheInfo;
import org.maachang.queue.access.admin.info.CreateCacheInfo;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.protocol.ResultTelegram;
import org.maachang.queue.access.protocol.admin.AdminCacheBean;
import org.maachang.queue.access.protocol.admin.AdminCacheProtocol;


/**
 * 管理キャッシュ実行処理.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ExecutionCache {
    
    /**
     * デフォルトタイムアウト.
     */
    public static final long DEFAULT_TIMEOUT = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
    
    /**
     * キャッシュ追加処理.
     * <BR><BR>
     * キャッシュ追加処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param info 対象のキャッシュ情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void addCache(
        ConnectObject connect,int userId,boolean userOwner,CreateCacheInfo info )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = CacheTelegram.createAddCache(
                outId,userOwner,userId,info ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminCacheProtocol.TYPE_ADD_CACHE,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キャッシュ削除処理.
     * <BR><BR>
     * キャッシュ削除処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のキャッシュ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void removeCache(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = CacheTelegram.createRemoveCache(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminCacheProtocol.TYPE_REMOVE_CACHE,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キャッシュ取得処理.
     * <BR><BR>
     * キャッシュ取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のキャッシュ名を設定します.
     * @return CacheInfo 対象のキャッシュ構成情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final CacheInfo getCache(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = CacheTelegram.createGetCache(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminCacheProtocol.TYPE_GET_CACHE,
                    telegram ) ;
                
            }
            // 受信電文が、キャッシュ情報の場合.
            else if( result == AdminCacheProtocol.CATEGORY_TYPE_ADMIN_CACHE ) {
                
                // キャッシュ情報を解析.
                AdminCacheBean bean = new AdminCacheBean() ;
                CacheTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                if( bean.getName() == null || bean.getName().length() <= 0 ) {
                    throw new MaachangQException(
                        "対象名(" + name + ")のキャッシュ名は存在しません",
                        MaachangQErrorCode.ERROR_NOT_CACHE ) ;
                }
                
                CacheInfo ret = new CacheInfo() ;
                ret.setName( bean.getName() ) ;
                ret.setSize( bean.getMaxSize() ) ;
                ret.setUseSize( bean.getUseSize() ) ;
                
                if( bean.getMaxSize() <= 0 || bean.getUseSize() <= 0 ) {
                    ret.setUsePersent( 0 ) ;
                }
                else {
                    ret.setUsePersent(
                        ( int )(
                            ( ( double )bean.getUseSize() /
                                ( double )bean.getMaxSize()
                        ) * 100.0f ) ) ;
                }
                
                return ret ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キャッシュ名群取得処理.
     * <BR><BR>
     * キャッシュ名群取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return String[] 対象のキャッシュ名群が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final String[] getNames(
        ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = CacheTelegram.createGetNamesCache(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminCacheProtocol.TYPE_GET_CACHE_NAMES,
                    telegram ) ;
                
            }
            // 受信電文が、キャッシュ情報の場合.
            else if( result == AdminCacheProtocol.CATEGORY_TYPE_ADMIN_CACHE ) {
                
                // キャッシュ情報を解析.
                AdminCacheBean bean = new AdminCacheBean() ;
                CacheTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                return bean.getParams() ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キャッシュ数取得処理.
     * <BR><BR>
     * キャッシュ数取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return int 対象のキャッシュ数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final int getSize(
        ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = CacheTelegram.createGetSizeCache(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminCacheProtocol.TYPE_GET_CACHE_SIZE,
                    telegram ) ;
                
            }
            // 受信電文が、キャッシュ情報の場合.
            else if( result == AdminCacheProtocol.CATEGORY_TYPE_ADMIN_CACHE ) {
                
                // キャッシュ情報を解析.
                AdminCacheBean bean = new AdminCacheBean() ;
                CacheTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                return bean.getSize() ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キャッシュ存在チェック処理.
     * <BR><BR>
     * キャッシュ存在チェック処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のキャッシュ名を設定します.
     * @return boolean 対象キャッシュ存在結果が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean isCache(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = CacheTelegram.createUseCache(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                try {
                    // 処理戻り条件をチェック.
                    ResultTelegram.checkResult(
                        outId[ 0 ],AdminCacheProtocol.TYPE_IS_CACHE,
                        telegram ) ;
                    
                    return true ;
                } catch( Exception e ) {
                    return false ;
                }
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
}

