/*
 * @(#)ExecutionConnect.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.ConnectInfo;
import org.maachang.queue.access.admin.info.CreateConnectInfo;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.protocol.ResultTelegram;
import org.maachang.queue.access.protocol.admin.AdminConnectBean;
import org.maachang.queue.access.protocol.admin.AdminConnectProtocol;


/**
 * 管理コネクション実行処理.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ExecutionConnect {
    
    /**
     * デフォルトタイムアウト.
     */
    public static final long DEFAULT_TIMEOUT = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
    
    /**
     * コネクション追加処理.
     * <BR><BR>
     * コネクション追加処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param info 対象のコネクション情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void addConnect(
        ConnectObject connect,int userId,boolean userOwner,CreateConnectInfo info )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ConnectTelegram.createAddConnect(
                outId,userOwner,userId,info ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminConnectProtocol.TYPE_ADD_CONNECT,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * コネクション再設定処理.
     * <BR><BR>
     * コネクション再設定処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param info 対象のコネクション情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void setConnect(
        ConnectObject connect,int userId,boolean userOwner,ConnectInfo info )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ConnectTelegram.createSetConnect(
                outId,userOwner,userId,info ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminConnectProtocol.TYPE_SET_CONNECT,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * コネクション削除処理.
     * <BR><BR>
     * コネクション削除処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のコネクション名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void removeConnect(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ConnectTelegram.createRemoveConnect(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminConnectProtocol.TYPE_REMOVE_CONNECT,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * コネクション削除処理.
     * <BR><BR>
     * コネクション削除処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のコネクション名を設定します.
     * @return ConnectInfo コネクション構成情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final ConnectInfo getConnect(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ConnectTelegram.createGetConnect(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminConnectProtocol.TYPE_GET_CONNECT,
                    telegram ) ;
                
            }
            // 受信電文が、コネクション情報の場合.
            else if( result == AdminConnectProtocol.CATEGORY_TYPE_ADMIN_CONNECT ) {
                
                // コネクション情報を解析.
                AdminConnectBean bean = new AdminConnectBean() ;
                ConnectTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                if( bean.getName() == null || bean.getName().length() <= 0 ) {
                    throw new MaachangQException(
                        "対象名(" + name + ")のコネクション名は存在しません",
                        MaachangQErrorCode.ERROR_NOT_CONNECT ) ;
                }
                
                ConnectInfo ret = new ConnectInfo() ;
                ret.setName( bean.getName() ) ;
                ret.setMqCacheName( bean.getCacheName() ) ;
                ret.setBuffer( bean.getBuffer() ) ;
                ret.setBindAddress( bean.getInetAddress() ) ;
                ret.setBindPort( bean.getPort() ) ;
                ret.setCb32( bean.isCb32() ) ;
                ret.setCb32Word( bean.getParams() ) ;
                
                return ret ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * コネクション名群取得処理.
     * <BR><BR>
     * コネクション名群取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return String[] コネクション名群が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final String[] getNames(
        ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ConnectTelegram.createGetNamesConnect(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminConnectProtocol.TYPE_GET_CONNECT_NAMES,
                    telegram ) ;
                
            }
            // 受信電文が、コネクション情報の場合.
            else if( result == AdminConnectProtocol.CATEGORY_TYPE_ADMIN_CONNECT ) {
                
                // コネクション情報を解析.
                AdminConnectBean bean = new AdminConnectBean() ;
                ConnectTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                return bean.getParams() ;
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * コネクション数取得処理.
     * <BR><BR>
     * コネクション数取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return int コネクション数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final int getSize(
        ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ConnectTelegram.createGetSizeConnect(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminConnectProtocol.TYPE_GET_CONNECT_SIZE,
                    telegram ) ;
                
            }
            // 受信電文が、コネクション情報の場合.
            else if( result == AdminConnectProtocol.CATEGORY_TYPE_ADMIN_CONNECT ) {
                
                // コネクション情報を解析.
                AdminConnectBean bean = new AdminConnectBean() ;
                ConnectTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                return bean.getSize() ;
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * コネクション名存在チェック処理.
     * <BR><BR>
     * コネクション名存在チェック処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のコネクション名を設定します.
     * @return boolean コネクション名存在結果が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean isConnect(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = ConnectTelegram.createUseConnect(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                try {
                    // 処理戻り条件をチェック.
                    ResultTelegram.checkResult(
                        outId[ 0 ],AdminConnectProtocol.TYPE_IS_CONNECT,
                        telegram ) ;
                    
                    return true ;
                } catch( Exception e ) {
                    return false ;
                }
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    
}

