/*
 * @(#)BaseProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 基本プロトコル.
 *  
 * @version 2007/01/07
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class BaseProtocol {
    
    /**
     * 電文基本情報設定.
     * <BR><BR>
     * 電文基本情報を設定します.
     * <BR>
     * @param addPoint 対象電文を追加するためのポイントを格納されます.
     * @param binaryLength バイナリ長を設定します.
     * @param id プロトコルIDを設定します.
     * @param type 処理タイプを設定します.
     * @param queueManagerName 対象のキューマネージャ名を設定します.
     * @param queueName 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createCommonTelegram(
        int[] addPoint,int binaryLength,int type,String queueManagerName,String queueName,int queueType )
        throws Exception {
        
        // 引数が不正.
        if( addPoint == null || addPoint.length <= 0 || binaryLength <= 0 ||
            ( queueManagerName = queueManagerName.trim().toLowerCase() ).length() <= 0 ||
            ( queueName = queueName.trim().toLowerCase() ).length() <= 0 ||
            ( queueType != MaachangQAccessDefine.TYPE_RECEIVE && queueType != MaachangQAccessDefine.TYPE_SEND ) ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // キューマネージャ名をバイナリ変換.
        byte[] managerBin = queueManagerName.getBytes( CommonProtocol.CHARSET ) ;
        int managerLen = managerBin.length ;
        
        // キュー名をバイナリ変換.
        byte[] queueBin = queueName.getBytes( CommonProtocol.CHARSET ) ;
        int queueLen = queueBin.length ;
        
        // 全体電文情報長を計算.
        int allBinaryLength = 
            4 +                 // 全体電文長.
            4 +                 // 処理タイプ.
            4 +                 // キューマネージャ名長.
            managerLen +        // キューマネージャ名.
            4 +                 // キュー名長.
            queueLen +          // キュー名.
            4 +                 // キュータイプ.
            binaryLength +      // 指定電文長.
            4 ;                 // チェックコード.
        
        // 電文情報を生成.
        int pnt = 0 ;
        byte[] telegram = new byte[ allBinaryLength ] ;
        
        // 全体電文長.
        ConvertBinary.convertInt( telegram,pnt,allBinaryLength ) ;
        pnt += 4 ;
        
        // 処理タイプ
        ConvertBinary.convertInt( telegram,pnt,type ) ;
        pnt += 4 ;
        
        // キューマネージャ名長.
        ConvertBinary.convertInt( telegram,pnt,managerLen ) ;
        pnt += 4 ;
        
        // キューマネージャ名.
        System.arraycopy( managerBin,0,telegram,pnt,managerLen ) ;
        pnt += managerLen ;
        managerBin = null ;
        
        // キュー名長.
        ConvertBinary.convertInt( telegram,pnt,queueLen ) ;
        pnt += 4 ;
        
        // キュー名.
        System.arraycopy( queueBin,0,telegram,pnt,queueLen ) ;
        pnt += queueLen ;
        queueBin = null ;
        
        // キュータイプ.
        ConvertBinary.convertInt( telegram,pnt,queueType ) ;
        pnt += 4 ;
        
        // この情報の間に、データを設定する必要がある.
        addPoint[ 0 ] = pnt ;
        
        return telegram ;
        
    }
    
    /**
     * 電文基本情報を取得.
     * <BR><BR>
     * 電文基本情報を取得します.
     * <BR>
     * @param readPoint 対象電文読み込み開始位置が格納されます.
     * @param out 電文基本情報を格納されます.
     * @param telegram 対象の電文情報を設定します.
     * @exception Exception 例外.
     */
    public static final void analysisCommonTelegram(
        int[] readPoint,BaseProtocolBean out,byte[] telegram )
        throws Exception {
        
        if( readPoint == null || readPoint.length <= 0 ||
            out == null || telegram == null || telegram.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // 電文長をチェック.
        CommonProtocol.checkTelegramLength( telegram ) ;
        int pnt = 4 ;
        
        // チェックコードを判別.
        CommonProtocol.checkCheckCode( telegram ) ;
        
        // 処理タイプを取得.
        int type = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キューマネージャ名長を取得.
        int managerLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キューマネージャ名を取得.
        byte[] managerBin = new byte[ managerLen ] ;
        System.arraycopy( telegram,pnt,managerBin,0,managerLen ) ;
        String manager = new String( managerBin,CommonProtocol.CHARSET ) ;
        managerBin = null ;
        pnt += managerLen ;
        
        // キュー名長を取得.
        int queueLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キュー名を取得.
        byte[] queueBin = new byte[ queueLen ] ;
        System.arraycopy( telegram,pnt,queueBin,0,queueLen ) ;
        String queue = new String( queueBin,CommonProtocol.CHARSET ) ;
        queueBin = null ;
        pnt += queueLen ;
        
        // キュータイプを取得.
        int queueType = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // ここから読み込み開始.
        readPoint[ 0 ] = pnt ;
        
        // Beanに設定.
        out.setType( type ) ;
        out.setQueueManager( manager ) ;
        out.setQueue( queue ) ;
        out.setQueueType( queueType ) ;
        
    }
    
}

