/*
 * @(#)ResultProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol ;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 処理結果用プロトコル.
 * <BR><BR>
 * 処理結果用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/08
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class ResultProtocol {
    
    /**
     * 処理結果用ヘッダ.
     */
    public static final byte[] HEADER_RESULT = {
        ( byte )0x00000F5,( byte )0x00000017 } ;
    
    /**
     * 処理カテゴリ : 処理結果系.
     */
    public static final int CATEGORY_TYPE_RESULT = 0x20000000 ;
    
    /**
     * 処理タイプ : 正常処理結果.
     */
    public static final int TYPE_SUCCESS = 0x00000001 | CATEGORY_TYPE_RESULT ;
    
    /**
     * 処理タイプ : 異常処理結果.
     */
    public static final int TYPE_ERROR = 0x00000002 | CATEGORY_TYPE_RESULT ;
    
    
    /**
     * 処理タイプチェック.
     * <BR><BR>
     * 処理タイプをチェックします.
     * <BR>
     * @param type 処理タイプを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                  [true]が返された場合、処理タイプは存在します.<BR>
     *                  [false]が返された場合、処理タイプは存在しません.
     */
    public static final boolean checkType( int type ) {
        switch( type ) {
            case TYPE_SUCCESS :
            case TYPE_ERROR :
            return true ;
        }
        return false ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param type 処理タイプを設定します.
     * @param bean 対象のメッセージBeanを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int type,ResultBean bean )
        throws Exception {
        
        if( bean == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        return ResultProtocol.createProtocol(
            id,type,bean.getQueueManager(),bean.getQueue(),bean.getQueueType(),
            bean.getSrcType(),bean.getResultCode(),
            bean.getErrorMessage() ) ;
        
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param type 処理タイプを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param srcType 処理結果の元を示す処理タイプを設定します.
     * @param resultCode 対象のエラーコードを設定します.
     * @param errorMessage 対象のエラーメッセージを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int type,String manager,
        String queue,int queueType,int srcType,int resultCode,String errorMessage )
        throws Exception {
        
        // 処理タイプが不正な場合.
        if( ResultProtocol.checkType( type ) == false ) {
            throw new MaachangQException(
                "処理タイプ(" + type + ")が不正です",
                MaachangQErrorCode.ERROR_NOT_TYPE ) ;
        }
        
        int errorMsgLen = 0 ;
        byte[] errorMsgBin = null ;
        if( errorMessage != null && ( errorMessage = errorMessage.trim() ).length() >= 0 ) {
            errorMsgBin = errorMessage.getBytes( CommonProtocol.CHARSET ) ;
            errorMsgLen = errorMsgBin.length ;
        }
        
        // 電文情報長を計算.
        int binaryLength = 
            4 +                 // 処理元のタイプ.
            4 +                 // エラーコード.
            4 +                 // エラーメッセージ長.
            errorMsgLen ;       // エラーメッセージ.
        
        // 電文情報を生成.
        int[] nextPnt = new int[ 1 ] ;
        byte[] telegram = BaseProtocol.createCommonTelegram(
            nextPnt,binaryLength,type,manager,queue,queueType ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // 処理元タイプ.
        ConvertBinary.convertInt( telegram,pnt,srcType ) ;
        pnt += 4 ;
        
        // 処理結果コード.
        ConvertBinary.convertInt( telegram,pnt,resultCode ) ;
        pnt += 4 ;
        
        // エラーメッセージ長.
        ConvertBinary.convertInt( telegram,pnt,errorMsgLen ) ;
        pnt += 4 ;
        
        // エラーメッセージ
        if( errorMsgLen > 0 ) {
            System.arraycopy( errorMsgBin,0,telegram,pnt,errorMsgLen ) ;
            pnt += errorMsgLen ;
            errorMsgBin = null ;
        }
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_RESULT,id,telegram ) ;
        
    }
    
    /**
     * プロトコルを解析.
     * <BR><BR>
     * プロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( ResultBean out,byte[] telegram )
        throws Exception {
        
        // 電文情報を解析.
        int[] nextPnt = new int[ 1 ] ;
        BaseProtocol.analysisCommonTelegram( nextPnt,out,telegram ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // 処理元タイプを取得.
        int srcType = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 処理結果コードを取得.
        int resultCode = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // エラーメッセージ長を取得.
        int errorMsgLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // エラーメッセージを取得.
        String errorMessage = null ;
        if( errorMsgLen > 0 ) {
            byte[] errorMsgBin = new byte[ errorMsgLen ] ;
            System.arraycopy( telegram,pnt,errorMsgBin,0,errorMsgLen ) ;
            errorMessage = new String( errorMsgBin,CommonProtocol.CHARSET ) ;
            errorMsgBin = null ;
        }
        
        // Beanに設定.
        out.setSrcType( srcType ) ;
        out.setResultCode( resultCode ) ;
        out.setErrorMessage( errorMessage ) ;
        
    }
    
}

