/*
 * @(#)ByteUtil.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.util;

import java.util.Arrays;

/**
 * バイトユーティリティ.
 *
 * @version 2007/01/08
 * @author  Masahito Suzuki
 * @since   MaachangQ-Access 1.00
 */
public class ByteUtil {
    
    /**
     * 配列縮小係数.
     */
    public static final double REDUCTION_ARRAY = 0.375 ;
    
    /**
     * 開始時配列管理数 : デフォルト.
     */
    public static final int DEFAULT_START_LENGTH = 16 ;
    
    /**
     * 開始時配列管理数 : 最小値.
     */
    public static final int MIN_START_LENGTH = 2 ;
    
    /**
     * 開始時配列管理数 : 最大値.
     */
    public static final int MAX_START_LENGTH = 9999 ;
    
    /**
     * デフォルトデータ値.
     */
    public static final int DEFAULT_DATA = 0 ;
    
    /**
     * 配列管理.
     */
    private byte[] array = null ;
    
    /**
     * 開始時管理配列数.
     */
    private int startLength = DEFAULT_START_LENGTH ;
    
    /**
     * 配列管理数.
     */
    private int length = DEFAULT_START_LENGTH ;
    
    /**
     * 現在格納配列数.
     */
    private int nowSize = 0 ;
    
    /**
     * コンストラクタ.
     */
    public ByteUtil() {
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象のバッファ係数を設定します.
     * <BR>
     * @param length 対象のバッファ係数を設定します.<BR>
     *               設定可能な最大値は[9999]です.<BR>
     *               設定可能な最小値は[2]です.<BR>
     *               また、これらの設定範囲外を設定した場合、
     *               デフォルトの値[16]となります.
     */
    public ByteUtil( int length ) {
        
        this.startLength = 
            (
                length < MIN_START_LENGTH ||
                length > MAX_START_LENGTH
            ) ? DEFAULT_START_LENGTH :length ;
        this.length = this.startLength ;
    }
    
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        this.clear() ;
    }
    
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 対象の情報をクリアします.
     */
    public final void clear() {
        
        this.array = null ;
        this.length = this.startLength ;
        this.nowSize = 0 ;
        
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[Byte]情報を追加します.
     * @exception Exception 例外.
     */
    public final void add( Byte value )
        throws Exception {
        
        if( value == null ){
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        this.add( value.byteValue() ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[byte(8bit)]情報を追加します.
     */
    public final void add( byte value ) {
        int length ;
        int nowSize ;
        
        byte[] info = null ;
        byte[] tmp = null ;
        
        info = this.array ;
        length = this.length ;
        nowSize = this.nowSize ;
        
        if( info == null ) {
            info = new byte[ length ] ;
            info[ nowSize ] = value ;
            
            this.array = info ;
        }
        else if( info.length <= nowSize ) {
            length *= 2 ;
            tmp = info ;
            info = new byte[ length ] ;
            
            System.arraycopy( tmp,0,info,0,nowSize ) ;
            
            info[ nowSize ] = value ;
            
            this.length = length ;
            this.array = info ;
        }
        else {
            info[ nowSize ] = value ;
        }
        
        this.nowSize ++ ;
        
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[byte(8bit)]配列情報を追加します.
     * @exception Exception 例外.
     */
    public final void add( byte[] value )
        throws Exception {
        this.add( 0,-1,value ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param off 追加対象情報のオフセット値を設定します.
     * @param value 設定対象の数値[byte(8bit)]配列情報を追加します.
     * @exception Exception 例外.
     */
    public final void add( int off,byte[] value )
        throws Exception {
        this.add( off,-1,value ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param value 設定対象の数値[byte(8bit)]配列情報を追加します.
     * @param length 追加対象情報のデータ長を設定します.
     * @exception Exception 例外.
     */
    public final void add( byte[] value,int length )
        throws Exception {
        this.add( 0,length,value ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param off 追加対象情報のオフセット値を設定します.
     * @param length 追加対象情報のデータ長を設定します.
     * @param value 設定対象の数値[byte(8bit)]配列情報を追加します.
     * @exception Exception 例外.
     */
    public final void add( int off,int length,byte[] value )
        throws Exception {
        
        int arrayLen ;
        int createLen ;
        int valueLen ;
        int nowSize ;
        
        byte[] info = null ;
        byte[] tmp = null ;
        
        if( value == null || ( valueLen = value.length ) <= 0 ){
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }else if( off < 0 || off >= valueLen ){
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        length = ( length <= 0 || ( length + off ) > valueLen ) ?
            valueLen - off : length ;
        
        info = this.array ;
        arrayLen = this.length ;
        nowSize = this.nowSize ;
        
        try{
            
            System.arraycopy( info,nowSize,value,off,length ) ;
            
        }catch( NullPointerException nul ){
            
            for( ; arrayLen <= length ; ){
                arrayLen *= 2 ;
            }
            
            info = new byte[ arrayLen ] ;
            System.arraycopy( info,nowSize,value,off,length ) ;
            
            this.length = arrayLen ;
            this.array = info ;
            
        }catch( IndexOutOfBoundsException io ){
            
            createLen = length + nowSize ;
            
            for( ; arrayLen <= createLen ; ){
                arrayLen *= 2 ;
            }
            
            tmp = info ;
            info = new byte[ arrayLen ] ;
            
            System.arraycopy( tmp,0,info,0,nowSize ) ;
            System.arraycopy( info,nowSize,value,off,length ) ;
            
            this.length = arrayLen ;
            this.array = info ;
            
        }finally{
            
            tmp = null ;
            info = null ;
            
        }
        
        this.nowSize += length ;
        
    }
    
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象数値[Byte]情報を設定します.
     * @exception Exception 例外.
     */
    public final void set( int no,Byte value )
        throws Exception {
        if( value == null ){
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        this.set( no,value.byteValue() ) ;
    }
    
    /**
     * 情報設定.
     * <BR><BR>
     * 対象の位置に情報をセットします.
     * <BR>
     * @param no 設定対象項番を設定します.
     * @param value 設定対象数値[byte(8bit)]情報を設定します.
     * @exception Exception 例外.
     */
    public final void set( int no,byte value )
        throws Exception {
        int nowLen ;
        
        nowLen = this.nowSize ;
        
        if( no < 0 || no >= nowLen ){
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        this.array[ no ] = value ;
        
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除します.
     * <BR>
     * @param no 削除対象の項番を設定します.
     * @return byte 削除された数値[byte(8bit)]情報が返されます.<BR>
     *              削除情報が存在しない場合等は、[Byte.MAX_VALUE]が返されます.
     */
    public final byte remove( int no ) {
        int nowSize ;
        int length ;
        int newLength ;
        
        byte[] info = null ;
        byte[] tmp = null ;
        
        byte ret = Byte.MAX_VALUE ;
        
        nowSize = this.nowSize ;
        length = this.length ;
        
        if( no < 0 || no >= nowSize || nowSize == 0 ){
            return Byte.MAX_VALUE ;
        }
        
        info = this.array ;
        
        ret = info[ no ] ;
        info[ no ] = ( byte )DEFAULT_DATA ;
        
        if( no == 0 ){
            
            tmp = info ;
            System.arraycopy( tmp,1,info,0,( nowSize-1 ) ) ;
            
            info[ nowSize - 1 ] = ( byte )DEFAULT_DATA ;
            
        }else if( ( nowSize - no ) != 1 ){
            
            tmp = info ;
            System.arraycopy( tmp,0,info,0,no ) ;
            System.arraycopy( tmp,no+1,info,no,nowSize-(no+1) ) ;
            
            info[ nowSize - 1 ] = ( byte )DEFAULT_DATA ;
            
        }
        
        nowSize -- ;
        
        if(
            nowSize != 0 &&
            ( length * REDUCTION_ARRAY ) >= nowSize
        )
        {
            
            newLength = length / 2 ;
            tmp = new byte[ newLength ] ;
            System.arraycopy( info,0,tmp,0,newLength ) ;
            
            info = null ;
            info = tmp ;
            
            this.length = newLength ;
            
        }else if( nowSize == 0 ){
            
            info = null ;
            
        }
        
        this.array = info ;
        this.nowSize = nowSize ;
        
        info = null ;
        tmp = null ;
        
        return ret ;
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除します.
     * <BR>
     * @param no 削除対象の項番を設定します.
     * @param length 削除対象の長さを設定します.
     * @return byte[] 削除された数値[byte(8bit)]配列情報が返されます.<BR>
     *                削除情報が存在しない場合等は、[null]が返されます.
     */
    public final byte[] remove( int no,int length ) {
        int nowSize ;
        int arrayLen ;
        int createLen ;
        
        byte[] info = null ;
        byte[] tmp = null ;
        
        byte[] ret = null ;
        
        nowSize = this.nowSize ;
        arrayLen = this.length ;
        
        if( no < 0 || no >= nowSize || nowSize == 0 ){
            return null ;
        }
        
        length = ( length <= 0 || ( length + no ) <= nowSize ) ?
            nowSize - no : length ;
        
        info = this.array ;
        
        ret = new byte[ length ] ;
        System.arraycopy( info,no,ret,0,length ) ;
        Arrays.fill( info,nowSize - length,nowSize,( byte )DEFAULT_DATA ) ;
        
        if( no == 0 ){
            
            tmp = info ;
            System.arraycopy( tmp,no,info,0,( nowSize - length ) ) ;
            
        }else if( ( nowSize - no ) != length ){
            
            tmp = info ;
            System.arraycopy( tmp,0,info,0,no ) ;
            System.arraycopy( tmp,no+length,info,no,( nowSize - (no+length ) ) ) ;
            
        }
        
        nowSize -= length ;
        
        if(
            nowSize != 0 &&
            ( arrayLen * REDUCTION_ARRAY ) >= nowSize
        )
        {
            createLen = ( int )( arrayLen * REDUCTION_ARRAY ) ;
            
            for( ; arrayLen <= createLen ; ){
                arrayLen /= 2 ;
            }
            
            arrayLen *= 2 ;
            
            tmp = new byte[ arrayLen ] ;
            System.arraycopy( info,0,tmp,0,arrayLen ) ;
            
            info = null ;
            info = tmp ;
            
            this.length = arrayLen ;
            
        }else if( nowSize == 0 ){
            
            info = null ;
            
        }
        
        this.array = info ;
        this.nowSize = nowSize ;
        
        info = null ;
        tmp = null ;
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return byte 取得された数値[byte(8bit)]情報が返されます.
     *             削除情報が存在しない場合等は、[Byte.MAX_VALUE]が返されます.
     * @exception ArrayIndexOutOfBoundsException 配列例外.
     */
    public final byte get( int no ) throws ArrayIndexOutOfBoundsException {
        
        if( no < 0 || no >= this.nowSize ){
            throw new ArrayIndexOutOfBoundsException(
                "指定位置[" + no + "]は範囲外(" + this.nowSize + ")です" ) ;
        }
        
        return this.array[ no ] ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @param length 取得対象の長さを設定します.
     * @return byte[] 取得された数値[byte(8bit)]配列情報が返されます.
     *                削除情報が存在しない場合等は、[null]が返されます.
     */
    public final byte[] get( int no,int length ) {
        int nowSize ;
        byte[] ret = null ;
        
        nowSize = this.nowSize ;
        
        if( no < 0 || no >= nowSize ){
            return null ;
        }
        
        length = ( length <= 0 || ( length + no ) <= nowSize ) ?
            nowSize - no : length ;
        
        ret = new byte[ length ] ;
        System.arraycopy( this.array,no,ret,0,length ) ;
        
        return ret ;
        
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param out 取得されたデータが格納されます.
     * @param no 取得対象の項番を設定します.
     * @return int 取得された長さが返されます.
     */
    public final int get( byte[] out,int no ) {
        
        int ret ;
        int nowSize ;
        
        nowSize = this.nowSize ;
        
        if( out == null || no < 0 || no >= nowSize ){
            return 0 ;
        }
        
        ret = out.length ;
        ret = ( ret <= 0 || ( ret + no ) <= nowSize ) ?
            nowSize - no : ret ;
        
        System.arraycopy( this.array,no,out,0,ret ) ;
        
        return ret ;
        
    }
    
    /**
     * データを昇順でソート.
     * <BR><BR>
     * データを昇順でソートします.
     */
    public final void sort() {
        if( this.nowSize != 0 ){
            Arrays.sort( this.array,0,this.nowSize ) ;
        }
    }
    
    /**
     * 対象の条件が一致する先頭の値を取得.
     * <BR><BR>
     * 対象の条件と一致する先頭の値を取得します.
     * また、この処理の実行前に１度ソートする必要があります.
     * <BR>
     * @param key 対象の条件を設定します.
     * @return int 結果情報が返されます.<BR>
     *             [-1]が返された場合、条件の内容は存在しません.
     */
    public final int indexOf( byte key ) {
        return this.indexOf( key,0 ) ;
    }
    
    /**
     * 対象の条件が一致する先頭の値を取得.
     * <BR><BR>
     * 対象の条件と一致する先頭の値を取得します.
     * また、この処理の実行前に１度ソートする必要があります.
     * <BR>
     * @param key 対象の条件を設定します.
     * @return int 結果情報が返されます.<BR>
     *             [-1]が返された場合、条件の内容は存在しません.
     */
    public final int indexOf( byte key,int index ) {
        int len ;
        int ret ;
        byte[] tmp = null ;
        
        if( this.nowSize != 0 ){
            len = this.nowSize - index ;
            tmp = new byte[ len ] ;
            System.arraycopy( this.array,index,tmp,0,len ) ;
            Arrays.sort( tmp ) ;
            ret = Arrays.binarySearch( tmp,key ) ;
            tmp = null ;
            ret = ( ret < 0 || ret >= this.nowSize ) ? -1 : ret ;
        }
        else{
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の数値[byte(8bit)]配列を取得.
     * <BR><BR>
     * 対象の数値[byte(8bit)]配列を取得します.<BR>
     * また、この数値[byte(8bit)]配列は配列の再構成を行った場合、
     * 情報の保証は行われません.<BR>
     * また、この数値[byte(8bit)]群は基本的の読み込み専用で利用する
     * ことを進めます.
     * <BR>
     * @return byte[] 対象の数値[byte(8bit)]配列が返されます.
     */
    public final byte[] getObjects() {
        return this.array ;
    }
    
    /**
     * 格納情報数の取得.
     * <BR><BR>
     * 格納されている情報数を取得します.
     * <BR>
     * @return int 格納されている情報数が返されます.
     */
    public final int size() {
        return this.nowSize ;
    }
    
}

