/*
 * @(#)MaachangResult.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core ;

import org.maachang.index.MaachangResult;
import org.maachang.index.ResultChild;
import org.maachang.index.SubKey;

/**
 * 検索結果情報.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
class MaachangResultImpl implements MaachangResult {
    
    /**
     * デフォルト長.
     */
    public static final int DEF_LENGTH = IndexDef.DEFAULT_VIEW_LENGTH ;
    
    /**
     * 検索結果内容保持.
     */
    private ResultChildImpl[] result = null ;
    
    /**
     * 検索ワード.
     */
    private String word = null ;
    
    /**
     * サブ検索キー群.
     */
    private SubKey[] subkeys = null ;
    
    /**
     * 総件数.
     */
    private int maxResult = 0 ;
    
    /**
     * オフセット値.
     */
    private int offset = 0 ;
    
    /**
     * １度に保管するデータ長.
     */
    private int length = 0 ;
    
    /**
     * 検索結果ヒント情報表示最大長.
     */
    private int partLength = 0 ;
    
    /**
     * コンストラクタ.
     */
    private MaachangResultImpl() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param length １度に結果を保持するオブジェクトを設定します.
     * @param partLength 検索結果のヒント表示文字列長を設定します.
     */
    protected MaachangResultImpl( int length,int partLength ) {
        if( length <= 0 ) {
            length = DEF_LENGTH ;
        }
        this.length = length ;
        this.result = new ResultChildImpl[ length ] ;
        this.offset = 0 ;
        this.maxResult = 0 ;
        this.partLength = partLength ;
    }
    
    /**
     * 検索処理新規作成.
     * <BR><BR>
     * 検索処理を新規で作成します.
     * <BR>
     * @param word 検索ワードを設定します.
     * @param subkeys 検索サブキーを設定します.
     */
    public void create( String word,SubKey[] subkeys ) {
        this.word = word ;
        this.subkeys = subkeys ;
        this.maxResult = 0 ;
        this.offset = 0 ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear() {
        word = null ;
        subkeys = null ;
        maxResult = 0 ;
        offset = 0 ;
    }
    
    /**
     * 検索結果情報をクリア.
     * <BR><BR>
     * 検索結果情報をクリアします.
     */
    public void cleanResult() {
        int len = result.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            result[ i ] = null ;
        }
    }
    
    /**
     * 次データ遷移.
     * <BR><BR>
     * 次データ遷移する時の準備を行います.
     * <BR>
     * @return boolean 遷移結果が返されます.<BR>
     *                 [true]の場合は、遷移成功.
     */
    public boolean next() {
        int t = maxResult - ( offset + length ) ;
        if( t >= 0 ) {
            offset += length ;
            return true ;
        }
        return true ;
    }
    
    /**
     * 前データ遷移.
     * <BR><BR>
     * 前データ遷移する時の準備を行います.
     * <BR>
     * @return boolean 遷移結果が返されます.<BR>
     *                 [true]の場合は、遷移成功.
     */
    public boolean before() {
        int t = offset - length ;
        if( t >= 0 ) {
            offset = t ;
            return true ;
        }
        offset = 0 ;
        return true ;
    }
    
    /**
     * 指定項番でデータ遷移.
     * <BR><BR>
     * 指定項番からデータ遷移を行います.
     * <BR>
     * @param pageNo 項番を指定します.
     * @return boolean 遷移結果が返されます.<BR>
     *                 [true]の場合は、遷移成功.
     */
    public boolean change( int no ) {
        if( no <= 0 ) {
            return false ;
        }
        int t = ( no-1 ) * length ;
        if( t >= maxResult ) {
            return false ;
        }
        offset = t ;
        return true ;
    }
    
    /**
     * 表示ヒント文字列長を取得.
     * <BR><BR>
     * 表示ヒント文字列長を取得します.
     * <BR>
     * @return int 表示ヒント文字列長が返されます.
     */
    public int getPartLength() {
        return this.partLength ;
    }
    
    /**
     * 最大データ件数を設定.
     * <BR><BR>
     * 最大データ件数を設定します.
     * <BR>
     * @param maxReslut 最大データ件数を設定します.
     */
    public void setMaxResult( int maxResult ) {
        if( maxResult <= 0 ) {
            this.maxResult = 0 ;
        }
        else {
            this.maxResult = maxResult ;
        }
    }
    
    /**
     * 検索結果内容を設定.
     * <BR><BR>
     * 検索結果内容を設定します.
     * <BR>
     * @param ch 検索結果内容を設定します.
     * @param no 設定対象項番を設定します.
     */
    public void setResult( ResultChild ch,int no ) {
        if( no < 0 || no >= length ) {
            return ;
        }
        result[ no ] = ( ResultChildImpl )ch ;
    }
    
    /**
     * 検索キーワードを取得.
     * <BR><BR>
     * 検索キーワードを取得します.
     * <BR>
     * @return String 検索キーワードが返されます.
     */
    public String getSearchWord() {
        return word ;
    }
    
    /**
     * 検索サブキー群を取得.
     * <BR><BR>
     * 検索サブキー群を取得します.
     * <BR>
     * @return SubKey[] 検索サブキー群が返されます.
     */
    public SubKey[] getSubKeys() {
        return subkeys ;
    }
    
    /**
     * 最大データ件数を取得.
     * <BR><BR>
     * 最大データ件数を取得します.
     * <BR>
     * @return int 最大データ件数が返されます.
     */
    public int getMaxResult() {
        return maxResult ;
    }
    
    /**
     * 現在検索開始位置の取得.
     * <BR><BR>
     * 現在の検索開始位置を取得します.
     * <BR>
     * @return int 現在の検索開始位置が返されます.
     */
    public int getOffset() {
        return offset ;
    }
    
    /**
     * １度に検索結果を取得する検索情報件数を取得.
     * <BR><BR>
     * １度に検索結果を取得する検索情報件数を取得します.
     * <BR>
     * @return int １度に検索結果を取得する情報件数を取得します.
     */
    public int getViewLength() {
        return length ;
    }
    
    /**
     * 現在の有効件数を取得.
     * <BR><BR>
     * 現在の有効件数を取得します.
     * <BR>
     * @return int 現在の有効件数が返されます.
     */
    public int size() {
        if( maxResult <= length ) {
            return maxResult ;
        }
        int t = maxResult - offset ;
        if( t >= length ) {
            return length ;
        }
        return t ;
    }
    
    /**
     * 最大ページ項番を取得.
     * <BR><BR>
     * 最大ページ項番を取得します.
     * <BR>
     * @return int 最大ページ項番が返されます.
     */
    public int getMaxPageNo() {
        int t = maxResult / length ;
        if( ( maxResult % length ) != 0 ) {
            t ++ ;
        }
        return t ;
    }
    
    /**
     * 現在のページ項番を取得.
     * <BR><BR>
     * 現在のページ項番を取得します.
     * <BR>
     * @return int 現在のページ項番が返されます.
     */
    public int getNowPageNo() {
        return ( offset / length ) + 1 ;
    }
    
    /**
     * 検索結果内容を取得.
     * <BR><BR>
     * 検索結果内容を取得します.
     * <BR>
     * @param no 取得対象項番を設定します.
     * @return ResultChild 検索結果内容が返されます.
     */
    public ResultChild getResult( int no ) {
        if( no < 0 || no >= length ) {
            return null ;
        }
        return result[ no ] ;
    }
    
    /**
     * 次ページの情報が存在するかチェック.
     * <BR><BR>
     * 次ページの情報が存在するかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、次ページ情報は存在します.
     */
    public boolean isNext() {
        int t = maxResult - ( offset + length ) ;
        if( t >= 0 ) {
            return true ;
        }
        t = ( offset + length ) - maxResult ;
        if( t >= length ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * 前ページの情報が存在するかチェック.
     * <BR><BR>
     * 前ページの情報が存在するかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、前ページ情報は存在します.
     */
    public boolean isBefore() {
        int t = offset - length ;
        if( t >= 0 ) {
            return true ;
        }
        return false ;
    }
    
    /**
     * 指定ページに遷移できるかチェック.
     * <BR><BR>
     * 指定ページに遷移できるかチェックします.
     * <BR>
     * @param no 対象のページNoを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、指定ページ情報は存在します.
     */
    public boolean isPage( int no ) {
        if( no <= 0 ) {
            return false ;
        }
        int t = ( no-1 ) * length ;
        if( t >= maxResult ) {
            return false ;
        }
        return true ;
    }
}

