/*
 * @(#)IndexChipCache.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import org.maachang.util.FileUtil;

/**
 * IndexChipCache.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class IndexChipCache {
    
    /**
     * キャッシュサイズ : 最小値.
     */
    private static final int MIN_CACHE_SIZE = 128 ;
    
    /**
     * キャッシュサイズ : 最大値.
     */
    private static final int MAX_CACHE_SIZE = 262144 ;
    
    /**
     * キャッシュ管理.
     */
    private CacheMap<String,IndexChip> map = null ;
    
    /**
     * インデックス格納用ディレクトリ.
     */
    private String indexDir = null ;
    
    /**
     * モード.
     */
    private boolean mode = false ;
    
    /**
     * コンストラクタ.
     */
    private IndexChipCache() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 最大キャッシュ容量を設定してオブジェクトを生成.
     * <BR>
     * @param mode インデックス生成時の場合は[true]を設定します.
     * @param maxCache 最大キャッシュ容量を設定します.
     * @param indexDir 対象のインデックスディレクトリ名を設定します.
     */
    public IndexChipCache( boolean mode,int maxCache,String indexDir ) {
        if( maxCache <= MIN_CACHE_SIZE ) {
            maxCache = MIN_CACHE_SIZE ;
        }
        else if( maxCache >= MAX_CACHE_SIZE ) {
            maxCache = MAX_CACHE_SIZE ;
        }
        this.indexDir = indexDir ;
        this.map = new CacheMap<String,IndexChip>( maxCache ) ;
        this.mode = mode ;
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 情報を削除します.
     */
    public void clear() {
        map.clear() ;
    }
    
    /**
     * キャッシュ情報を全て出力.
     * <BR><BR>
     * キャッシュ情報を全て出力します.
     */
    public void flush() {
        int len = this.map.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            IndexChip bean = this.map.remove() ;
            if( bean != null ) {
                if( bean.size() <= 0 ) {
                    IndexChip.removeSerializable(
                        this.indexDir,bean.getIndexCode() ) ;
                    bean = null ;
                }
                else if( bean.isUpdate() == true ) {
                    bean.putSerialize( mode,this.indexDir ) ;
                }
            }
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 情報を追加します.
     * <BR>
     * @param bean 対象のIndexBean情報を設定します.
     * @exception IllegalArgumentException 入力例外.
     */
    public void add( IndexChip bean ) throws IllegalArgumentException {
        if( bean == null || bean.getIndexCode() == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( this.map.isKey( bean.getIndexCode() ) == true ) {
            return ;
        }
        IndexChip c = this.map.put( bean.getIndexCode(),bean ) ;
        if( c != null && c.isUpdate() == true && mode == true ) {
            if( c.size() > 0 ) {
                c.putSerialize( mode,this.indexDir ) ;
            }
            else {
                IndexChip.removeSerializable( this.indexDir,
                    c.getIndexCode() ) ;
                c = null ;
            }
        }
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象キーコードに対する情報を削除します.
     * <BR>
     * @param key 対象のキーコードを設定します.
     * @return IndexChip 情報が返されます.
     */
    public void remove( String key ) {
        String out = this.indexDir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( key ).toString() ;
        if( FileUtil.isFileExists( out ) == true ) {
            FileUtil.removeFile( out ) ;
        }
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象キーコードに対する情報を取得します.
     * <BR>
     * @param key 対象のキーコードを設定します.
     * @return IndexChip 情報が返されます.
     */
    public IndexChip get( String key ) {
        if( key == null ) {
            return null ;
        }
        IndexChip ret = this.map.get( key ) ;
        if( ret == null ) {
            ret = IndexChip.getSerializable( mode,this.indexDir,key ) ;
            if( ret != null ) {
                this.add( ret ) ;
            }
        }
        return ret ;
    }
    
    /**
     * インデックスディレクトリ名を取得.
     * <BR><BR>
     * インデックスディレクトリ名を取得します.
     * <BR>
     * @return String インデックスディレクトリ名が返されます.
     */
    public String getIndexDirectory() {
        return this.indexDir ;
    }
    
    /**
     * 最大キャッシュ容量を取得.
     * <BR><BR>
     * 最大キャッシュ容量を取得します.
     * <BR>
     * @return int 最大キャッシュ容量が返されます.
     */
    public int getMaxCache() {
        return this.map.getMaxLength() ;
    }
}

