/*
 * @(#)IndexNameCache.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;


/**
 * IndexNameCache.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class IndexNameCache {
    
    /**
     * キャッシュサイズ : 最小値.
     */
    private static final int MIN_CACHE_SIZE = 256 ;
    
    /**
     * キャッシュサイズ : 最大値.
     */
    private static final int MAX_CACHE_SIZE = 8192 ;
    
    /**
     * キャッシュ格納用.
     */
    private CacheMap<String,IndexName> map = null ;
    
    /**
     * 格納用ディレクトリ.
     */
    private String baseDir = null ;
    
    /**
     * コンストラクタ.
     */
    private IndexNameCache() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 最大キャッシュ容量を設定してオブジェクトを生成.
     * <BR>
     * @param maxCache 最大キャッシュ容量を設定します.
     * @param baseDir 対象のIndexBaseディレクトリ名を設定します.
     */
    public IndexNameCache( int maxCache,String baseDir ) {
        if( maxCache <= MIN_CACHE_SIZE ) {
            maxCache = MIN_CACHE_SIZE ;
        }
        else if( maxCache >= MAX_CACHE_SIZE ) {
            maxCache = MAX_CACHE_SIZE ;
        }
        this.baseDir = baseDir ;
        this.map =  new CacheMap<String,IndexName>( maxCache ) ;
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 情報を削除します.
     */
    public void clear() {
        map.clear() ;
    }
    
    /**
     * 格納中のキー内容をファイル出力.
     * <BR><BR>
     * 格納中のキー内容をファイルに出力します.
     */
    public void flush() {
        if( map != null ) {
            map.clear() ;
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 情報を追加します.
     * <BR>
     * @param name 対象名を設定します.
     * @param id 対象IDを設定します.
     * @param length 対象データ長を設定します.
     */
    public void add( String name,Long id,int length ) {
        if( this.map.get( name ) == null &&
            IndexName.isUse( this.baseDir,name ) == false ) {
            try {
                IndexName indexName = new IndexName() ;
                indexName.setName( name ) ;
                indexName.setId( id ) ;
                indexName.setLength( length ) ;
                indexName.putSerializable( this.baseDir ) ;
                this.map.put( name,indexName ) ;
            } catch( Exception e ) {
            }
        }
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 情報を削除します.
     * <BR>
     * @param name 対象名を設定します.
     * @param id 対象IDを設定します.
     */
    public void remove( String name ) {
        IndexName.removeSerializable( this.baseDir,name ) ;
        this.map.remove( name ) ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象条件の内容を取得します.
     * <BR>
     * @param name 対象の名前を設定します.
     * @return IndexName 情報が返されます.
     */
    public IndexName get( String name ) {
        IndexName ret = this.map.get( name ) ;
        if( ret == null ) {
            try {
                ret = IndexName.getSerializable( this.baseDir,name ) ;
                if( ret != null ) {
                    this.map.put( name,ret ) ;
                }
            } catch( Exception e ) {
                ret = null ;
            }
        }
        return ret ;
    }
    
    /**
     * 最大キャッシュ容量を取得.
     * <BR><BR>
     * 最大キャッシュ容量を取得します.
     * <BR>
     * @return int 最大キャッシュ容量が返されます.
     */
    public int getMaxCache() {
        return this.map.getMaxLength() ;
    }
}

