/*
 * @(#)IndexDef.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;

import org.maachang.index.core.IndexUtil;
import org.maachang.util.ConvertBinary;
import org.maachang.util.FileUtil;
import org.maachang.util.ObjectArray;
import org.maachang.util.ReadBinary;

/**
 * 検索結果データ.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class SearchValue {
    
    /**
     * 検索文字列.
     */
    private String searchString = null ;
    
    /**
     * 検索結果リスト一覧.
     */
    private ObjectArray list = null ;
    
    /**
     * 追加条件.
     */
    private boolean addFlag = false ;
    
    /**
     * シリアライズ条件.
     */
    private boolean serializeFlag = false ;
    
    /**
     * コンストラクタ.
     */
    private SearchValue() {
        serializeFlag = true ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 検索文字列を設定してオブジェクトを生成します.
     * <BR>
     * @param searchString 検索文字列を設定します.
     * @exception IllegalArgumentException 入力例外.
     */
    public SearchValue( String searchString )
        throws IllegalArgumentException {
        
        if( searchString == null || ( searchString = searchString.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        this.searchString = searchString ;
        this.list = new ObjectArray() ;
        serializeFlag = false ;
    }
    
    /**
     * シリアライズ化.
     * <BR><BR>
     * SearchBeanをシリアライズ化します.
     * <BR>
     * @param dir 対象のディレクトリ名を設定します.
     * @param bean 対象のSearchBeanを設定します.
     */
    public void putSerialize( String dir ) {
        BufferedOutputStream o = null ;
        String out = dir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( IndexUtil.getStringByNumber( this.searchString ) ).toString() ;
        try{
            o = new BufferedOutputStream( new FileOutputStream( out ) ) ;
            toFile( o ) ;
        }catch( Exception e ){
        }finally{
            try{
                o.close() ;
            }catch( Exception e ){
            }
            o = null ;
        }
    }
    
    /**
     * SearchBean情報を取得.
     * <BR><BR>
     * シリアライズされた情報から、SearchBeanを取得します.
     * <BR>
     * @param dir 対象のディレクトリ名を設定します.
     * @param string 対象の検索文字列を設定します.
     * @return SearchValue シリアライズから復元したSearchBeanオブジェクトが返されます.
     */
    public static final SearchValue getSerializable( String dir,String string ) {
        BufferedInputStream in = null ;
        String out = dir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( IndexUtil.getStringByNumber( string ) ).toString() ;
        if( FileUtil.isFileExists( out ) == false ) {
            return null ;
        }
        try{
            in = new BufferedInputStream(
                new FileInputStream( out )
            ) ;
            SearchValue ret = new SearchValue() ;
            ret.toObject( in ) ;
            return ret ;
        }catch( Exception e ){
        }finally{
            try{
                in.close() ;
            }catch( Exception e ){
            }
            in = null ;
        }
        return null ;
    }
    
    /**
     * SearchBean情報を削除.
     * <BR><BR>
     * SearchBean情報を削除します.
     * <BR>
     * @param dir 対象のディレクトリ名を設定します.
     * @param string 対象の検索文字列を設定します.
     */
    public static final void removeSerializable( String dir,String string ) {
        String out = dir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( IndexUtil.getStringByNumber( string ) ).toString() ;
        if( FileUtil.isFileExists( out ) == true ) {
            FileUtil.removeFile( out ) ;
        }
    }
    
    /**
     * 検索IDを追加.
     * <BR><BR>
     * 検索IDを追加します.
     * <BR>
     * @param id 対象のIDを設定します.
     * @param score 対象の得点を設定します.
     */
    public void add( Long id,int score ) {
        if( id == null || id.longValue() <= 0L || score < 0 ) {
            return ;
        }
        addFlag = true ;
        SearchChild ch = new SearchChild() ;
        ch.setId( id ) ;
        ch.setScore( score ) ;
        list.add( ch ) ;
    }
    
    /**
     * 検索文字列を取得.
     * <BR><BR>
     * 検索文字列を取得します.
     * <BR>
     * @return String 検索文字列が返されます.
     */
    public String getSearchString() {
        return this.searchString ;
    }
    
    /**
     * 検索IDを取得.
     * <BR><BR>
     * 検索IDを取得します.
     * <BR>
     * @param no 対象の項番を設定します.
     * @return Long 対象のIDが返されます.
     */
    public Long getId( int no ) {
        if( no < 0 || no >= list.size() ) {
            return null ;
        }
        SearchChild ch = ( SearchChild )list.get( no ) ;
        if( ch != null ) {
            return ch.getId() ;
        }
        return null ;
    }
    
    /**
     * 検索スコアを取得.
     * <BR><BR>
     * 検索スコアを取得します.
     * <BR>
     * @param no 対象の項番を設定します.
     * @return int 対象のスコアが返されます.
     */
    public int getScore( int no ) {
        if( no < 0 || no >= list.size() ) {
            return -1 ;
        }
        SearchChild ch = ( SearchChild )list.get( no ) ;
        if( ch != null ) {
            return ch.getScore() ;
        }
        return -1 ;
    }
    
    /**
     * 情報数を取得.
     * <BR><BR>
     * 情報数を取得します.
     * <BR>
     * @return int 検索情報数が返されます.
     */
    public int size() {
        return list.size() ;
    }
    
    /**
     * ソート処理.
     * <BR><BR>
     * ソート処理を実施.
     */
    public void sort() {
        
        if( addFlag == false ) {
            return ;
        }
        addFlag = false ;
        list.sort() ;
    }
    
    /**
     * シリアライズフラグ.
     */
    public boolean isSerialize() {
        return serializeFlag ;
    }
    
    /**
     * オブジェクトをファイル化.
     */
    private void toFile( BufferedOutputStream bo )
        throws Exception {
        byte[] bin = null ;
        bin = ConvertBinary.convertString( searchString ) ;
        bo.write( ConvertBinary.convertInt( bin.length ) ) ;
        bo.write( bin ) ;bin = null ;
        if( list != null && list.size() > 0 ) {
            int len = list.size() ;
            bo.write( ConvertBinary.convertInt( len ) ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                bo.write( ( ( SearchChild )list.get( i ) ).toBinary() ) ;
            }
        }
        else {
            bo.write( ConvertBinary.convertInt( 0 ) ) ;
        }
        bo.flush() ;
    }
    
    /**
     * ファイルからオブジェクトに変換.
     */
    private void toObject( BufferedInputStream bi )
        throws Exception {
        int len ;
        ReadBinary r = new ReadBinary( bi ) ;
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        searchString = ConvertBinary.convertString( 0,len,r.getBinary( len ) ) ;
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        this.list = new ObjectArray() ;
        for( int i = 0 ; i < len ; i ++ ) {
            SearchChild ch = new SearchChild() ;
            ch.toObject( r.getBinary( SearchChild.BINARY_LENGTH ) ) ;
            list.add( ch ) ;
        }
        addFlag = false ;
    }
}

/**
 * 検索条件子要素.
 */
class SearchChild implements Comparable {
    
    /**
     * バイナリ長.
     */
    public static final int BINARY_LENGTH = 12 ;
    
    /**
     * 検索ID.
     */
    private Long id = null ;
    
    /**
     * 検索得点.
     */
    private int score = 0 ;
    
    /**
     * コンストラクタ.
     */
    protected SearchChild() {
        
    }
    
    /**
     * IDを設定.
     * <BR><BR>
     * IDを設定します.
     * <BR>
     * @param id 対象のIDを設定します.
     */
    public void setId( Long id ) {
        this.id = id ;
    }
    
    /**
     * IDを取得.
     * <BR><BR>
     * IDを取得します.
     * <BR>
     * @return Long IDが返されます.
     */
    public Long getId() {
        return this.id ;
    }
    
    /**
     * 得点を設定.
     * <BR><BR>
     * 得点を設定します.
     * <BR>
     * @param score 得点を設定します.
     */
    public void setScore( int score ) {
        this.score = score ;
    }
    
    /**
     * 得点を取得.
     * <BR><BR>
     * 得点を取得します.
     * <BR>
     * @return int 得点が返されます.
     */
    public int getScore() {
        return score ;
    }
    
    /**
     * ソート条件で利用.
     * <BR><BR>
     * ソート条件で利用します.
     * <BR>
     * @param o 対象のオブジェクト(IndexChild)を設定します.
     * @return int 結果が返されます.
     */
    public int compareTo( Object o ) {
        if( o == null || ( o instanceof SearchChild ) == false ) {
            return 0 ;
        }
        
        return ( ( SearchChild )o ).score - this.score ;
    }
    
    /**
     * オブジェクト内容をバイナリ化.
     * <BR><BR>
     * オブジェクト内容をバイナリ化します.
     * <BR>
     * @return byte[] バイナリ情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] toBinary() throws Exception {
        ByteArrayOutputStream bo = new ByteArrayOutputStream() ;
        bo.write( ConvertBinary.convertLong( id.longValue() ) ) ;
        bo.write( ConvertBinary.convertInt( score ) ) ;
        return bo.toByteArray() ;
    }
    
    /**
     * 対象バイナリからオブジェクトを生成.
     * <BR><BR>
     * 対象バイナリからオブジェクトを生成します.
     * <BR>
     * @param bin 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public void toObject( byte[] bin ) throws Exception {
        id = new Long( ConvertBinary.convertLong( 0,bin ) ) ;
        score = ConvertBinary.convertInt( 8,bin ) ;
    }
}
