package org.maachang.dbm.engine ;

import java.io.IOException;
import java.util.Enumeration;

/**
 * MaachangDbmEngine.
 * 
 * @version 2008/01/18
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MDbmEngine {
    
    /**
     * キー最大長.
     */
    public static final int MAX_KEY_LENGTH = MKey.MAX_KEY_LENGTH ;
    
    /**
     * Key管理.
     */
    private MKey key = null ;
    
    /**
     * 要素管理.
     */
    private MValue value = null ;
    
    /**
     * 基本同期.
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private MDbmEngine() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param key 対象のキー管理オブジェクトを設定します.
     * @param value 対象の要素管理オブジェクトを設定します.
     * @exception Exception 例外.
     */
    public MDbmEngine( MKey key,MValue value ) throws Exception {
        if( key == null || key.isUse() == false ||
            value == null || value.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.key = key ;
        this.value = value ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        synchronized( sync ) {
            if( key != null ) {
                key.destroy() ;
            }
            if( value != null ) {
                value.destroy() ;
            }
            this.key = null ;
            this.value = null ;
        }
    }
    
    /**
     * 強制書き込み.
     * <BR>
     * @exception Exception 例外.
     */
    public void flush() throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            key.flush() ;
            value.flush() ;
        }
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( byte[] key,byte[] value ) throws Exception {
        if( key == null || key.length <= 0 ||
            value == null || value.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( key.length > MKey.MAX_KEY_LENGTH ) {
            throw new IllegalArgumentException( "最大キー長["+MKey.MAX_KEY_LENGTH+
                "]を越しています" ) ;
        }
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            int code = getHash( key ) ;
            // 格納可能かチェック(key).
            if( this.key.get( code,key ) == null ) {
                if( this.key.putCheck( 1 ) == false ) {
                    throw new IOException( "格納最大数["+MKey.MAX_LENGTH+
                        "]を越しているため、put処理はできません" ) ;
                }
            }
            // 格納可能かチェック(value).
            if( this.value.putCheck( MSctArray.sectorLength( value.length ) ) == false ) {
                throw new IOException( "容量が足りないためput処理はできません" ) ;
            }
            int[] putPos = null ;
            int[] befPos = null ;
            putPos = this.value.put( value ) ;
            befPos = this.key.put( code,key,putPos[ 0 ],putPos[ 1 ] ) ;
            if( befPos != null ) {
                this.value.remove( befPos[ 0 ],befPos[ 1 ] ) ;
            }
        }
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( byte[] key ) throws Exception {
        if( key == null || key.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( key.length > MKey.MAX_KEY_LENGTH ) {
            throw new IllegalArgumentException( "最大キー長["+MKey.MAX_KEY_LENGTH+
                "]を越しています" ) ;
        }
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            int code = getHash( key ) ;
            int[] valPos = null ;
            valPos = this.key.remove( code,key ) ;
            if( valPos != null ) {
                this.value.remove( valPos[ 0 ],valPos[ 1 ] ) ;
            }
        }
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( byte[] key ) throws Exception {
        if( key == null || key.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( key.length > MKey.MAX_KEY_LENGTH ) {
            throw new IllegalArgumentException( "最大キー長["+MKey.MAX_KEY_LENGTH+
                "]を越しています" ) ;
        }
        byte[] ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            int code = getHash( key ) ;
            int[] valPos = null ;
            valPos = this.key.get( code,key ) ;
            if( valPos != null ) {
                ret = this.value.get( valPos[ 0 ],valPos[ 1 ] ) ;
            }
        }
        return ret ;
    }
    
    /**
     * 指定キーが存在するかチェック.
     * <BR>
     * @param key チェック対象のキー内容を設定します.
     * @return boolean [true]の場合、情報が存在します.
     * @exception Exception 例外.
     */
    public boolean containsKey( byte[] key) throws Exception {
        if( key == null || key.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( key.length > MKey.MAX_KEY_LENGTH ) {
            throw new IllegalArgumentException( "最大キー長["+MKey.MAX_KEY_LENGTH+
                "]を越しています" ) ;
        }
        boolean ret = false ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            int code = getHash( key ) ;
            if( this.key.get( code,key ) != null ) {
                ret = true ;
            }
        }
        return ret ;
    }
    
    /**
     * キー情報一覧を取得.
     * <BR>
     * @param nextKey 次のキー位置を保持するオブジェクトを設定します.<BR>
     *                [null]を設定した場合、初めから取得します.
     * @return NextKey 次のキー情報を格納したオブジェクトが返されます.<BR>
     *                 [null]の場合、それ以上キー情報は存在しません.
     * @exception Exception 例外.
     */
    public NextKey next( NextKey nextKey ) throws Exception {
        NextKey ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            ret = key.nextKey( nextKey ) ;
        }
        return ret ;
    }
    
    /**
     * キー内容を列挙.
     * <BR><BR>
     * @return  Enumeration<byte[]> 列挙オブジェクトが返されます.
     */
    public Enumeration<byte[]> elements() {
        return new MDbmEnumeration( this ) ;
    }
    
    /**
     * 新しいシーケンスIDを取得.
     * <BR>
     * @param no シーケンスNoを設定します.<BR>
     *           [0-31]まで利用可能です.
     * @return long 新しいシーケンスIDが返されます.
     * @exception Exception 例外.
     */
    public long sequenceId( int no ) throws Exception {
        long ret = -1L ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            ret = key.getMHash().sequenceId( no ) ;
        }
        return ret ;
    }
    
    /**
     * 格納情報数を取得.
     * <BR>
     * @return int 格納情報数が返されます.<BR>
     *             [-1]が返された場合、オブジェクトは既に破棄されています.
     */
    public int size() {
        int ret = -1 ;
        synchronized( sync ) {
            if( check() == true ) {
                ret = key.size() ;
            }
        }
        return ret ;
    }
    
    /**
     * キー管理オブジェクトを取得.
     * <BR>
     * @return MKey キー管理オブジェクトを取得します.
     */
    public MKey getMKey() {
        MKey ret = null ;
        synchronized( sync ) {
            if( check() == true ) {
                ret = this.key ;
            }
        }
        return ret ;
    }
    
    /**
     * データ管理オブジェクトを取得.
     * <BR>
     * @return MValue データ管理オブジェクトを取得します.
     */
    public MValue getMValue() {
        MValue ret = null ;
        synchronized( sync ) {
            if( check() == true ) {
                ret = this.value ;
            }
        }
        return ret ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        boolean ret = false ;
        synchronized( sync ) {
            ret = check() ;
        }
        return ret ;
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR>
     * @return Object 同期オブジェクトが返されます.
     */
    public Object sync() {
        return sync ;
    }
    
    /**
     * 指定データ長をセクター数に変換.
     * <BR><BR>
     * 指定データ長をセクター数に変換します.
     * <BR>
     * @param length 対象のデータ長を設定します.
     * @return int 変換されたセクター数が返されます.
     */
    public static final int convertLengthBySectorLength( int length ) {
        return MSctArray.sectorLength( length ) ;
    }
    
    /**
     * 指定バイナリのHash値を取得.
     * <BR><BR>
     * 指定バイナリのHash値を取得します.
     * <BR>
     * @param binary 対象のバイナリを設定します.
     * @return int 対象のHash値が返されます.
     * @exception Exception 例外.
     */
    public static final int getBinaryHash( byte[] binary ) throws Exception {
        if( binary == null || binary.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( binary.length > MKey.MAX_KEY_LENGTH ) {
            throw new IllegalArgumentException( "最大キー長["+MKey.MAX_KEY_LENGTH+
                "]を越しています" ) ;
        }
        return FnvHash.fnv32a( binary ) ;
    }
    
    /**
     * まとまった情報を追加する場合に、追加可能かチェック.
     * <BR><BR>
     * まとまった情報を追加する場合に、追加可能かチェックします.
     * <BR>
     * @param addKeyLength 新しく追加するキー数を設定します.
     * @param addSectorLength 新しく追加するセクター数を設定します.
     * @exception Exception 例外.
     */
    public void addChecks( int addKeyLength,int addSectorLength )
        throws Exception {
        if( addKeyLength > 0 ) {
            if( this.key.putCheck( addKeyLength ) == false ) {
                throw new IOException( "格納最大数["+MKey.MAX_LENGTH+
                    "]を越しています" ) ;
            }
        }
        if( addSectorLength > 0 ) {
            if( this.value.putCheck( addKeyLength ) == false ) {
                throw new IOException( "データ追加に対して、容量が足りません" ) ;
            }
        }
    }
    
    private boolean check() {
        if( key == null || key.isUse() == false ||
            value == null || value.isUse() == false ) {
            return false ;
        }
        return true ;
    }
    
    private int getHash( byte[] key ) {
        return FnvHash.fnv32a( key ) ;
    }
}
