package org.maachang.dbm.engine ;

import java.io.IOException;

import org.maachang.util.FileUtil;

/**
 * セクター管理オブジェクト.
 * 
 * @version 2008/01/17
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MSector {
    
    /**
     * 1セクター数.
     */
    protected static final int ONE_SECTOR = 8206 ;
    
    /**
     * １データ長.
     */
    protected static final int ONE_DATA = 8192 ;
    
    /**
     * １つのファイルで管理するセクター数.
     */
    protected static final int MAX_SECTOR = 65536 ;
    
    /**
     * オフセット値.
     */
    private static final int OFFSET = 8204 ;
    
    /**
     * キー管理用ファイル.
     */
    private RandIO fp = null ;
    
    /**
     * 空き情報管理用.
     */
    private FlagMapFile space = null ;
    
    /**
     * 番号.
     */
    private int fileNo = -1 ;
    
    /**
     * 同期オブジェクト.
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private MSector() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を指定して、オブジェクトを生成します.
     * <BR>
     * @param fileNo 対象のファイルNoを設定します.
     * @param filename 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public MSector( int fileNo,String filename ) throws Exception {
        if( fileNo <= -1 || fileNo >= MKey.MAX_VALUE_FILE_SIZE ||
            filename == null || ( filename = filename.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        filename = FileUtil.getFullPath( filename ) ;
        boolean isFile = FileUtil.isFileExists( filename ) ;
        RandIO fp = null ;
        if( isFile == false ) {
            fp = new RandIO( ONE_SECTOR,OFFSET + ( ONE_SECTOR * MAX_SECTOR ),filename ) ;
        }
        else {
            fp = new RandIO( ONE_SECTOR,filename ) ;
        }
        this.space = initSpace( isFile,fp ) ;
        this.fp = fp ;
        this.fileNo = fileNo ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        synchronized( sync ) {
            if( this.space != null ) {
                this.space.clear() ;
            }
            if( this.fp != null ) {
                this.fp.destroy() ;
            }
            this.fp = null ;
            this.space = null ;
            this.fileNo = -1 ;
        }
    }
    
    /**
     * オブジェクト更新.
     * <BR><BR>
     * オブジェクトを更新します.
     * <BR>
     * @exception Exception 例外.
     */
    public void flush() throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            space.flush() ;
        }
    }
    
    private static final FlagMapFile initSpace( boolean isFile,RandIO fp ) throws Exception {
        return new FlagMapFile( isFile == false,fp.getRandomAccessFile(),0,MAX_SECTOR ) ;
    }
    
    /**
     * 新しいセクターに予約.
     * <BR><BR>
     * 新しいセクターに予約します.
     * <BR>
     * @param sector 対象のセクター情報を設定します.
     * @return boolean [true]の場合、予約できました.
     * @exception Exception 例外.
     */
    public boolean add( ValueSector sector ) throws Exception {
        if( sector == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            int pos = space.usePos( 0 ) ;
            if( pos <= -1 ) {
                return false ;
            }
            sector.setPosition( pos ) ;
            sector.setFileNo( this.fileNo ) ;
            space.setPos( pos ) ;
        }
        return true ;
    }
    
    /**
     * 既に存在するセクター位置に情報を書き込む.
     * <BR><BR>
     * 既に存在するセクター位置に情報を書き込みます.
     * <BR>
     * @param no セクター項番を設定します.
     * @param sector 対象のセクター情報を設定します.
     * @exception Exception 例外.
     */
    public void put( int no,ValueSector sector ) throws Exception {
        if( sector == null || sector.getLength() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            if( space.getPos( no ) == false ) {
                throw new IOException( "指定セクター("+no+")は[OFF]利用されていません" ) ;
            }
            saveBySector( no,sector ) ;
        }
    }
    
    /**
     * 指定セクター項番の内容を破棄.
     * <BR><BR>
     * 指定セクター項番の内容を破棄します.
     * <BR>
     * @param no セクター項番を設定します.
     * @return ValueSector 対象のセクター内容が返されます.
     * @exception Exception 例外.
     */
    public ValueSector remove( int no ) throws Exception {
        ValueSector ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            if( space.getPos( no ) == true ) {
                ret = posBySector( no ) ;
                if( ret != null ) {
                    //removeBySector( no ) ;
                    space.removePos( no ) ;
                }
            }
        }
        return ret ;
    }
    
    /**
     * 指定セクター項番の内容を取得.
     * <BR><BR>
     * 指定セクター項番の内容を取得します.
     * <BR>
     * @param no セクター項番を設定します.
     * @return ValueSector 対象のセクター内容が返されます.
     * @exception Exception 例外.
     */
    public ValueSector get( int no ) throws Exception {
        ValueSector ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            if( space.getPos( no ) == true ) {
                ret = posBySector( no ) ;
            }
        }
        return ret ;
    }
    
    /**
     * ファイル項番を取得.
     * <BR><BR>
     * ファイル項番を取得します.
     * <BR>
     * @return int ファイル項番が返されます.
     */
    public int getFileNo() {
        int ret = -1 ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = -1 ;
            }
            else {
                ret = this.fileNo ;
            }
        }
        return ret ;
    }
    
    /**
     * ファイル名を取得.
     * <BR><BR>
     * ファイル名を取得します.
     * <BR>
     * @return String ファイル名が返されます.
     */
    public String getFileName() {
        String ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = null ;
            }
            else {
                ret = this.fp.getName() ;
            }
        }
        return ret ;
    }
    
    /**
     * 現在のデータ数を取得.
     * <BR><BR>
     * 現在のデータ数が返されます.
     * <BR>
     * @return int 現在のデータ数が返されます.
     */
    public int size() {
        int ret = 0 ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = -1 ;
            }
            else {
                ret = space.size() ;
            }
        }
        return ret ;
    }
    
    /**
     * セクター空き容量を取得.
     * <BR><BR>
     * セクター空き容量を取得します.
     * <BR>
     * @return int セクター空き容量が返されます.
     */
    public int useSector() {
        int ret = 0 ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = -1 ;
            }
            else {
                ret = space.maxSize() - space.size() ;
            }
        }
        return ret ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        boolean ret = false ;
        synchronized( sync ) {
            ret = check() ;
        }
        return ret ;
    }
    
    /**
     * １つのセクター管理ファイルサイズを取得.
     */
    protected static final long oneSectorFileSize() {
        return ( long )( OFFSET + ( ONE_SECTOR * MAX_SECTOR ) ) ;
    }
    
    private boolean check() {
        if( fp == null || space == null ) {
            return false ;
        }
        return true ;
    }
    
    private static final int getPosByAddress( int pos ) {
        return OFFSET + ( pos * ONE_SECTOR ) ;
    }
    
    private ValueSector posBySector( int pos ) throws Exception {
        byte[] b = new byte[ ONE_SECTOR ] ;
        int addr = getPosByAddress( pos ) ;
        synchronized( sync ) {
            fp.read( b,( long )addr ) ;
        }
        ValueSector ret = ValueSector.load( b ) ;
        ret.setPosition( pos ) ;
        ret.setFileNo( this.fileNo ) ;
        return ret ;
    }
    
    private void saveBySector( int pos,ValueSector sector ) throws Exception {
        if( sector == null ) {
            throw new IOException( "不正なセクタ["+getPosByAddress( pos )+
                "]保存を行おうとしました" ) ;
        }
        byte[] b = sector.save() ;
        int addr = getPosByAddress( pos ) ;
        synchronized( sync ) {
            fp.write( b,( long )addr ) ;
        }
        sector.setPosition( pos ) ;
        sector.setFileNo( this.fileNo ) ;
    }
    
    /*private void removeBySector( int pos ) throws Exception {
        int addr = getPosByAddress( pos ) ;
        byte[] b = new byte[ ONE_SECTOR ] ;
        Arrays.fill( b,( byte )-1 ) ;
        synchronized( sync ) {
            fp.write( b,( long )addr ) ;
        }
    }*/
}

