package org.maachang.dbm.engine ;

import java.io.IOException;
import java.util.Arrays;

import org.maachang.util.ConvertParam;

/**
 * １つのセクター情報.
 * 
 * @version 2008/01/17
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
class ValueSector {
    
    /**
     * 前セクター位置 : ファイルNo.
     */
    private int beforeFileNo = -1 ;
    
    /**
     * 前セクター位置 : ファイル項番.
     */
    private int beforeFilePos = -1 ;
    
    /**
     * 次セクター位置 : ファイルNo.
     */
    private int nextFileNo = -1 ;
    
    /**
     * 次セクター位置 : ファイル項番.
     */
    private int nextFilePos = -1 ;
    
    /**
     * 有効データ長.
     */
    private int length = -1 ;
    
    /**
     * データ格納領域.
     */
    private byte[] data = null ;
    
    /**
     * データポジション.
     */
    private int position = -1 ;
    
    /**
     * ファイルNo.
     */
    private int fileNo = -1 ;
    
    /**
     * コンストラクタ.
     */
    public ValueSector() {
        this.length = 0 ;
        this.data = null ;
    }
    
    /**
     * 前セクターファイルNoを設定.
     * <BR>
     * @param beforeFileNo 前セクターファイルNoを設定します.
     */
    public void setBeforeFileNo( int beforeFileNo ) {
        this.beforeFileNo = beforeFileNo ;
    }
    
    /**
     * 前セクターファイルNoを取得.
     * <BR>
     * @return int 前セクターファイルNoが返されます.
     */
    public int getBeforeFileNo() {
        return this.beforeFileNo ;
    }
    
    /**
     * 前セクターファイル項番を設定.
     * <BR>
     * @param beforeFilePos 前セクターファイル項番を設定します.
     */
    public void setBeforeFilePos( int beforeFilePos ) {
        this.beforeFilePos = beforeFilePos ;
    }
    
    /**
     * 前セクターファイル項番を取得.
     * <BR>
     * @return int 前セクターファイル項番が返されます.
     */
    public int getBeforeFilePos() {
        return this.beforeFilePos ;
    }
    
    /**
     * 次セクターファイルNoを設定.
     * <BR>
     * @param nextFileNo 次セクターファイルNoを設定します.
     */
    public void setNextFileNo( int nextFileNo ) {
        this.nextFileNo = nextFileNo ;
    }
    
    /**
     * 次セクターファイルNoを取得.
     * <BR>
     * @return int 次セクターファイルNoが返されます.
     */
    public int getNextFileNo() {
        return this.nextFileNo ;
    }
    
    /**
     * 次セクターファイル項番を設定.
     * <BR>
     * @param nextFilePos 次セクターファイル項番を設定します.
     */
    public void setNextFilePos( int nextFilePos ) {
        this.nextFilePos = nextFilePos ;
    }
    
    /**
     * 前セクターファイル項番を取得.
     * <BR>
     * @return int 前セクターファイル項番が返されます.
     */
    public int getNextFilePos() {
        return this.nextFilePos ;
    }
    
    /**
     * 有効データ長を設定.
     * <BR>
     * @param length 有効データ長を設定します.
     */
    public void setLength( int length ) {
        if( length < 0 || length >= MSector.ONE_DATA ) {
            return ;
        }
        this.length = length ;
    }
    
    /**
     * 有効データ長を取得.
     * <BR>
     * @return int 有効データ長が返されます.
     */
    public int getLength() {
        return this.length ;
    }
    
    /**
     * データ格納内容を取得.
     * <BR>
     * @return byte[] データ格納内容が返されます.
     */
    public byte[] getData() {
        return this.data ;
    }
    
    /**
     * データを取得.
     * <BR>
     * @return byte[] 指定セクター内の有効バイナリ内容が返されます.
     * @exception Exception 例外.
     */
    public byte[] read() throws Exception {
        if( this.length <= 0 ) {
            return null ;
        }
        byte[] ret = new byte[ this.length ] ;
        System.arraycopy( this.data,0,ret,0,this.length ) ;
        return ret ;
    }
    
    /**
     * データを取得.
     * <BR>
     * @param out データ格納バイナリを設定します.
     * @param pos 取得開始ポジションを設定します.
     * @return int 取得されたデータ長が返されます.
     * @exception Exception 例外.
     */
    public int read( byte[] out,int pos ) throws Exception {
        return read( out,pos,0,out.length ) ;
    }
    
    /**
     * データを取得.
     * <BR>
     * @param out データ格納バイナリを設定します.
     * @param pos 取得開始ポジションを設定します.
     * @param offset データ格納バイナリオフセット値を設定します.
     * @param length データ取得長を設定します.
     * @return int 取得されたデータ長が返されます.
     * @exception Exception 例外.
     */
    public int read( byte[] out,int pos,int offset,int length )
        throws Exception {
        if( pos + length >= this.length ) {
            length = this.length - pos ;
        }
        System.arraycopy( this.data,pos,out,offset,length ) ;
        return length ;
    }
    
    /**
     * データを書き込む.
     * <BR>
     * @param in データ書き込み対象バイナリを設定します.
     * @param pos 書き込み開始ポジションを設定します.
     * @exception Exception 例外.
     */
    public void write( byte[] in,int pos ) throws Exception {
        write( in,pos,0,in.length ) ;
    }
    
    /**
     * データを書き込む.
     * <BR>
     * @param in データ書き込み対象バイナリを設定します.
     * @param pos 書き込み開始ポジションを設定します.
     * @param offset データ書き込み対象バイナリオフセット値を設定します.
     * @param length データ書き込み長を設定します.
     * @exception Exception 例外.
     */
    public void write( byte[] in,int pos,int offset,int length )
        throws Exception {
        if( pos + length > MSector.ONE_DATA ) {
            throw new IllegalArgumentException( "指定条件(pos:" + pos + " len:" + length +
                ")は１セクタ範囲を超しています" ) ;
        }
        if( this.data == null ) {
            this.data = new byte[ MSector.ONE_DATA ] ;
        }
        System.arraycopy( in,offset,this.data,pos,length ) ;
        if( this.length <= pos + length ) {
            this.length = pos + length ;
        }
    }
    
    /**
     * セクターポジションを設定.
     * <BR>
     * @param position セクターポジションを設定します.
     */
    public void setPosition( int position ) {
        this.position = position ;
    }
    
    /**
     * セクターポジションを取得.
     * <BR>
     * @return int セクターポジションが返されます.
     */
    public int getPosition() {
        return this.position ;
    }
    
    /**
     * ファイルNoを設定.
     * <BR>
     * @param fileNo ファイルNoを設定します.
     */
    public void setFileNo( int fileNo ) {
        this.fileNo = fileNo ;
    }
    
    /**
     * ファイルNoを取得.
     * <BR>
     * @return int ファイルNoが返されます.
     */
    public int getFileNo() {
        return this.fileNo ;
    }
    
    protected ValueSector copy() {
        ValueSector ret = new ValueSector() ;
        ret.beforeFileNo = this.beforeFileNo ;
        ret.beforeFilePos = this.beforeFilePos ;
        ret.nextFileNo = this.nextFileNo ;
        ret.nextFilePos = this.nextFilePos ;
        ret.length = this.length ;
        ret.position = this.position ;
        ret.fileNo = this.fileNo ;
        System.arraycopy( this.data,0,ret.data,0,this.length ) ;
        return ret ;
    }
    
    protected void clear() {
        this.beforeFileNo = -1 ;
        this.beforeFilePos = -1 ;
        this.nextFileNo = -1 ;
        this.nextFilePos = -1 ;
        this.length = -1 ;
        this.position = -1 ;
        this.fileNo = -1 ;
        Arrays.fill( this.data,( byte )0 ) ;
    }
    
    protected static final ValueSector load( byte[] binary ) throws Exception {
        if( binary == null || binary.length != MSector.ONE_SECTOR ) {
            throw new IllegalArgumentException( "対象バイナリは不正です" ) ;
        }
        ValueSector ret = new ValueSector() ;
        int pnt = 0 ;
        ret.beforeFileNo = ( int )( ConvertParam.convertShort( pnt,binary ) & 0x0000ffff ) ;
        ret.beforeFileNo = ( ( ret.beforeFileNo & 0xfffff400 ) != 0 ) ? -1 : ret.beforeFileNo ;
        pnt += 2 ;
        ret.beforeFilePos = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        ret.nextFileNo = ( int )( ConvertParam.convertShort( pnt,binary ) & 0x0000ffff ) ;
        ret.nextFileNo = ( ( ret.nextFileNo & 0xfffff400 ) != 0 ) ? -1 : ret.nextFileNo ;
        pnt += 2 ;
        ret.nextFilePos = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        ret.length = ( int )( ConvertParam.convertShort( pnt,binary ) & 0x0000ffff ) ;
        pnt += 2 ;
        ret.data = new byte[ MSector.ONE_DATA ] ;
        System.arraycopy( binary,pnt,ret.data,0,ret.length ) ;
        return ret ;
    }
    
    protected byte[] save() throws Exception {
        if( this.data == null || this.length <= 0 ) {
            throw new IOException( "保存対象の情報は存在しません" ) ;
        }
        byte[] ret = new byte[ MSector.ONE_SECTOR ] ;
        int pnt = 0 ;
        ConvertParam.convertShort( ret,pnt,( short )( this.beforeFileNo & 0x0000ffff ) ) ;
        pnt += 2 ;
        ConvertParam.convertInt( ret,pnt,this.beforeFilePos ) ;
        pnt += 4 ;
        ConvertParam.convertShort( ret,pnt,( short )( this.nextFileNo & 0x0000ffff ) ) ;
        pnt += 2 ;
        ConvertParam.convertInt( ret,pnt,this.nextFilePos ) ;
        pnt += 4 ;
        ConvertParam.convertShort( ret,pnt,( short )( this.length & 0x0000ffff ) ) ;
        pnt += 2 ;
        System.arraycopy( this.data,0,ret,pnt,this.length ) ;
        return ret ;
    }
}
