package org.maachang.dbm.service ;

import java.io.IOException;
import java.net.InetAddress;

import org.maachang.dbm.MDbmManager;

/**
 * MaachangDbmサーバオブジェクト.
 * 
 * @version 2008/01/18
 * @author masahito suzuki
 * @since MaachangDBM 1.03
 */
public class MDbmServer {
    
    /**
     * 最小PoolThread.
     */
    private static final int MIN_POOL = 1 ;
    
    /**
     * 最大PoolThread.
     */
    private static final int MAX_POOL = 127 ;
    
    /**
     * コネクション管理オブジェクト.
     */
    private SocketManager conns = null ;
    
    /**
     * 受信開始キュー.
     */
    private ReceiveQueue queue = null ;
    
    /**
     * AcceptThread.
     */
    private AcceptThread accepts = null ;
    
    /**
     * ReceiveThread.
     */
    private ReceiveThread receives = null ;
    
    /**
     * PoolThread群.
     */
    private PoolThread[] pools = null ;
    
    /**
     * コンストラクタ.
     */
    private MDbmServer() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 指定条件でオブジェクトを生成します.<BR>
     * また、このオブジェクトを生成する前に、MDbmManagerを初期化する
     * 必要があります.
     * <BR>
     * @param addr 対象のバインドアドレスを設定します.
     * @param port 対象のバインドポート番号を設定します.
     * @param backlog 対象の最大接続数を設定します.
     * @param pool 対象のプーリングスレッド数を設定します.
     * @exception Exception 例外.
     */
    public MDbmServer( InetAddress addr,int port,int backlog,int pool )
        throws Exception {
        if( MDbmManager.getInstance().isClose() == true ) {
            throw new IOException( "MaachangDbmは開始していません" ) ;
        }
        if( pool <= MIN_POOL ) {
            pool = MIN_POOL ;
        }
        if( pool >= MAX_POOL ) {
            pool = MAX_POOL ;
        }
        try {
            this.conns = new SocketManager( addr,port,backlog ) ;
            this.queue = new ReceiveQueue() ;
            this.accepts = new AcceptThread( conns ) ;
            this.receives = new ReceiveThread( conns,queue ) ;
            this.pools = new PoolThread[ pool ] ;
            for( int i = 0 ; i < pool ; i ++ ) {
                this.pools[ i ] = new PoolThread( conns,queue ) ;
            }
        } catch( Exception e ) {
            this.destroy() ;
            throw e ;
        }
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        if( accepts != null ) {
            accepts.destroy() ;
        }
        if( receives != null ) {
            receives.destroy() ;
        }
        if( pools != null ) {
            int len = pools.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( pools[ i ] != null ) {
                    pools[ i ].destroy() ;
                }
            }
        }
        if( queue != null ) {
            queue.destroy() ;
        }
        if( conns != null ) {
            conns.destroy() ;
        }
    }
    
    /**
     * バインドアドレスを取得.
     * <BR>
     * @return InetAddress バインドアドレスが返されます.
     */
    public InetAddress getInetAddress() {
        if( conns != null ) {
            return conns.getServerSocket().getInetAddress() ;
        }
        return null ;
    }
    
    /**
     * バインドポートを取得.
     * <BR>
     * @return int バインドポートが返されます.
     */
    public int getLocalPort() {
        if( conns != null ) {
            return conns.getServerSocket().getLocalPort() ;
        }
        return -1 ;
    }
    
    /**
     * 現在の接続数を取得.
     * <BR>
     * @return int 現在の接続数が返されます.
     */
    public int size() {
        if( conns != null ) {
            return conns.size() ;
        }
        return -1 ;
    }
    
    /**
     * オブジェクトが動作しているかチェック.
     * <BR>
     * @return boolean [true]の場合、動作しています.
     */
    public boolean isUse() {
        if( conns == null || queue == null || accepts == null ||
            receives == null || pools == null ) {
            return false ;
        }
        return conns.isClosed() == false ;
    }
    
}
