package org.maachang.util;

import java.math.BigDecimal;

/**
 * プリミティブ型の情報を、バイナリ情報にコンバートする為の処理群. <BR>
 * <BR>
 * プリミティブ型の情報を、バイナリ情報にコンバートする為の処理群です.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaachangDao 1.00
 */
public class ConvertParam {

    /**
     * 対象キャラクタセット.
     */
    public static final String CHARSET = "UTF8";

    /**
     * 数値キャラクタ終了コード.
     */
    private static final char ENDCHAR = 0x0000;

    /**
     * boolean : true.
     */
    private static final int BOOLEAN_TRUE_CD = 0x00000001;

    /**
     * boolean : false.
     */
    private static final int BOOLEAN_FALSE_CD = 0x00000000;

    /**
     * データ長 : boolean.
     */
    public static final int LENGTH_BOOLEAN = 1;

    /**
     * データ長 : byte.
     */
    public static final int LENGTH_BYTE = 1;

    /**
     * データ長 : char.
     */
    public static final int LENGTH_CHAR = 2;

    /**
     * データ長 : short.
     */
    public static final int LENGTH_SHORT = 2;

    /**
     * データ長 : int.
     */
    public static final int LENGTH_INT = 4;

    /**
     * データ長 : long.
     */
    public static final int LENGTH_LONG = 8;

    /**
     * コンストラクタ.
     */
    private ConvertParam() {
    }

    /**
     * 2バイト情報を反転. <BR>
     * <BR>
     * 2バイト(short)情報を反転します. <BR>
     * 
     * @param value
     *            反転対象の情報を設定します.
     * @return short 反転された情報が返されます.
     */
    public static final short swap(short value) {
        return (short) (((value & 0x000000ff) << 8) | ((value & 0x0000ff00) >> 8));
    }

    /**
     * 2バイト情報を反転. <BR>
     * <BR>
     * 2バイト(char)情報を反転します. <BR>
     * 
     * @param value
     *            反転対象の情報を設定します.
     * @return char 反転された情報が返されます.
     */
    public static final char swap(char value) {
        return (char) (((value & 0x000000ff) << 8) | ((value & 0x0000ff00) >> 8));
    }

    /**
     * 4バイト情報を反転. <BR>
     * <BR>
     * 4バイト(int)情報を反転します. <BR>
     * 
     * @param value
     *            反転対象の情報を設定します.
     * @return int 反転された情報が返されます.
     */
    public static final int swap(int value) {
        return (int) (((value & 0x000000ff) << 24)
                | ((value & 0x0000ff00) << 8) | ((value & 0x00ff0000) >> 8) | ((value & 0xff000000) >> 24));
    }

    /**
     * 8バイト情報を反転. <BR>
     * <BR>
     * 8バイト(long)情報を反転します. <BR>
     * 
     * @param value
     *            反転対象の情報を設定します.
     * @return long 反転された情報が返されます.
     */
    public static final long swap(long value) {
        return (long) (((value & 0x00000000000000ffL) << 56)
                | ((value & 0x000000000000ff00L) << 40)
                | ((value & 0x0000000000ff0000L) << 24)
                | ((value & 0x00000000ff000000L) << 8)
                | ((value & 0x000000ff00000000L) >> 8)
                | ((value & 0x0000ff0000000000L) >> 24)
                | ((value & 0x00ff000000000000L) >> 40) | ((value & 0xff00000000000000L) >> 56));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * boolean情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertBoolean(boolean value) {

        byte[] ret = null;

        ret = new byte[1];
        ret[0] = (byte) ((value == true) ? BOOLEAN_TRUE_CD : BOOLEAN_FALSE_CD);

        return ret;

    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * boolean情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     */
    public static final void convertBoolean(byte[] out, int offset,
            boolean value) {
        out[offset] = (byte) ((value == true) ? BOOLEAN_TRUE_CD
                : BOOLEAN_FALSE_CD);
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をboolean変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return boolean 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final boolean convertBoolean(int off, byte[] binary)
            throws Exception {
        return ((binary[off] & 0x000000ff) == BOOLEAN_TRUE_CD) ? true : false;
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * byte情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertByte(byte value) {

        byte[] ret = null;

        ret = new byte[1];
        ret[0] = value;

        return ret;

    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * byte情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     */
    public static final void convertByte(byte[] out, int offset, byte value) {
        out[offset] = value;
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をbyte変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return byte 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final byte convertByte(int off, byte[] binary)
            throws Exception {
        return binary[off];
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * char情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertChar(char value) {

        byte[] ret = null;

        ret = new byte[2];
        ret[0] = (byte) (value & 0x000000ff);
        ret[1] = (byte) ((value & 0x0000ff00) >> 8);

        return ret;

    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * char情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     */
    public static final void convertChar(byte[] out, int offset, char value) {
        out[offset] = (byte) (value & 0x000000ff);
        out[offset + 1] = (byte) ((value & 0x0000ff00) >> 8);
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をchar変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return char 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final char convertChar(int off, byte[] binary)
            throws Exception {
        return (char) ((binary[off] & 0x000000ff) | ((binary[off + 1] & 0x000000ff) << 8));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * short情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertShort(short value) {

        byte[] ret = null;

        ret = new byte[2];
        ret[0] = (byte) (value & 0x000000ff);
        ret[1] = (byte) ((value & 0x0000ff00) >> 8);

        return ret;

    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * short情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     */
    public static final void convertShort(byte[] out, int offset, short value) {
        out[offset] = (byte) (value & 0x000000ff);
        out[offset + 1] = (byte) ((value & 0x0000ff00) >> 8);
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をshort変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return short 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final short convertShort(int off, byte[] binary)
            throws Exception {
        return (short) ((binary[off] & 0x000000ff) | ((binary[off + 1] & 0x000000ff) << 8));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * int情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertInt(int value) {

        byte[] ret = null;

        ret = new byte[4];
        ret[0] = (byte) (value & 0x000000ff);
        ret[1] = (byte) ((value & 0x0000ff00) >> 8);
        ret[2] = (byte) ((value & 0x00ff0000) >> 16);
        ret[3] = (byte) ((value & 0xff000000) >> 24);

        return ret;

    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * int情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     */
    public static final void convertInt(byte[] out, int offset, int value) {
        out[offset] = (byte) (value & 0x000000ff);
        out[offset + 1] = (byte) ((value & 0x0000ff00) >> 8);
        out[offset + 2] = (byte) ((value & 0x00ff0000) >> 16);
        out[offset + 3] = (byte) ((value & 0xff000000) >> 24);
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をint変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return int 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final int convertInt(int off, byte[] binary) throws Exception {
        return (int) ((binary[off] & 0x000000ff)
                | ((binary[off + 1] & 0x000000ff) << 8)
                | ((binary[off + 2] & 0x000000ff) << 16) | ((binary[off + 3] & 0x000000ff) << 24));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * long情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertLong(long value) {

        byte[] ret = null;

        ret = new byte[8];
        ret[0] = (byte) (value & 0x00000000000000ffL);
        ret[1] = (byte) ((value & 0x000000000000ff00L) >> 8);
        ret[2] = (byte) ((value & 0x0000000000ff0000L) >> 16);
        ret[3] = (byte) ((value & 0x00000000ff000000L) >> 24);
        ret[4] = (byte) ((value & 0x000000ff00000000L) >> 32);
        ret[5] = (byte) ((value & 0x0000ff0000000000L) >> 40);
        ret[6] = (byte) ((value & 0x00ff000000000000L) >> 48);
        ret[7] = (byte) ((value & 0xff00000000000000L) >> 56);

        return ret;

    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * long情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     */
    public static final void convertLong(byte[] out, int offset, long value) {
        out[offset] = (byte) (value & 0x00000000000000ffL);
        out[offset + 1] = (byte) ((value & 0x000000000000ff00L) >> 8);
        out[offset + 2] = (byte) ((value & 0x0000000000ff0000L) >> 16);
        out[offset + 3] = (byte) ((value & 0x00000000ff000000L) >> 24);
        out[offset + 4] = (byte) ((value & 0x000000ff00000000L) >> 32);
        out[offset + 5] = (byte) ((value & 0x0000ff0000000000L) >> 40);
        out[offset + 6] = (byte) ((value & 0x00ff000000000000L) >> 48);
        out[offset + 7] = (byte) ((value & 0xff00000000000000L) >> 56);
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をlong変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return long 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final long convertLong(int off, byte[] binary)
            throws Exception {
        return (long) ((binary[off] & 0x00000000000000ffL)
                | ((binary[off + 1] & 0x00000000000000ffL) << 8)
                | ((binary[off + 2] & 0x00000000000000ffL) << 16)
                | ((binary[off + 3] & 0x00000000000000ffL) << 24)
                | ((binary[off + 4] & 0x00000000000000ffL) << 32)
                | ((binary[off + 5] & 0x00000000000000ffL) << 40)
                | ((binary[off + 6] & 0x00000000000000ffL) << 48) | ((binary[off + 7] & 0x00000000000000ffL) << 56));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * float情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertFloat(float value) {
        int len;

        byte[] ret = null;
        String str = null;

        str = String.valueOf(value);

        len = str.length();
        len = ((len + 1) / 2) + 2;
        ret = new byte[len];

        ConvertParam.convertFloatStringByBinary(ret, 0, str);

        str = null;

        return ret;
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * float情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertFloat(byte[] out, int offset, float value) {
        return ConvertParam.convertFloatStringByBinary(out, offset, String
                .valueOf(value));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をfloat変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return float 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final float convertFloat(int off, byte[] binary)
            throws Exception {
        return Float.parseFloat(ConvertParam.convertFloatBinaryByString(off,
                binary));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * double情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertDouble(double value) {
        int len;

        byte[] ret = null;
        String str = null;

        str = String.valueOf(value);

        len = str.length();
        len = ((len + 1) / 2) + 2;
        ret = new byte[len];

        ConvertParam.convertFloatStringByBinary(ret, 0, str);

        str = null;

        return ret;
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * double情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertDouble(byte[] out, int offset, double value) {
        return ConvertParam.convertFloatStringByBinary(out, offset, String
                .valueOf(value));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をdouble変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return double 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final double convertDouble(int off, byte[] binary)
            throws Exception {
        return Double.parseDouble(ConvertParam.convertFloatBinaryByString(off,
                binary));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * Decimal情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertDecimal(BigDecimal value) {
        int len;

        byte[] ret = null;
        String str = null;

        str = value.toString();

        len = str.length();
        len = ((len + 1) / 2) + 2;
        ret = new byte[len];

        ConvertParam.convertFloatStringByBinary(ret, 0, str);

        str = null;

        return ret;
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * Decimal情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     * @return int 変換された情報長が返されます.
     */
    public static final int convertDecimal(byte[] out, int offset,
            BigDecimal value) {
        return ConvertParam.convertFloatStringByBinary(out, offset, value
                .toString());
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をBigDecimal変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return BigDecimal 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final BigDecimal convertDecimal(int off, byte[] binary)
            throws Exception {
        return new BigDecimal(ConvertParam.convertFloatBinaryByString(off,
                binary));
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * char配列情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     */
    public static final byte[] convertChars(char[] value) {
        int i, j;
        int len;

        byte[] ret = null;

        len = value.length;
        ret = new byte[len * 2];

        for (i = 0, j = 0; i < len; i++, j += 2) {

            ret[j] = (byte) (value[i] & 0x000000ff);
            ret[j + 1] = (byte) ((value[i] & 0x0000ff00) >> 8);

        }

        return ret;
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * char配列情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     */
    public static final void convertChars(byte[] out, int offset, char[] value) {
        int i, j;
        int len;

        len = value.length;
        for (i = 0, j = offset; i < len; i++, j += 2) {

            out[j] = (byte) (value[i] & 0x000000ff);
            out[j + 1] = (byte) ((value[i] & 0x0000ff00) >> 8);

        }
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をchar配列変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param length
     *            変換対象バイナリの変換データ長を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return char[] 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final char[] convertChars(int off, int length, byte[] binary)
            throws Exception {
        int i, j;
        int len;

        char[] ret = null;

        if (binary == null || length <= 0 || (length & 0x00000001) == 1) {
            if ((length & 0x00000001) == 1) {
                throw new IllegalArgumentException("length長は不正です");
            }
            throw new IllegalArgumentException("引数は不正です");
        }

        ret = new char[length / 2];
        len = length + off;

        for (i = off, j = 0; i < len; i += 2, j++) {

            ret[j] = (char) ((binary[i] & 0x000000ff) | ((binary[i + 1] & 0x000000ff) << 8));

        }
        return ret;

    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * String情報をバイナリ変換します. <BR>
     * 
     * @param value
     *            変換対象の処理を設定します.
     * @return byte[] 変換されたバイナリ情報が返されます.
     * @exception Exception
     *                例外.
     */
    public static final byte[] convertString(String value) throws Exception {
        return value.getBytes(ConvertParam.CHARSET);
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * String情報をバイナリ変換します. <BR>
     * 
     * @param out
     *            設定対象のバイナリ情報を設定します.
     * @param offset
     *            設定対象のオフセット位置を設定します.
     * @param value
     *            変換対象の処理を設定します.
     * @return int 変換されたバイナリ長が返されます.
     * @exception Exception
     *                例外.
     */
    public static final int convertString(byte[] out, int offset, String value)
            throws Exception {

        int ret;
        byte[] tmp = null;

        tmp = value.getBytes(ConvertParam.CHARSET);
        ret = tmp.length;
        System.arraycopy(tmp, 0, out, offset, ret);
        return ret;
    }

    /**
     * コンバート処理. <BR>
     * <BR>
     * 対象のバイナリ情報をString変換します. <BR>
     * 
     * @param off
     *            変換対象バイナリのオフセット値を設定します.
     * @param length
     *            変換対象バイナリの変換データ長を設定します.
     * @param binary
     *            変換対象のバイナリ情報を設定します.
     * @return String 変換された情報が返されます.
     * @exception Exception
     *                入力例外.
     */
    public static final String convertString(int off, int length, byte[] binary)
            throws Exception {
        byte[] bin = null;
        String ret = null;

        if (binary == null || length <= 0) {
            throw new IllegalArgumentException("引数は不正です");
        }

        bin = new byte[length];
        System.arraycopy(binary, off, bin, 0, length);

        ret = new String(bin, ConvertParam.CHARSET);

        return ret;

    }

    /**
     * 小数点情報をバイナリに変換.
     */
    private static final int convertFloatStringByBinary(byte[] out, int off,
            String data) {
        int i, j;
        int len;
        byte dat;
        int ret;

        len = data.length();
        ret = (len + 1) / 2;

        out[off] = (byte) (ret & 0x000000ff);
        out[off + 1] = (byte) ((ret & 0x0000ff00) >> 8);

        for (i = 0, j = off + 2; i < len; i++) {

            dat = ConvertParam.cnvCharByNumBin(data.charAt(i));

            if ((i & 0x00000001) == 0) {
                out[j] = (byte) (dat & 0x0000000f);
            } else {
                out[j] |= (byte) ((dat & 0x0000000f) << 4);
                j++;
            }

        }

        return ret;

    }

    /**
     * 小数点バイナリ情報を文字列に変換.
     */
    public static final String convertFloatBinaryByString(int off, byte[] binary)
            throws Exception {

        int i, j;
        int len;

        char dat;
        StringBuffer buf = null;
        String ret = null;

        if (binary == null || binary.length <= 0) {
            throw new IllegalArgumentException("引数は不正です");
        }

        buf = new StringBuffer();

        len = (int) ((binary[off] & 0x000000ff) | ((binary[off + 1] & 0x000000ff) << 8));

        for (i = 0, j = off + 2; i < len; i++, j++) {

            dat = ConvertParam.cnvNumBinByChar((byte) (binary[j] & 0x0000000f));
            if (dat == ConvertParam.ENDCHAR) {
                break;
            }

            buf.append(dat);

            dat = ConvertParam
                    .cnvNumBinByChar((byte) ((binary[j] & 0x000000f0) >> 4));
            if (dat == ConvertParam.ENDCHAR) {
                break;
            }

            buf.append(dat);

        }

        ret = buf.toString();
        return ret;

    }

    /**
     * 数値変換 : 文字コードから数値バイナリ変換コードに変換.
     */
    private static final byte cnvCharByNumBin(char cd) {

        byte ret;

        switch (cd) {

        case '0':
            ret = 0x01;
            break;
        case '1':
            ret = 0x02;
            break;
        case '2':
            ret = 0x03;
            break;
        case '3':
            ret = 0x04;
            break;
        case '4':
            ret = 0x05;
            break;
        case '5':
            ret = 0x06;
            break;
        case '6':
            ret = 0x07;
            break;
        case '7':
            ret = 0x08;
            break;
        case '8':
            ret = 0x09;
            break;
        case '9':
            ret = 0x0a;
            break;
        case '+':
            ret = 0x0b;
            break;
        case '-':
            ret = 0x0c;
            break;
        case '.':
            ret = 0x0d;
            break;

        default:
            ret = 0x0f;
            break;
        }

        return ret;

    }

    /**
     * 数値変換 : 数値バイナリ変換コードから、文字コードに変換.
     */
    private static final char cnvNumBinByChar(byte cd) {

        char ret;

        switch (cd) {

        case 0x00:
            ret = ConvertParam.ENDCHAR;
            break;
        case 0x01:
            ret = '0';
            break;
        case 0x02:
            ret = '1';
            break;
        case 0x03:
            ret = '2';
            break;
        case 0x04:
            ret = '3';
            break;
        case 0x05:
            ret = '4';
            break;
        case 0x06:
            ret = '5';
            break;
        case 0x07:
            ret = '6';
            break;
        case 0x08:
            ret = '7';
            break;
        case 0x09:
            ret = '8';
            break;
        case 0x0a:
            ret = '9';
            break;
        case 0x0b:
            ret = '+';
            break;
        case 0x0c:
            ret = '-';
            break;
        case 0x0d:
            ret = '.';
            break;

        default:
            ret = '?';
            break;

        }

        return ret;

    }

}
