package org.maachang.comet.httpd.engine.auth ;

import java.util.HashSet;

/**
 * HTTPD-認証用要素.
 *
 * @version 2007/08/19
 * @author  masahito suzuki
 * @since   MaachangComet 1.00
 */
public class HttpdAuthElement {
    
    /**
     * 認証タイプ : basic.
     */
    public static final int AUTH_BY_BASIC = 0 ;
    
    /**
     * 認証タイプ : digest.
     */
    public static final int AUTH_BY_DIGEST = 1 ;
    
    /**
     * realm名.
     */
    private String realm = null ;
    
    /**
     * 認証タイプ.
     */
    private int authType = -1 ;
    
    /**
     * 読み込み対象ディレクトリ名.
     */
    private String targetDirectory = null ;
    
    /**
     * 読み込み時のファイルアップロード時間.
     */
    private long fileTime = -1L ;
    
    /**
     * 認証ユーザ群.
     */
    private HashSet<String> users = null ;
    
    /**
     * コンストラクタ.
     */
    public HttpdAuthElement() {
        
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.clear() ;
        this.users = null ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param realm 対象のrealm名を設定します.
     * @parma authType 対象の認証タイプを設定します.
     * @param targetDirectory 対象のディレクトリ名を設定します.
     * @param fileTime 対象のファイル時間を設定します.
     * @exception Exception 例外.
     */
    public synchronized void create( String realm,int authType,String targetDirectory,long fileTime )
        throws Exception {
        if( realm == null || ( realm = realm.trim() ).length() <= 0 ||
            targetDirectory == null || ( targetDirectory = targetDirectory.trim() ).length() <= 0 ||
            ( authType != AUTH_BY_BASIC && authType != AUTH_BY_DIGEST ) ) {
            throw new IllegalArgumentException( "引数は不正です[realm:" +
                realm + " authType:" + authType + " targetDirectory:" + targetDirectory + "]" ) ;
        }
        this.realm = realm ;
        this.authType = authType ;
        this.targetDirectory = targetDirectory ;
        this.fileTime = fileTime ;
        this.users = new HashSet<String>() ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public synchronized void clear() {
        this.realm = null ;
        this.authType = -1 ;
        this.targetDirectory = null ;
        this.fileTime = -1L ;
        this.users.clear() ;
    }
    
    /**
     * ユーザ名を追加.
     * <BR><BR>
     * 対象のユーザ名を追加します.
     * <BR>
     * @param user 対象のユーザ名を設定します.
     */
    public synchronized void addUser( String user ) {
        if( user == null || ( user = user.trim() ).length() <= 0 ) {
            return ;
        }
        users.add( user ) ;
    }
    
    /**
     * realm名を取得.
     * <BR><BR>
     * 設定されているrealm名を取得します.
     * <BR>
     * @return String 設定されているrealm名が返されます.
     */
    public synchronized String getRealm() {
        return realm ;
    }
    
    /**
     * 認証タイプを取得.
     * <BR><BR>
     * 認証タイプを取得します.
     * <BR>
     * @return int 認証タイプが返されます.
     */
    public synchronized int getAuthType() {
        return authType ;
    }
    
    /**
     * 認証タイプを文字で取得.
     * <BR><BR>
     * 認証タイプを文字で取得します.
     * <BR>
     * @return String 認証タイプが返されます.
     */
    public synchronized String getAuthTypeByString() {
        switch( authType ) {
            case AUTH_BY_BASIC :
                return "BASIC" ;
            case AUTH_BY_DIGEST :
                return "DIGEST" ;
        }
        return "UNKNOWN" ;
    }
    
    /**
     * ディレクトリ名を取得.
     * <BR><BR>
     * ディレクトリ名を取得します.
     * <BR>
     * @return String ディレクトリ名が返されます.
     */
    public synchronized String getTargetDirectory() {
        return targetDirectory ;
    }
    
    /**
     * 読み込み時のファイル時刻を取得.
     * <BR><BR>
     * 読み込み時のファイル時刻を取得します.
     * <BR>
     * @return long 読み込み時のファイル時刻が返されます.
     */
    public synchronized long getFileTime() {
        return fileTime ;
    }
    
    /**
     * ユーザ名一覧オブジェクトを取得.
     * <BR><BR>
     * ユーザ名一覧オブジェクトを取得します.
     * <BR>
     * @return HashSet<String> ユーザ名一覧オブジェクトが返されます.
     */
    public synchronized HashSet<String> getUsers() {
        return users ;
    }
    
    /**
     * 文字列に変換.
     */
    public synchronized String toString() {
        StringBuilder buf = new StringBuilder() ;
        buf.append( "realm:" ).append( getRealm() ).
            append( " auth_type:" ).append( getAuthTypeByString() ) ;
        buf.append( " users:" ) ;
        if( users != null && users.size() > 0 ) {
            Object[] objs = users.toArray() ;
            int len = objs.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( i != 0 ) {
                    buf.append( "," ) ;
                }
                buf.append( ( String )objs[ i ] ) ;
            }
        }
        return buf.toString() ;
    }
}
