package org.maachang.dao.dbms;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.maachang.conf.Config;
import org.maachang.conf.ReadIni;
import org.maachang.dao.dbms.kind.KindManager;
import org.maachang.dao.dbms.kind.SupportKind;

/**
 * DB情報を取得.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaachangDao 1.00
 */
public class DbmsInfo {
    
    /**
     * セクション名.
     */
    private static final String DBMS_SECTION = "dbms" ;
    
    /**
     * レコードファクトリを取得.
     * <BR><BR>
     * 指定条件からレコードファクトリを取得します.
     * <BR>
     * @param conf 対象のコンフィグ情報を設定します.
     * @return RecordFactory 対象のレコードファクトリが返されます.
     * @exception Exception 例外.
     */
    public static final RecordFactory getRecordFactory( Config conf )
        throws Exception {
        // ＤＢアクセス.
        String adapter = conf.get(DBMS_SECTION,"adapter",0);
        String driver = conf.get(DBMS_SECTION,"driver",0);
        String url = conf.get(DBMS_SECTION,"url",0);
        String user = conf.get(DBMS_SECTION,"user",0);
        String passwd = conf.get(DBMS_SECTION,"passwd",0);
        String max = conf.get(DBMS_SECTION,"max",0);
        String autoCommit = conf.get(DBMS_SECTION,"autoCommit",0);
        String checkSQL = conf.get(DBMS_SECTION,"checksql",0);
        String debug = conf.get(DBMS_SECTION,"debug",0);
        
        int maxInt;
        try {
            maxInt = Integer.parseInt(max);
        } catch (Exception e) {
            maxInt = 15;
        }
        if (maxInt <= 0) {
            maxInt = 15;
        }
        boolean autoCommitFlag = false ;
        try {
            autoCommit = autoCommit.trim().toLowerCase() ;
            autoCommitFlag = autoCommit.equals( "true" ) ;
        } catch( Exception e ) {
            autoCommitFlag = false ;
        }
        //System.out.println( "autoCommit:" + autoCommit + "/" + autoCommitFlag ) ;
        boolean debugFlag = false ;
        try {
            debug = debug.trim().toLowerCase() ;
            debugFlag = debug.equals( "true" ) ;
        } catch( Exception e ) {
            debugFlag = false ;
        }
        return getRecordFactory( debugFlag, adapter, driver, url, user, passwd, maxInt, autoCommitFlag, checkSQL ) ;
    }
    
    /**
     * レコードファクトリを生成.
     * <BR><BR>
     * 指定条件から、レコードファクトリ情報を生成します.
     * <BR>
     * @param debug 対象のデバッグモードを設定します.
     * @param adapter 対象のアダプタ名を設定します.
     * @para driver 対象のドライバー名を設定します.
     * @param url 対象のURLを設定します.
     * @param user 対象のユーザ名を設定します.
     * @param passwd 対象のパスワードを設定します.
     * @param max 対象の最大コネクション数を設定します.
     * @param autoCommit 対象のオートコミット条件を設定します.
     * @param checkSQL 対象のコネクション確認用SQL文を設定します.
     * @return RecordFactory 対象のレコードファクトリが返されます.
     * @exception Exception 例外.
     */
    public static final RecordFactory getRecordFactory(
        boolean debug,String adapter,String driver,String url,String user,String passwd,
        int max,boolean autoCommit,String checkSQL ) throws Exception {
        SupportKind kind = KindManager.get( adapter ) ;
        if( kind == null ) {
            throw new IllegalArgumentException("非サポートのアダプタ名("+ adapter +
                ")が設定されています");
        }
        RecordFactory ret = new RecordFactory();
        ret.create(kind,debug,driver, url, user, passwd, max, autoCommit, checkSQL);
        return ret ;
    }
    
    /**
     * 指定されたコンフィグファイルを取得.
     * <BR><BR>
     * 指定されたコンフィグファイルを取得します.
     * <BR>
     * @param name 対象のコンフィグファイル名を設定します.
     * @return Config 対象のコンフィグファイルが返されます.
     * @exception Exception 例外.
     */
    public static final Config getConfig( String name )
        throws Exception {
        return getConfig( new FileInputStream( name ) ) ;
    }
    
    /**
     * 指定されたコンフィグファイルを取得.
     * <BR><BR>
     * 指定されたコンフィグファイルを取得します.
     * <BR>
     * @param in 対象のInputStreamを設定します.
     * @return Config 対象のコンフィグファイルが返されます.
     * @exception Exception 例外.
     */
    public static final Config getConfig( InputStream in )
        throws Exception {
        BufferedReader buf = null ;
        try {
            Config ret = new Config() ;
            buf = new BufferedReader( new InputStreamReader( in,"UTF8" ) ) ;
            ReadIni.analisys( ret,buf ) ;
            buf.close() ;
            buf = null ;
            return ret ;
        } finally {
            if( buf != null ) {
                try {
                    buf.close() ;
                } catch( Exception e ) {
                }
            }
            buf = null ;
        }
    }
}

