package org.maachang.dbm.service.client ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;

import org.maachang.dbm.service.ProtocolDef;
import org.maachang.util.ConvertParam;

/**
 * MaachangDbm操作用オブジェクト.
 * 
 * @version 2008/01/20
 * @author masahito suzuki
 * @since MaachangDBM 1.03
 */
class ClientSession {
    
    private Socket socket = null ;
    private InputStream input = null ;
    private OutputStream output = null ;
    private ResultClientQueue queue = null ;
    private boolean receiveFlag = false ;
    private final Object sync = new Object() ;
    private Object driverSync = null ;
    
    public ClientSession( Object driverSync,InetAddress addr,int port )
        throws Exception {
        if( addr == null ) {
            addr = InetAddress.getByName( "127.0.0.1" ) ;
        }
        if( port < 0 || port > 65535 ) {
            port = ProtocolDef.BIND_PORT ;
        }
        this.driverSync = driverSync ;
        this.socket = createSocket( addr,port ) ;
        this.input = new BufferedInputStream( socket.getInputStream() ) ;
        this.output = new BufferedOutputStream( socket.getOutputStream() ) ;
        this.queue = new ResultClientQueue() ;
    }
    
    private static final int RECV_TIMEOUT = 5000 ;
    private static final int CONN_TIMEOUT = 2500 ;
    private static final int BASE_BUFFER = 262144 ;
    
    private static final Socket createSocket( InetAddress addr,int port )
        throws Exception {
        Socket ret = new Socket() ;
        ret.setSendBufferSize( BASE_BUFFER ) ;
        ret.setReceiveBufferSize( BASE_BUFFER ) ;
        ret.setSoLinger( false,0 ) ;
        ret.setKeepAlive( true ) ;
        ret.setTcpNoDelay( true ) ;
        ret.setReuseAddress( true ) ;
        ret.connect( new InetSocketAddress( addr,port ),CONN_TIMEOUT ) ;
        ret.setSoTimeout( RECV_TIMEOUT ) ;
        return ret ;
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    public void destroy() {
        synchronized( sync ) {
            if( input != null ) {
                try {
                    input.close() ;
                } catch( Exception e ) {
                }
            }
            if( output != null ) {
                try {
                    output.close() ;
                } catch( Exception e ) {
                }
            }
            if( socket != null ) {
                try {
                    socket.close() ;
                } catch( Exception e ) {
                }
            }
            queue = null ;
            socket = null ;
            input = null ;
            output = null ;
            receiveFlag = false ;
        }
    }
    
    public InputStream input() {
        InputStream ret = null ;
        synchronized( sync ) {
            ret = input ;
        }
        return ret ;
    }
    
    public OutputStream output() {
        OutputStream ret = null ;
        synchronized( sync ) {
            ret = output ;
        }
        return ret ;
    }
    
    public Socket socket() {
        Socket ret = null ;
        synchronized( sync ) {
            ret = socket ;
        }
        return ret ;
    }
    
    public ResultClientQueue getQueue() {
        ResultClientQueue ret = null ;
        synchronized( sync ) {
            ret = queue ;
        }
        return ret ;
    }
    
    public boolean isUse() {
        boolean ret = true ;
        synchronized( sync ) {
            try {
                if( socket == null || input == null ||
                    output == null || queue == null ||
                    queue.isUse() == false ||
                    socket.isBound() == false ||
                    socket.isClosed() == true ) {
                    ret = false ;
                }
            } catch( Exception e ) {
                ret = false ;
            }
        }
        return ret ;
    }
    
    public boolean isReceiveMethod() throws Exception {
        boolean ret = false ;
        synchronized( sync ) {
            ret = this.receiveFlag ;
        }
        return ret ;
    }
    
    public boolean isReceive() throws Exception {
        boolean ret = false ;
        synchronized( sync ) {
            if( isUse() == false ) {
                throw new IOException( "既にクローズされています" ) ;
            }
            if( this.input.available() > 0 ) {
                ret = true ;
            }
        }
        return ret ;
    }
    
    public void startReceive() {
        synchronized( sync ) {
            this.receiveFlag = true ;
        }
    }
    
    public void exitReceive() {
        synchronized( sync ) {
            this.receiveFlag = false ;
        }
    }
    
    public byte[] receive() {
        ByteArrayOutputStream bs = null ;
        try {
            startReceive() ;
            bs = new ByteArrayOutputStream() ;
            if( isUse() == false ) {
                this.destroy() ;
                return null ;
            }
            int len = -1 ;
            byte[] ret = null ;
            InputStream in = null ;
            synchronized( sync ) {
                in = input ;
            }
            int cnt = 0 ;
            for( ;; ) {
                int n = in.read() ;
                if( n <= -1 ) {
                    break ;
                }
                bs.write( n ) ;
                cnt ++ ;
                if( len <= -1 ) {
                    if( cnt >= ProtocolDef.OFFSET ) {
                        byte[] bin = bs.toByteArray() ;
                        len = ConvertParam.convertInt( 0,bin ) ;
                        if( len <= 0 ) {
                            sendHeartBeat() ;
                            return null ;
                        }
                    }
                }
                if( len > 0 ) {
                    if( cnt >= len ) {
                        ret = bs.toByteArray() ;
                        break ;
                    }
                }
            }
            bs.close() ;
            bs = null ;
            return ret ;
        } catch( SocketTimeoutException st ) {
        } catch( Exception e ) {
            this.destroy() ;
        } finally {
            if( bs != null ) {
                try {
                    bs.close() ;
                } catch( Exception e ) {
                }
            }
            bs = null ;
        }
        return null ;
    }
    
    public Object getDriverSync() {
        return driverSync ;
    } 
    
    private void sendHeartBeat() throws Exception {
        if( output != null ) {
            synchronized( output ) {
                output.write( ConvertParam.convertInt( 0 ) ) ;
                output.flush() ;
            }
        }
    }
    
}
