package org.maachang.dbm.service.client ;

import java.io.IOException;
import java.io.OutputStream;
import java.net.InetAddress;
import java.util.ArrayList;

import org.maachang.dbm.engine.MDbmEngine;
import org.maachang.dbm.service.ProtocolDef;
import org.maachang.util.ConvertParam;

/**
 * MaachangDbmクライアントコネクション
 * 
 * @version 2008/01/20
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
class MDbmClientConnection {
    
    private static final String NOT_ACCESS = "コネクション接続が確立していません" ;
    private static final long TIMEOUT = 15000L ;
    private ClientSession session = null ;
    
    private MDbmClientConnection() {
        
    }
    
    public MDbmClientConnection( Object sync,InetAddress addr,int port )
        throws Exception {
        this.session = new ClientSession( sync,addr,port ) ;
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    public void destroy() {
        try {
            close() ;
        } catch( Exception e ) {
        }
    }
    
    public void close() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_CLOSE ) ) ;
            output.flush() ;
        }
        if( session != null ) {
            session.destroy() ;
        }
    }
    
    public ClientSession getSession() {
        return session ;
    }
    
    public void commit() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_COMMIT ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_SUCCESS ) {
                throw new IOException( "[commit]結果プロトコルは不正です" ) ;
            }
        }
    }
    
    public void rollback() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_ROLLBACK ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_SUCCESS ) {
                throw new IOException( "[rollback]結果プロトコルは不正です" ) ;
            }
        }
    }
    
    public void check() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_TRANSACTOIN ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_SUCCESS ) {
                throw new IOException( "[check]結果プロトコルは不正です" ) ;
            }
        }
    }
    
    public void put( byte[] key,byte[] value ) throws Exception {
        if( key == null || key.length <= 0 || key.length > MDbmEngine.MAX_KEY_LENGTH ) {
            if( key == null || key.length <= 0 ) {
                throw new IllegalArgumentException( "指定キーは不正です" ) ;
            }
            throw new IllegalArgumentException( "指定キーは最大長["+MDbmEngine.MAX_KEY_LENGTH+"を越しています" ) ;
        }
        if( value == null || value.length <= 0 ) {
            throw new IllegalArgumentException( "指定要素は不正です" ) ;
        }
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET + 8 + key.length + value.length ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_PUT ) ) ;
            output.write( ConvertParam.convertInt( key.length ) ) ;
            output.write( key ) ;
            output.write( ConvertParam.convertInt( value.length ) ) ;
            output.write( value ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_SUCCESS ) {
                throw new IOException( "[put]結果プロトコルは不正です" ) ;
            }
        }
    }
    
    public void remove( byte[] key ) throws Exception {
        if( key == null || key.length <= 0 || key.length > MDbmEngine.MAX_KEY_LENGTH ) {
            if( key == null || key.length <= 0 ) {
                throw new IllegalArgumentException( "指定キーは不正です" ) ;
            }
            throw new IllegalArgumentException( "指定キーは最大長["+MDbmEngine.MAX_KEY_LENGTH+"を越しています" ) ;
        }
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET + 4 + key.length ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_REMOVE ) ) ;
            output.write( ConvertParam.convertInt( key.length ) ) ;
            output.write( key ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_SUCCESS ) {
                throw new IOException( "[remove]結果プロトコルは不正です" ) ;
            }
        }
    }
    
    public byte[] get( byte[] key ) throws Exception {
        if( key == null || key.length <= 0 || key.length > MDbmEngine.MAX_KEY_LENGTH ) {
            if( key == null || key.length <= 0 ) {
                throw new IllegalArgumentException( "指定キーは不正です" ) ;
            }
            throw new IllegalArgumentException( "指定キーは最大長["+MDbmEngine.MAX_KEY_LENGTH+"を越しています" ) ;
        }
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET + 4 + key.length ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_GET ) ) ;
            output.write( ConvertParam.convertInt( key.length ) ) ;
            output.write( key ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_DATA ) {
                throw new IOException( "[get]結果プロトコルは不正です" ) ;
            }
            return dataResult( bin ) ;
        }
    }
    
    public boolean containsKey( byte[] key) throws Exception {
        if( key == null || key.length <= 0 || key.length > MDbmEngine.MAX_KEY_LENGTH ) {
            if( key == null || key.length <= 0 ) {
                throw new IllegalArgumentException( "指定キーは不正です" ) ;
            }
            throw new IllegalArgumentException( "指定キーは最大長["+MDbmEngine.MAX_KEY_LENGTH+"を越しています" ) ;
        }
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET + 4 + key.length ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_CONTAINS ) ) ;
            output.write( ConvertParam.convertInt( key.length ) ) ;
            output.write( key ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_BOOL ) {
                throw new IOException( "[containsKey]結果プロトコルは不正です" ) ;
            }
            return booleanResult( bin ) ;
        }
    }
    
    public int size() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_SIZE ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_SIZE ) {
                throw new IOException( "[size]結果プロトコルは不正です" ) ;
            }
            return sizeResult( bin ) ;
        }
    }
    
    public String getDirectory() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_DIRECTORY ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_DATA ) {
                throw new IOException( "[getDirectory]結果プロトコルは不正です" ) ;
            }
            byte[] res = dataResult( bin ) ;
            if( res == null ) {
                return null ;
            }
            return new String( res,"UTF8" ) ;
        }
    }
    
    public void initKey() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_INIT_KEY ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_SUCCESS ) {
                throw new IOException( "[initKey]結果プロトコルは不正です" ) ;
            }
        }
    }
    
    public boolean hasKey() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_HAS_KEY ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_BOOL ) {
                throw new IOException( "[hasKey]結果プロトコルは不正です" ) ;
            }
            return booleanResult( bin ) ;
        }
    }
    
    public ArrayList<byte[]> getKey() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_GET_KEY ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_KEY ) {
                throw new IOException( "[getKey]結果プロトコルは不正です" ) ;
            }
            return dataResultKey( bin ) ;
        }
    }
    
    public long sequenceId( int no ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        OutputStream output = session.output() ;
        synchronized( output ) {
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_OFFSET + 4 ) ) ;
            output.write( ConvertParam.convertInt( ProtocolDef.SEND_SEQUENCE_ID ) ) ;
            output.write( ConvertParam.convertInt( no ) ) ;
            output.flush() ;
            byte[] bin = receive() ;
            if( resultType( bin ) != ProtocolDef.RESULT_ID ) {
                throw new IOException( "[sequenceId]結果プロトコルは不正です" ) ;
            }
            return idResult( bin ) ;
        }
    }
    
    public boolean isUse() {
        boolean ret = true ;
        try {
            if( session == null || session.isUse() == false ) {
                ret = false ;
            }
        } catch( Exception e ) {
            ret = false ;
        }
        return ret ;
    }
    
    private static final int getType( byte[] bin )
        throws Exception {
        if( bin == null || bin.length < ProtocolDef.SEND_OFFSET ) {
            throw new IOException( "不正なプロトコルです" ) ;
        }
        return ConvertParam.convertInt( ProtocolDef.OFFSET,bin ) ;
    }
    
    private static final int resultType( byte[] bin ) throws Exception {
        int type = getType( bin ) ;
        switch( type ) {
            case ProtocolDef.RESULT_SUCCESS : return ProtocolDef.RESULT_SUCCESS ;
            case ProtocolDef.RESULT_ERROR : throw new IOException( errorResult( bin ) ) ;
            case ProtocolDef.RESULT_BOOL : return ProtocolDef.RESULT_BOOL ;
            case ProtocolDef.RESULT_DATA : return ProtocolDef.RESULT_DATA ;
            case ProtocolDef.RESULT_KEY : return ProtocolDef.RESULT_KEY ;
            case ProtocolDef.RESULT_SIZE : return ProtocolDef.RESULT_SIZE ;
            case ProtocolDef.RESULT_ID : return ProtocolDef.RESULT_ID ;
        }
        throw new IOException( "不正な条件("+type+")が返されました" ) ;
    }
    
    private static final String errorResult( byte[] bin ) throws Exception {
        int len = ConvertParam.convertInt( ProtocolDef.SEND_OFFSET,bin ) ;
        int all = ProtocolDef.SEND_OFFSET + 4 + len ;
        if( all != bin.length ) {
            throw new IOException( "[error]プロトコル内容は不正です("+
                all+"/"+bin.length+")" ) ;
        }
        byte[] b = new byte[ len ] ;
        System.arraycopy( bin,ProtocolDef.SEND_OFFSET+4,b,0,len ) ;
        return new String( b,"UTF8" ) ;
    }
    
    private static final boolean booleanResult( byte[] bin ) throws Exception {
        int all = ProtocolDef.SEND_OFFSET + 1 ;
        if( all != bin.length ) {
            throw new IOException( "[boolean]プロトコル内容は不正です("+
                all+"/"+bin.length+")" ) ;
        }
        return ConvertParam.convertBoolean( ProtocolDef.SEND_OFFSET,bin ) ;
    }
    
    private static final int sizeResult( byte[] bin ) throws Exception {
        int all = ProtocolDef.SEND_OFFSET + 4 ;
        if( all != bin.length ) {
            throw new IOException( "[int]プロトコル内容は不正です("+
                all+"/"+bin.length+")" ) ;
        }
        return ConvertParam.convertInt( ProtocolDef.SEND_OFFSET,bin ) ;
    }
    
    private static final long idResult( byte[] bin ) throws Exception {
        int all = ProtocolDef.SEND_OFFSET + 8 ;
        if( all != bin.length ) {
            throw new IOException( "[long]プロトコル内容は不正です("+
                all+"/"+bin.length+")" ) ;
        }
        return ConvertParam.convertLong( ProtocolDef.SEND_OFFSET,bin ) ;
    }
    
    private static final byte[] dataResult( byte[] bin ) throws Exception {
        int len = ConvertParam.convertInt( ProtocolDef.SEND_OFFSET,bin ) ;
        int all = ProtocolDef.SEND_OFFSET + 4 + len ;
        if( all != bin.length ) {
            throw new IOException( "[data]プロトコル内容は不正です("+
                all+"/"+bin.length+")" ) ;
        }
        if( len <= 0 ) {
            return null ;
        }
        byte[] ret = new byte[ len ] ;
        System.arraycopy( bin,ProtocolDef.SEND_OFFSET+4,ret,0,len ) ;
        return ret ;
    }
    
    private static final ArrayList<byte[]> dataResultKey( byte[] bin ) throws Exception {
        int len = ConvertParam.convertInt( ProtocolDef.SEND_OFFSET,bin ) ;
        int pos = ProtocolDef.SEND_OFFSET + 4 ;
        ArrayList<byte[]> ret = new ArrayList<byte[]>() ;
        for( int i = 0 ; i < len ; i ++ ) {
            int bLen = ConvertParam.convertInt( pos,bin ) ;
            pos += 4 ;
            byte[] b = new byte[ bLen ] ;
            System.arraycopy( bin,pos,b,0,bLen ) ;
            pos += bLen ;
            ret.add( b ) ;
        }
        return ret ;
    }
    
    private byte[] receive() throws Exception {
        if( isUse() == false ) {
            throw new IOException( NOT_ACCESS ) ;
        }
        long time = System.currentTimeMillis() + TIMEOUT ;
        ResultClientQueue queue = session.getQueue() ;
        byte[] ret = null ;
        for( ;; ) {
            ret = queue.getQueue() ;
            if( ret != null ) {
                break ;
            }
            // キューに情報が存在しない場合は、直接受信処理を実施.
            Object driverSync = session.getDriverSync() ;
            boolean rcvFlag = false ;
            try {
                synchronized( driverSync ) {
                    if( isUse() == false ) {
                        destroy() ;
                        return null ;
                    }
                    rcvFlag = ( session.isReceiveMethod() == false && session.isReceive() == true ) ;
                    if( rcvFlag == true ) {
                        // 受信処理開始.
                        session.startReceive() ;
                    }
                }
                if( rcvFlag == true ) {
                    ret = session.receive() ;
                    if( ret != null ) {
                        break ;
                    }
                }
            } finally {
                if( rcvFlag == true ) {
                    // 受信処理終了.
                    try {
                        session.exitReceive() ;
                    } catch( Exception e ) {
                    }
                }
            }
            if( isUse() == false ) {
                throw new IOException( NOT_ACCESS ) ;
            }
            if( time <= System.currentTimeMillis() ) {
                throw new IOException( "受信タイムアウト" ) ;
            }
            Thread.sleep( 5L ) ;
        }
        return ret ;
    }
}
