package org.maachang.dao ;

import java.util.HashMap;

import org.maachang.dao.dbms.DbUtil;
import org.maachang.dao.dbms.MetaColumn;
import org.maachang.dao.dbms.Record;

/**
 * DaoSession.
 *
 * @version 2007/11/03
 * @author  masahito suzuki
 * @since MaachangDao 1.00
 */
public class DaoSession {
    
    /**
     * テーブル情報.
     */
    private HashMap<String,MetaColumn> tables = null ;
    
    /**
     * コネクション情報.
     */
    private Record record = null ;
    
    /**
     * コンストラクタ.
     */
    private DaoSession() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 指定条件を設定してオブジェクトを生成します.
     * <BR>
     * @param record 対象のレコードオブジェクトを設定します.
     * @exception MaachangDaoException MaachangDao例外.
     */
    public DaoSession( Record record ) throws MaachangDaoException {
        if( record == null ) {
            throw new MaachangDaoException( "引数は不正です" ) ;
        }
        this.record = record ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     s* 指定条件を設定してオブジェクトを生成します.
     * <BR>
     * @param record 対象のレコードオブジェクトを設定します.
     * @param name 対象のモデル名を設定します.
     * @param meta 対象のメタカラムオブジェクトを設定します.
     * @exception MaachangDaoException MaachangDao例外.
     */
    public DaoSession( Record record,String name,MetaColumn meta ) throws MaachangDaoException {
        if( record == null || name == null || meta == null || ( name = name.trim() ).length() <= 0 ) {
            throw new MaachangDaoException( "引数は不正です" ) ;
        }
        this.record = record ;
        this.tables = new HashMap<String,MetaColumn>() ;
        this.tables.put( name,meta ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        if( record != null ) {
            try {
                record.commit() ;
            } catch( Exception e ) {
            }
            try {
                record.close() ;
            } catch( Exception e ) {
            }
        }
        this.record = null ;
        this.tables = null ;
    }
    
    /**
     * オブジェクトクリア.
     * <BR><BR>
     * オブジェクトの内容をクリアします.<BR>
     * このメソッドを呼び出した場合、内部のレコードは、クローズされません.
     */
    public void clear() {
        this.record = null ;
        this.tables = null ;
    }
    
    /**
     * レコード情報を取得.
     * <BR><BR>
     * レコード情報を取得します.
     * <BR>
     * @return Record レコード情報が返されます.
     */
    public Record getRecord() {
        return record ;
    }
    
    /**
     * 指定テーブル名のメタ情報を取得.
     * <BR><BR>
     * 指定テーブル名のメタ情報を取得します.
     * <BR>
     * @param name 対象のテーブル名を設定します.
     * @return MetaColumn 対象のメタカラムが返されます.
     * @exception MaachangDaoExceotion MaachangDao例外.
     */
    public synchronized MetaColumn getMeta( String name )
        throws MaachangDaoException {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new MaachangDaoException( "引数は不正です" ) ;
        }
        try {
            MetaColumn meta = null ;
            if( tables == null || ( meta = tables.get( name ) ) == null ) {
                if( tables == null ) {
                    tables = new HashMap<String,MetaColumn>() ;
                }
                meta = this.record.getMetaColumn( DbUtil.convertJavaNameByDBName( name ) ) ;
                if( meta == null ) {
                    throw new MaachangDaoException( "指定テーブル名[" + name + "]は存在しません" ) ;
                }
                tables.put( name,meta ) ;
            }
            return meta ;
        } catch( Exception e ) {
            throw new MaachangDaoException( e ) ;
        }
    }
}

