package org.maachang.comet.httpd.engine.script;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.script.ScriptContext;

import org.maachang.comet.httpd.engine.script.dao.MetaJson;
import org.maachang.comet.httpd.engine.script.image.ImageOp;
import org.maachang.dao.dbms.Record;

/**
 * モデル実行処理格納用.
 * 
 * @version 2007/08/27
 * @author masahito suzuki
 * @since MaachangComet 1.00
 */
public interface BaseModel {
    
    /**
     * 生成処理.
     * <BR><BR>
     * 条件を設定して生成します.
     * <BR>
     * @param mode [true]の場合、コンソール以外の実行モードです.
     * @param context 対象のスクリプトコンテキストを設定します.
     * @param record 対象のレコードオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public void create( boolean mode,ScriptContext context,Record record )
        throws Exception ;
    
    /**
     * クリア処理.
     */
    public void clear() ;
    
    /**
     * コミット処理.
     * <BR><BR>
     * コミット処理を実施します.
     * <BR>
     * @exception Exception 例外.
     */
    public void commit() throws Exception ;
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * ロールバック処理を実施します.
     * <BR>
     * @exception Exception 例外.
     */
    public void rollback() throws Exception ;
    
    /**
     * 生成チェック.
     * <BR><BR>
     * 生成チェック処理を行います.
     * <BR>
     * @return boolean [true]の場合生成されています.
     */
    public boolean isCreate() ;
    
    /**
     * オブジェクトタイプを取得.
     * <BR><BR>
     * オブジェクトタイプを取得します.
     * <BR>
     * @return boolean [true]の場合、実行可能オブジェクトです.
     */
    public boolean isType() ;
    
    /**
     * スクリプトコンテキストを取得.
     * <BR><BR>
     * スクリプトコンテキストを取得します.
     * <BR>
     * @return ScriptContext 対象のスクリプトコンテキストが返されます.
     */
    public ScriptContext getScriptContext() ;
    
    /**
     * イメージ操作オブジェクトを取得.
     * <BR><BR>
     * イメージ操作オブジェクトを取得します.
     * <BR>
     * @return ImageOp イメージ操作オブジェクトが返されます.
     */
    public ImageOp getImageOp() ;
    
    /**
     * コネクションオブジェクトを取得.
     * <BR><BR>
     * コネクションオブジェクトを取得します.
     * <BR>
     * @return Record コネクションオブジェクトが返されます.
     */
    public Record getRecord() ;
    
    /**
     * 指定モデル名から、メタデータを取得.
     * <BR><BR>
     * 指定モデル名から、メタデータを取得します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @return MetaJson 対象のメタデータが返されます.
     * @exception Exception 例外.
     */
    public MetaJson getMeta( String model )
        throws Exception ;
    
    /**
     * メタデータをJSON形式に変換.
     * <BR><BR>
     * メタデータをJSON形式に変換します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @return HashMap<String,Object> JSON形式のメタデータが返されます.
     * @exception Exception 例外.
     */
    public HashMap<String,Object> getMetaByJson( String model )
        throws Exception ;
    
    /**
     * データを保存.
     * <BR><BR>
     * スクリプトから渡されたデータを保存します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @param args 対象のデータを設定します.
     * @return Long 更新されたIDが返されます.
     * @exception Exception 例外.
     */
    public Long save( String model,HashMap<String,Object> args )
        throws Exception ;
    
    /**
     * データをアップデート.
     * <BR><BR>
     * スクリプトから渡されたデータを更新します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @param where 対象の条件を設定します.
     * @param args 対象のデータを設定します.
     * @exception Exception 例外.
     */
    public void update( String model,String where,ArrayList<Object> args )
        throws Exception ;
    
    /**
     * データを削除.
     * <BR><BR>
     * スクリプトから渡されたデータを削除します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @param im イメージマネージャを設定します.
     * @param delJoins 削除Joing群を設定します.
     * @param srcJoin 連結テーブル内容を設定します.
     * @param where 対象の条件を設定します.
     * @param args 対象のデータを設定します.
     * @exception Exception 例外.
     */
    public void remove( String model,ImageOp im,ArrayList<String> delJoins,
        Map<String,List<String>> srcJoin,String where,ArrayList<Object> args )
        throws Exception ;
    
    /**
     * データを削除.
     * <BR><BR>
     * スクリプトから渡されたデータを削除します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @param im イメージマネージャを設定します.
     * @param delJoins 削除Joing群を設定します.
     * @param srcJoin 連結テーブル内容を設定します.
     * @param args 対象のデータを設定します.
     * @exception Exception 例外.
     */
    public void removeData( String model,ImageOp im,ArrayList<String> delJoins,
        Map<String,List<String>> srcJoin,HashMap<String,Object> args )
        throws Exception ;
    
    /**
     * データを取得.
     * <BR><BR>
     * データを取得します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @param join 連結テーブル内容を設定します.
     * @param srcJoin 対象のsrcJoin条件を設定します.
     * @param where 対象の条件を設定します.
     * @param args 条件に対するパラメータを設定します.
     * @return List<Map<String,Object>> 処理結果が返されます.
     * @exception Exception 例外.
     */
    public List<Map<String,Object>> find( String model,Map<String,String> join,
        Map<String,List<String>> srcJoin,String where,ArrayList<Object> args )
        throws Exception ;
    
    /**
     * データを取得.
     * <BR><BR>
     * データを取得します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @param join 連結テーブル内容を設定します.
     * @param srcJoin 対象のsrcJoin条件を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param limit 対象のリミット値を設定します.
     * @param where 対象の条件を設定します.
     * @param args 条件に対するパラメータを設定します.
     * @param options 条件に対するオプションを設定します.
     * @return List<Map<String,Object>> 処理結果が返されます.
     * @exception Exception 例外.
     */
    public List<Map<String,Object>> limit( String model,Map<String,String> join,
        Map<String,List<String>> srcJoin,int offset,int limit,String where,ArrayList<Object> args )
        throws Exception ;
    
    /**
     * データを取得.
     * <BR><BR>
     * データを取得します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @param join 連結テーブル内容を設定します.
     * @param srcJoin 対象のsrcJoin条件を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param limit 対象のリミット値を設定します.
     * @param where 対象の条件を設定します.
     * @param args 条件に対するパラメータを設定します.
     * @param options 条件に対するオプションを設定します.
     * @return List<Map<String,Object>> 処理結果が返されます.
     * @exception Exception 例外.
     */
    public List<Map<String,Object>> first( String model,Map<String,String> join,
        Map<String,List<String>> srcJoin,String where,ArrayList<Object> args )
        throws Exception ;
    
    /**
     * データ数を取得.
     * <BR><BR>
     * データ数を取得します.
     * <BR>
     * @param model 対象のモデル名を設定します.
     * @param where 対象の条件を設定します.
     * @param args 条件に対するパラメータを設定します.
     * @return int 処理結果が返されます.
     * @exception Exception 例外.
     */
    public int count( String model,String where,ArrayList<Object> args )
        throws Exception ;
    
    /**
     * SQL実行.
     * <BR><BR>
     * SQL文を実行します.
     * <BR>
     * @param sql 対象の更新系SQL文を設定します.
     * @return Object 処理結果が返されます.
     * @exception Exception 例外.
     */
    public Object sql( String sql ) throws Exception ;
}

