package org.maachang.dbm.service.client ;

import java.net.InetAddress;

import org.maachang.connector.ClientConnectorFactory;
import org.maachang.dbm.MDbm;
import org.maachang.dbm.MDbmOp;
import org.maachang.dbm.service.ProtocolDef;

/**
 * MDbmドライバーマネージャ.
 * <BR><BR>
 * MDbmサーバと接続するためのドライバーマネージャです.<BR>
 * このオブジェクトは、動作しているMDBMサーバと接続し操作する、
 * MDbmオブジェクト及びMDbmOpオブジェクトを生成するためのマネージャです.<BR>
 * 利用方法は、下記の通りです.<BR>
 * <BR>
 * <div style="border:1px solid gray; padding:5px;">
 * <pre>
 * <font color="green">// MDBサーバの接続先ホスト名.</font>
 * InetAddress addr = InetAddress.getByName( "server" ) ;
 * 
 * <font color="green">// MDBMオブジェクトを取得.</font>
 * MDbm dbm = MDbmDriverManager.getConnection( addr ) ;
 *
 * <font color="green">// MDBM操作オブジェクトを取得.</font>
 * MDbmOp op = MDbmDriverManager.getConnectionOp( addr ) ;
 * 
 * </pre>
 * </div>
 * <BR>
 * 
 * @version 2008/01/20
 * @author masahito suzuki
 * @since MaachangDBM 1.03
 */
public class MDbmDriverManager {
    
    /**
     * タイムアウト値.
     */
    private static final int TIMEOUT = 30000 ;
    
    /**
     * プーリング数.
     */
    private static final int POOL = 15 ;
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @return Mdbm MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbm getConnection( String addr )
        throws Exception {
        return getConnection( addr,-1 ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @return Mdbm MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbm getConnection( InetAddress addr )
        throws Exception {
        return getConnection( addr,-1 ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return Mdbm MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbm getConnection( String addr,int port )
        throws Exception {
        if( addr == null ) {
            addr = "127.0.0.1" ;
        }
        return getConnection( InetAddress.getByName( addr ),port ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return Mdbm MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbm getConnection( InetAddress addr,int port )
        throws Exception {
        if( addr == null ) {
            addr = InetAddress.getByName( "127.0.0.1" ) ;
        }
        if( port < 0 || port >= 65535 ) {
            port = ProtocolDef.BIND_PORT ;
        }
        return new MDbmConnection( ClientConnectorFactory.getPoolingConnector(
            ProtocolDef.PROTOCOL_HEADER,addr,port,TIMEOUT,POOL ) ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @return MdbmOp MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbmOp getConnectionOp( String addr )
        throws Exception {
        return new MDbmOpConnection( getConnection( addr,-1 ) ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @return MdbmOp MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbmOp getConnectionOp( InetAddress addr )
        throws Exception {
        return new MDbmOpConnection( getConnection( addr,-1 ) ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return MdbmOp MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbmOp getConnectionOp( String addr,int port )
        throws Exception {
        return new MDbmOpConnection( getConnection( addr,port ) ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return MdbmOp MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbmOp getConnectionOp( InetAddress addr,int port )
        throws Exception {
        return new MDbmOpConnection( getConnection( addr,port ) ) ;
    }
}

