package org.maachang.util;

/**
 * ランダム基本処理.
 * 
 * @version 2008/12/11
 * @author masahito suzuki
 * @since MaachangBase 1.09
 */
public abstract class AbstractBaseRandom implements BaseRandom {
    
    /**
     * ランダム係数を設定.
     * @param seed ランダム係数を設定します.
     */
    public abstract void setSeed(long seed) ;
    
    /**
     * 乱数バイナリを取得.
     * @param binary 格納先のバイナリを設定します.
     */
    public void nextBytes(byte[] binary) {
        int rnd ;
        int cnt = 0 ;
        int len = binary.length ;
        while( true ) {
            if( cnt + 4 > len ) {
                int last = len - cnt ;
                switch( last ) {
                    case 0 : break ;
                    case 1 :
                        rnd = nextInt() ;
                        binary[ cnt ] = ( byte )( rnd & 0x000000ff ) ;
                        break ;
                    case 2 :
                        rnd = nextInt() ;
                        binary[ cnt ] = ( byte )( rnd & 0x000000ff ) ;
                        cnt ++ ;
                        binary[ cnt ] = ( byte )( ( rnd & 0x0000ff00 ) >> 8 ) ;
                        break ;
                    case 3 :
                        rnd = nextInt() ;
                        binary[ cnt ] = ( byte )( rnd & 0x000000ff ) ;
                        cnt ++ ;
                        binary[ cnt ] = ( byte )( ( rnd & 0x0000ff00 ) >> 8 ) ;
                        cnt ++ ;
                        binary[ cnt ] = ( byte )( ( rnd & 0x00ff0000 ) >> 16 ) ;
                        break ;
                }
                break ;
            }
            else {
                rnd = nextInt() ;
                binary[ cnt ] = ( byte )( rnd & 0x000000ff ) ;
                cnt ++ ;
                binary[ cnt ] = ( byte )( ( rnd & 0x0000ff00 ) >> 8 ) ;
                cnt ++ ;
                binary[ cnt ] = ( byte )( ( rnd & 0x00ff0000 ) >> 16 ) ;
                cnt ++ ;
                binary[ cnt ] = ( byte )( ( rnd & 0xff000000 ) >> 24 ) ;
                cnt ++ ;
            }
        }
    }
    
    /**
     * 32ビット乱数を取得.
     * @return int 32ビット乱数が返されます.
     */
    public abstract int nextInt() ;
    
    /**
     * 32ビット乱数を取得.
     * @param n 取得対象最大値を設定します.
     * @return int 32ビット乱数が返されます.
     */
    public int nextInt(int n) {
        if (n <= 0) {
            throw new IllegalArgumentException("引数は不正です");
        }
        if ((n & -n) == n)  // i.e., n is a power of 2
            return (int)((n * (long)( nextInt() & 0x7fffffff )) >> 31);
        int bits, val;
        do {
            bits = ( nextInt() & 0x7fffffff ) ;
            val = bits % n;
        } while (bits - val + (n-1) < 0);
        return val;
    }
    
    /**
     * 64ビット乱数を取得.
     * @return long 64Bit乱数が返されます.
     */
    public long nextLong() {
        return ((long)(nextInt()) << 32) + nextInt();
    }
    
    /**
     * Boolean乱数を取得.
     * @return boolean Boolean乱数が返されます.
     */
    public boolean nextBoolean() {
        return ( nextInt() & 0x00000001 ) != 0;
    }
    
    /**
     * Float乱数を取得.
     * @return float Float乱数が返されます.
     */
    public float nextFloat() {
        return ( nextInt() & 0x00ffffff ) / ((float)(1 << 24));
    }
    
    /**
     * Double乱数を取得.
     * @return double Double乱数が返されます.
     */
    public double nextDouble() {
        return (((long)( nextInt() & 0x03ffffff ) << 27) +
            ( nextInt() & 0x07ffffff )) / (double)(1L << 53);
    }
}

