package org.maachang.util;

import org.maachang.util.atomic.AtomicINT;

/**
 * ID発行オブジェクト(同期).
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaachangBase 1.00
 */
public class SequenceSync extends Sequence {

    /**
     * MAX-ID.
     */
    protected int maxId = 0;

    /**
     * 開始値.
     */
    protected int startId = 0;

    /**
     * 管理ID.
     */
    protected final AtomicINT id = new AtomicINT( -1 );

    /**
     * コンストラクタ. <BR>
     * <BR>
     * 任意の最大値を用いて情報を生成します.
     */
    public SequenceSync() {
        this.create();
    }

    /**
     * コンストラクタ. <BR>
     * <BR>
     * 情報を生成します. <BR>
     * 
     * @param start
     *            対象の開始値を設定します.
     * @param max
     *            管理を行うIDの幅(MAX値)を設定します.
     */
    public SequenceSync(int start, int max) {

        try {
            this.create(start, max);
        } catch (Exception e) {
        }

    }

    /**
     * ファイナライズ処理定義. <BR>
     * <BR>
     * ファイナライズ処理定義.
     * 
     * @exception Exception
     *                例外処理が返されます.
     */
    protected void finalize() throws Exception {
        this.clear();
    }

    /**
     * 情報生成. <BR>
     * <BR>
     * 情報を生成します.
     */
    public void create() {
        try {
            this.create(0, Integer.MAX_VALUE);
        } catch (Exception e) {
        }
    }

    /**
     * 情報生成. <BR>
     * <BR>
     * 情報を生成します. <BR>
     * 
     * @param start
     *            対象の開始値を設定します.
     * @param max
     *            管理を行うIDの幅(MAX値)を設定します.
     * @exception InputException
     *                入力例外.
     */
    public void create(int start, int max) throws Exception {

        if (start < 0 || max <= 1 || start >= (max - 1)
                || max > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("引数は不正です");
        }
        this.clear();
        this.maxId = max;
        this.startId = start;
        this.id.set( start );
    }

    /**
     * 情報クリア. <BR>
     * <BR>
     * 情報をクリアします.
     */
    public void clear() {
        this.id.set( this.startId );
    }

    /**
     * 一意なID項番を取得. <BR>
     * <BR>
     * 一意なID項番を取得します. <BR>
     * 
     * @return int 一意なID項番が返されます.
     */
    public int getId() {
        int ret = this.id.inc() ;
        if (ret >= this.maxId) {
            this.id.set( this.startId ) ;
            ret = this.startId ;
        }
        return ret ;
    }

    /**
     * 設定ID開始番号の取得. <BR>
     * <BR>
     * 設定されているID開始番号を取得します. <BR>
     * 
     * @return int 設定されているID開始番号が返されます.
     */
    public int getStartId() {
        return this.startId ;
    }

    /**
     * 設定ID幅(MAX値)を取得します. <BR>
     * <BR>
     * 設定されているID幅(MAX値)を取得します. <BR>
     * 
     * @return int 設定されているID幅(MAX値)が返されます.
     */
    public int getMaxID() {
        return this.maxId ;
    }
}
