package org.maachang.util.atomic ;

import java.io.Serializable;
import java.util.concurrent.locks.ReentrantLock;

import org.maachang.util.OList;

/**
 * (Read)AtomicListオブジェクト.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaachangBase 1.09
 */
public class ReadAtomicList implements OList,Serializable {
    private static final long serialVersionUID = -7622272629442735296L;
    
    /**
     * 配列管理.
     */
    private final AtomicOBJECT<AtomicARRAY<Object>> array = new AtomicOBJECT<AtomicARRAY<Object>>() ;
    
    /**
     * 開始時管理配列数.
     */
    private final AtomicINT startLength = new AtomicINT( DEFAULT_START_LENGTH ) ;
    
    /**
     * 配列管理数.
     */
    private final AtomicINT length = new AtomicINT( DEFAULT_START_LENGTH ) ;
    
    /**
     * 現在格納配列数.
     */
    private final AtomicINT nowLength = new AtomicINT( 0 ) ;
    
    /**
     * 同期.
     */
    private final ReentrantLock sync = new ReentrantLock() ;
    
    /**
     * コンストラクタ.
     */
    public ReadAtomicList() {
    }
    
    /**
     * コンストラクタ. <BR>
     * <BR>
     * 対象のバッファ係数を設定します. <BR>
     * 
     * @param length
     *            対象のバッファ係数を設定します.<BR>
     *            設定可能な最大値は[9999]です.<BR>
     *            設定可能な最小値は[2]です.<BR>
     *            また、これらの設定範囲外を設定した場合、 デフォルトの値[16]となります.
     */
    public ReadAtomicList(int length) {
        length = (length < MIN_START_LENGTH || length > MAX_START_LENGTH) ?
            DEFAULT_START_LENGTH : length ;
        this.startLength.set( length ) ;
        this.length.set( length ) ;
    }
    
    /**
     * ファイナライズ処理定義. <BR>
     * <BR>
     * ファイナライズ処理定義.
     * 
     * @exception Exception
     *                例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        try {
            this.clear();
        } catch (Exception t) {
        }
    }
    
    /**
     * 情報クリア. <BR>
     * <BR>
     * 対象の情報をクリアします.
     */
    public final void clear() {
        sync.lock() ;try {
            this.array.set( null ) ;
            this.length.set( this.startLength.get() ) ;
            this.nowLength.set( 0 ) ;
        } finally { sync.unlock() ;}
    }
    
    /**
     * 情報追加. <BR>
     * <BR>
     * 対象の情報を追加します. <BR>
     * 
     * @param value
     *            設定対象のオブジェクト情報を追加します.
     */
    public final void add(final Object value) {
        sync.lock() ;try {
            int length;
            int nowSize;
            AtomicARRAY<Object> info = null;
            AtomicARRAY<Object> tmp = null;
            info = this.array.get() ;
            length = this.length.get() ;
            nowSize = this.nowLength.get() ;
            if (info == null) {
                info = new AtomicARRAY<Object>( length ) ;
                info.set( value,nowSize ) ;
                this.array.set( info ) ;
                this.nowLength.inc() ;
            } else if (info.size() <= nowSize) {
                length *= 2;
                tmp = info;
                info = new AtomicARRAY<Object>( length ) ;
                arraycopy(tmp, 0, info, 0, nowSize);
                info.set( value,nowSize ) ;
                this.length.set( length ) ;
                this.array.set( info ) ;
                this.nowLength.inc() ;
            } else {
                info.set( value,nowSize ) ;
                this.nowLength.inc() ;
            }
        } finally { sync.unlock() ;}
    }
    
    /**
     * 情報設定. <BR>
     * <BR>
     * 対象の位置に情報をセットします. <BR>
     * 
     * @param no
     *            設定対象項番を設定します.
     * @param value
     *            設定対象情報を設定します.
     */
    public final void set(final int no, final Object value) {
        sync.lock() ;try {
            int nowLen;
            AtomicARRAY<Object> info ;
            nowLen = this.nowLength.get() ;
            info = this.array.get() ;
            if (no < 0 || no >= nowLen) {
                return;
            }
            info.set( value,no ) ;
        } finally { sync.unlock() ;}
    }
    
    /**
     * 情報削除. <BR>
     * <BR>
     * 対象の情報を削除します. <BR>
     * 
     * @param no
     *            削除対象の項番を設定します.
     * @return Object 削除されたオブジェクト情報が返されます.<BR>
     *         情報が存在しない場合[null]が返されます.
     */
    public final Object remove(final int no) {
        sync.lock() ;try {
            int nowSize;
            int length;
            int newLength;
            AtomicARRAY<Object> info = null;
            AtomicARRAY<Object> tmp = null;
            Object ret = null;
            nowSize = this.nowLength.get() ;
            length = this.length.get() ;
            newLength = this.length.get() ;
            if (no < 0 || no >= nowSize || nowSize == 0) {
                return null;
            }
            info = this.array.get() ;
            ret = info.get( no ) ;
            info.set( null,no ) ;
            if (no == 0) {
                tmp = info;
                arraycopy(tmp, 1, info, 0, (nowSize - 1));
                info.set( null,nowSize - 1 ) ;
            } else if ((nowSize - no) != 1) {
                tmp = info;
                arraycopy(tmp, 0, info, 0, no);
                arraycopy(tmp, no + 1, info, no, nowSize - (no + 1));
                info.set( null,nowSize - 1 ) ;
            }
            nowSize--;
            if (nowSize == 0) {
                info = null;
                newLength = this.startLength.get() ;
            }
            else if ((length * REDUCTION_ARRAY) >= nowSize) {
                newLength = length / 2;
                tmp = new AtomicARRAY<Object>( newLength );
                arraycopy(info, 0, tmp, 0, newLength);
                info = tmp;
            }
            this.array.set( info ) ;
            this.nowLength.set( nowSize ) ;
            this.length.set( newLength ) ;
            info = null;
            tmp = null;
            return ret;
        } finally { sync.unlock() ;}
    }
    
    /**
     * 情報取得. <BR>
     * <BR>
     * 対象の情報を取得します. <BR>
     * 
     * @param no
     *            取得対象の項番を設定します.
     * @return Object 取得された情報が返されます.
     */
    public final Object get(final int no) {
        return this.array.get().get( no );
    }
    
    /**
     * 格納情報数の取得. <BR>
     * <BR>
     * 格納されている情報数を取得します. <BR>
     * 
     * @return int 格納されている情報数が返されます.
     */
    public final int size() {
        return this.nowLength.get() ;
    }
    
    /**
     * 文字列変換.
     * @return String 文字列変換された内容が返されます.
     */
    public final String toString() {
        StringBuilder buf = new StringBuilder() ;
        buf.append( "[ " ) ;
        int len = this.nowLength.get() ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( i != 0 ) {
                buf.append( ", " ) ;
            }
            Object o = this.array.get().get( i ) ;
            if( o == null ) {
                buf.append( "null@null" ) ;
            }
            else {
                buf.append( o ).append( "@" ).append( o.getClass().getName() ) ;
            }
        }
        return buf.append( " ]" ).toString() ;
    }
    
    private static final void arraycopy( AtomicARRAY<Object> src,int sp,AtomicARRAY<Object> dest,int dp,int len ) {
        for( int i = 0 ; i < len ; i ++ ) {
            dest.set( src.get( sp+i ),dp+i ) ;
        }
    }
}
