package org.maachang.util.nativeio;

import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * RandomI/O処理(Java).
 * 
 * @version 2008/11/29
 * @author  masahito suzuki
 * @since   MaachangBase-1.09
 */
class JavaRandomIO implements RandomIO {
    
    /**
     * ランダムアクセスファイル.
     */
    private RandomAccessFile fp = null ;
    
    /**
     * ファイル名.
     */
    private String name = null ;
    
    /**
     * コンストラクタ.
     */
    private JavaRandomIO() {}
    
    /**
     * コンストラクタ.
     * @param name 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public JavaRandomIO( String name ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        name = NativeIODefine.fullPath( name ) ;
        RandomAccessFile fp = new RandomAccessFile( name,"rwd" ) ;
        this.fp = fp ;
        this.name = name ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        close() ;
    }
    
    /**
     * クローズ処理.
     */
    public void close() {
        if( isOpen() ) {
            try {
                fp.close() ;
            } catch( Exception e ) {
            }
        }
        fp = null ;
        name = null ;
    }
    
    /**
     * 読み込み処理.
     * @param binary 読み込み先バイナリを設定します.
     * @param seek シーク位置を設定します.
     * @param offset 読み込みオフセット値を設定します.
     * @param length 読み込みデータ長を設定します.
     * @return int 読み込みデータ長が返されます.
     * @exception Exception 例外.
     */
    public int read( byte[] binary,long seek,int offset,int length )
        throws Exception {
        if( binary == null || binary.length <= 0 || seek <= -1L && offset <= -1 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( isOpen() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        length = ( length <= 0 ) ? binary.length - offset : length ;
        if( length <= 0 || length > binary.length ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        fp.seek( seek ) ;
        return fp.read( binary,offset,length ) ;
    }
    
    /**
     * 書込み処理.
     * @param binary 書込み先バイナリを設定します.
     * @param seek シーク位置を設定します.
     * @param offset 書込みオフセット値を設定します.
     * @param length 書込みデータ長を設定します.
     * @exception Exception 例外.
     */
    public void write( byte[] binary,long seek,int offset,int length )
        throws Exception {
        if( binary == null || binary.length <= 0 || seek <= -1L && offset <= -1 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( isOpen() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        length = ( length <= 0 ) ? binary.length - offset : length ;
        if( length <= 0 || length > binary.length ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        fp.seek( seek ) ;
        fp.write( binary,offset,length ) ;
    }
    
    /**
     * 現在のファイルサイズを取得.
     * @return long ファイルサイズが返されます.
     * @exception Exception 例外.
     */
    public long length() throws Exception {
        if( isOpen() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        return fp.length() ;
    }
    
    /**
     * ファイルサイズを設定.
     * @param length ファイルサイズを設定します.
     * @exception Exception 例外.
     */
    public void length( long length ) throws Exception {
        if( length <= -1L ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( isOpen() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        fp.setLength( length ) ;
    }
    
    /**
     * オープンファイル名を取得.
     * @return String オープンファイル名が返されます.
     */
    public String getName() {
        return name ;
    }
    
    /**
     * オブジェクトタイプを取得.
     * @return int オブジェクトタイプが返されます.
     */
    public int getType() {
        return TYPE_JAVA ;
    }
    
    /**
     * ファイルがオープン中かチェック.
     * @return boolean [true]の場合、オープンしています.
     */
    public boolean isOpen() {
        return ( name != null ) ;
    }
}
