package org.maachang.util.nativeio;

import java.io.IOException;

/**
 * RandomI/O処理(Native).
 * 
 * @version 2008/11/29
 * @author  masahito suzuki
 * @since   MaachangBase-1.09
 */
class NativeRandomIO implements RandomIO {
    
    /**
     * ファイルハンドル.
     */
    private long handle = -1L ;
    
    /**
     * ファイル名.
     */
    private String name = null ;
    
    /**
     * コンストラクタ.
     */
    private NativeRandomIO() {}
    
    /**
     * コンストラクタ.
     * @param name 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public NativeRandomIO( String name ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        name = NativeIODefine.fullPath( name ) ;
        long handle = NativeIO.open( 1,2,NativeIODefine.nativeString( name ) ) ;
        if( handle <= -1L ) {
            throw new IOException( "ファイルオープンに失敗しました" ) ;
        }
        this.handle = handle ;
        this.name = name ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        close() ;
    }
    
    /**
     * クローズ処理.
     */
    public void close() {
        if( isOpen() ) {
            NativeIO.close( handle ) ;
        }
        handle = -1L ;
        name = null ;
    }
    
    /**
     * 読み込み処理.
     * @param binary 読み込み先バイナリを設定します.
     * @param seek シーク位置を設定します.
     * @param offset 読み込みオフセット値を設定します.
     * @param length 読み込みデータ長を設定します.
     * @return int 読み込みデータ長が返されます.
     * @exception Exception 例外.
     */
    public int read( byte[] binary,long seek,int offset,int length )
        throws Exception {
        if( binary == null || binary.length <= 0 || seek <= -1L && offset <= -1 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( isOpen() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        length = ( length <= 0 ) ? binary.length - offset : length ;
        if( length <= 0 || length > binary.length ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        return NativeIO.read( handle,seek,binary,offset,length ) ;
    }
    
    /**
     * 書込み処理.
     * @param binary 書込み先バイナリを設定します.
     * @param seek シーク位置を設定します.
     * @param offset 書込みオフセット値を設定します.
     * @param length 書込みデータ長を設定します.
     * @exception Exception 例外.
     */
    public void write( byte[] binary,long seek,int offset,int length )
        throws Exception {
        if( binary == null || binary.length <= 0 || seek <= -1L && offset <= -1 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( isOpen() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        length = ( length <= 0 ) ? binary.length - offset : length ;
        if( length <= 0 || length > binary.length ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( NativeIO.write( handle,seek,binary,offset,length ) <= -1 ) {
            throw new IOException( "書込み処理に失敗しました" ) ;
        }
    }
    
    /**
     * 現在のファイルサイズを取得.
     * @return long ファイルサイズが返されます.
     * @exception Exception 例外.
     */
    public long length() throws Exception {
        if( isOpen() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        return NativeIO.getLength( handle ) ;
    }
    
    /**
     * ファイルサイズを設定.
     * @param length ファイルサイズを設定します.
     * @exception Exception 例外.
     */
    public void length( long length ) throws Exception {
        if( length <= -1L ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( isOpen() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        NativeIO.setLength( handle,length ) ;
    }
    
    /**
     * オープンファイル名を取得.
     * @return String オープンファイル名が返されます.
     */
    public String getName() {
        return name ;
    }
    
    /**
     * オブジェクトタイプを取得.
     * @return int オブジェクトタイプが返されます.
     */
    public int getType() {
        return TYPE_NATIVE ;
    }
    
    /**
     * ファイルがオープン中かチェック.
     * @return boolean [true]の場合、オープンしています.
     */
    public boolean isOpen() {
        return ( name != null ) ;
    }
}
