package org.maachang.comet.conf ;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.InputStreamReader;

import org.maachang.comet.MaachangDef;
import org.maachang.conf.Config;
import org.maachang.conf.ReadIni;
import org.maachang.util.FileUtil;
import org.maachang.util.atomic.AtomicLONG;
import org.maachang.util.atomic.AtomicOBJECT;

/**
 * mime.conf用読み込みファイル.
 *
 * @version 2007/08/19
 * @author  masahito suzuki
 * @since   MaachangComet 1.00
 */
public class MimeConfig {
    
    /**
     * GZIP圧縮モード.
     */
    public static final String OPTION_GZIP = "gzip." ;
    
    /**
     * 読み込みファイル名.
     */
    private static final String READ_FILE = MaachangDef.DIRECTORY_CONFIG+"mime.conf" ;
    
    /**
     * キャラクタセット.
     */
    private static final String CHARSET = "UTF8" ;
    
    /**
     * 基本セクション名.
     */
    private static final String BASE_SECTION = "mime-type" ;
    
    /**
     * 基本MimeType.
     */
    private static final String DEFAULT_MIME_TYPE = "application/octet-stream" ;
    
    /**
     * 読み込み対象データ.
     */
    private final AtomicOBJECT<Config> iniParams = new AtomicOBJECT<Config>() ;
    
    /**
     * 最終更新日.
     */
    private final AtomicLONG lastUpdate = new AtomicLONG( -1L ) ;
    
    /**
     * 対象ファイル.
     */
    private final AtomicOBJECT<String> fileName = new AtomicOBJECT<String>() ;
    
    /**
     * コンストラクタ.
     */
    public MimeConfig() {
        
    }
    
    /**
     * オープン.
     * <BR><BR>
     * オープンします.
     * <BR>
     * @exception Exception 例外.
     */
    public void open() throws Exception {
        this.open( READ_FILE ) ;
    }
    
    /**
     * オープン.
     * <BR><BR>
     * オープンします.
     * <BR>
     * @param name 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public void open( String name ) throws Exception {
        this.fileName.set( FileUtil.getFullPath( name ) ) ;
        try {
            this.reload() ;
        } catch( Exception e ) {
            this.close() ;
            throw e ;
        } finally {
        }
    }
    
    /**
     * クローズ.
     * <BR><BR>
     * クローズ処理.
     */
    public void close() {
        iniParams.set( null ) ;
        lastUpdate.set( -1L ) ;
        fileName.set( null ) ;
    }
    
    /**
     * 指定名の条件を取得.
     * <BR><BR>
     * 指定名の条件を取得します.
     * <BR>
     * @param name 対象の名前を設定します.
     * @return String 対象の条件が返されます.
     */
    public String get( String name ) {
        Config c = getConfig() ;
        if( c == null ) {
            return DEFAULT_MIME_TYPE ;
        }
        String ret = c.get( BASE_SECTION,name,0 ) ;
        if( ret == null ) {
            ret = DEFAULT_MIME_TYPE ;
        }
        return ret ;
    }
    
    /**
     * 指定ファイル拡張子から、MimeTypeを取得.
     * <BR><BR>
     * 指定ファイル名を設定して、Mimeタイプを取得します.
     * <BR>
     * @param name 対象名を設定します.
     * @return String 対象MimeTypeが返されます.
     */
    public String getNameByMimeType( String name ) {
        Config c = getConfig() ;
        if( c == null ) {
            return DEFAULT_MIME_TYPE ;
        }
        String plus = getPlus( name ) ;
        if( plus == null ) {
            plus = name ;
        }
        String ret = c.get( BASE_SECTION,plus,0 ) ;
        if( ret == null ) {
            ret = DEFAULT_MIME_TYPE ;
        }
        return ret ;
    }
    
    /** 再読み込みしたCONFIG情報を取得. **/
    private Config getConfig() {
        try {
            long tm = FileUtil.getLastTime( fileName.get() ) ;
            if( lastUpdate.get() != tm ) {
                this.reload() ;
            }
        } catch( Exception e ) {
        }
        return iniParams.get() ;
    }
    
    /**
     * 再読み込み.
     */
    private final void reload()
        throws Exception {
        BufferedReader buf = null ;
        String fname = fileName.get() ;
        if( FileUtil.isFileExists( fname ) == true ) {
            try {
                buf = new BufferedReader(
                    new InputStreamReader(
                        new FileInputStream( fname ),CHARSET ) ) ;
                Config conf = new Config() ;
                ReadIni.analisys( conf,buf ) ;
                buf.close() ;
                buf = null ;
                long last = FileUtil.getLastTime( fname ) ;
                // 読み込みデータを設定.
                this.iniParams.set( conf ) ;
                this.lastUpdate.set( last ) ;
            } catch( Exception e ) {
                throw e ;
            } finally {
                if( buf != null ) {
                    try {
                        buf.close() ;
                    } catch( Exception e ) {
                    }
                }
            }
        }
    }
    
    /**
     * 拡張子を取得.
     */
    private static final String getPlus( String name ) {
        int p = name.lastIndexOf( "." ) ;
        if( p == -1 ) {
            return null ;
        }
        return name.substring( p+1,name.length() ) ;
    }
    
    /**
     * 文字列に変換.
     */
    public String toString() {
        Config c = getConfig() ;
        if( c != null ) {
            return c.toString() ;
        }
        return "null" ;
    }
}

