package org.maachang.comet.httpd.engine.script.cache;

import java.util.Map;

import javax.script.Bindings;
import javax.script.Compilable;
import javax.script.CompiledScript;
import javax.script.ScriptEngine;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.comet.MaachangDef;
import org.maachang.comet.httpd.engine.script.DummyBaseModel;
import org.maachang.comet.httpd.engine.script.ScriptDef;
import org.maachang.comet.httpd.engine.script.SrcScript;
import org.maachang.comet.httpd.engine.script.SrcScriptList;
import org.maachang.comet.httpd.engine.script.js.JsDef;
import org.maachang.jsr.script.api.ApiManager;
import org.maachang.jsr.script.api.ExternalBindings;
import org.maachang.util.FileUtil;
import org.maachang.util.atomic.AtomicBOOL;
import org.maachang.util.atomic.AtomicINT;
import org.maachang.util.atomic.AtomicLONG;
import org.maachang.util.atomic.AtomicOBJECT;

/**
 * ライブラリAPIマネージャ.
 * 
 * @version 2008/08/11
 * @author masahito suzuki
 * @since MaachangComet 1.23
 */
class LibApiManager {
    
    /**
     * LOG.
     */
    private static final Log LOG = LogFactory.getLog( LibApiManager.class ) ;
    
    /**
     * スクリプトエンジン名.
     */
    protected static final String ENGINE_NAME = "js" ;
    
    /**
     * スクリプト実行名.
     */
    protected static final String SCRIPT_NAME = "[lib_api]" ;
    
    /**
     * 読み込み対象ディレクトリ.
     */
    private static final String READ_DIR = baseDir() ;
    
    /**
     * 再読み込み確認時間.
     */
    private static final long CHECK_TIME = 30000L ;
    
    /**
     * 前回再読み込み時間.
     */
    private final AtomicLONG beforeTime = new AtomicLONG( -1L ) ;
    
    /**
     * ライブラリ管理コード.
     */
    private final AtomicOBJECT<String> libCode = new AtomicOBJECT<String>()  ;
    
    /**
     * スクリプト内容.
     */
    private final AtomicOBJECT<SrcScript> srcScript = new AtomicOBJECT<SrcScript>() ;
    
    /**
     * デフォルトパッケージ更新ID.
     */
    private final AtomicINT packageId = new AtomicINT( -1 ) ;
    
    /**
     * 再読み込みチェックフラグ.
     */
    private final AtomicBOOL reloadFlag = new AtomicBOOL( false ) ;
    
    /**
     * コンストラクタ.
     */
    public LibApiManager() throws Exception {
        try {
            loadScript() ;
        } catch( Exception e ) {
            LOG.error( "## [libApi] loadScriptError",e ) ;
            throw e ;
        }
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.clear() ;
    }
    
    /**
     * 基本ディレクトリを取得.
     */
    private static final String baseDir() {
        try {
            return FileUtil.getFullPath( MaachangDef.DIRECTORY_LIB ) + FileUtil.FILE_SPACE ;
        } catch( Exception e ) {
        }
        return null ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    protected void clear() {
        reloadFlag.set( false ) ;
        beforeTime.set( -1L ) ;
        libCode.set( null ) ;
        srcScript.set( null ) ;
        packageId.set( -1 ) ;
    }
    
    /**
     * 再読み込み処理.
     * @exception Exception 例外.
     */
    public void reload() throws Exception {
        loadScript() ;
    }
    
    /**
     * スクリプトキャッシュを有効にする.
     * この処理は、スクリプト呼び出しの前に必ず実行する必要があります.
     * @exception Exception 例外.
     */
    public void useScript() throws Exception {
        boolean reloadCheck = false ;
        try {
            if( reloadFlag.get() ) {
                reloadFlag.set( false ) ;
                reloadCheck = true ;
                String cd = libCode.get() ;
                if( cd == null || cd.equals( createLibMonitor() ) == false ) {
                    loadScript() ;
                    return ;
                }
            }
            int pid = this.packageId.get() ;
            if( pid != -1 && pid != JsDef.getDefaultPackageId() ) {
                loadScript() ;
                return ;
            }
            if( beforeTime.get() + CHECK_TIME <= System.currentTimeMillis() ) {
                String cd = libCode.get() ;
                if( reloadCheck == false &&
                    ( cd == null || cd.equals( createLibMonitor() ) == false ) ) {
                    loadScript() ;
                }
                else {
                    beforeTime.set( System.currentTimeMillis() ) ;
                }
            }
        } catch( Exception e ) {
            throw e ;
        }
    }
    
    /**
     * スクリプト内容を取得.
     * <BR><BR>
     * スクリプト内容を取得します.
     * <BR>
     * @return SrcScript スクリプト内容が返されます.
     */
    public SrcScript getSrcScript() {
        return srcScript.get() ;
    }
    
    /**
     * 指定パスがこのキャッシュ内の条件であるかチェック.
     * <BR><BR>
     * 指定パスがこのキャッシュ内の条件であるかチェックします.
     * <BR>
     * @param path チェック対象のパスを設定します.
     * @return boolean [true]の場合はこのキャッシュ内の条件です.
     */
    public boolean isCache( String path ) {
        if( path == null || ( path = path.trim() ).length() <= 0 ||
            path.equals( SCRIPT_NAME ) == false ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * リロードフラグを設定.
     * @param reloadFlag [true]の場合、再読み込みチェックを行います.
     */
    public void setReloadFlag( boolean reloadFlag ) {
        this.reloadFlag.set( reloadFlag ) ;
    }
    
    /**
     * ディレクトリ名を取得.
     * @return String ディレクトリ名が返されます.
     */
    public String getDirectory() {
        return MaachangDef.DIRECTORY_LIB ;
    }
    
    /**
     * スクリプト内容を読み込む.
     */
    private void loadScript() throws Exception {
        int cnt = 0 ;
        String beforeManagerCode = this.libCode.get() ;
        int beforePackageId = this.packageId.get() ;
        long beforeBeforeTime = this.beforeTime.get() ;
        try {
            // モデル内容を読み込み.
            String[] libs = ScriptDef.getUseScript( READ_DIR ) ;
            int startPos = 0 ;
            // 対象となるモデル内容が存在する場合.
            if( libs != null && libs.length > 0 ) {
                StringBuilder buf = new StringBuilder() ;
                // 更新された内容を一時保存.
                int[] outLine = new int[ 1 ] ;
                this.libCode.set( ScriptDef.getScriptManagerCode( libs,READ_DIR ) ) ;
                this.packageId.set( JsDef.pushDefaultPackage( outLine,buf ) ) ;
                this.beforeTime.set( System.currentTimeMillis() ) ;
                
                startPos = outLine[ 0 ] ;
                outLine = null ;
                
                // モデル内容を読み込む.
                int len = libs.length ;
                if( this.srcScript.get() == null ) {
                    this.srcScript.set( new SrcScript() ) ;
                }
                else {
                    this.srcScript.get().clear() ;
                }
                for( int i = 0 ; i < len ; i ++ ) {
                    String s = FileUtil.getFileByString( READ_DIR+"/"+libs[i],"UTF8" ) ;
                    if( s != null ) {
                        if( LOG.isDebugEnabled() ) {
                            LOG.debug( ">read[lib] - " + libs[i] ) ;
                        }
                        if( cnt != 0 ) {
                            buf.append( "\n" ) ;
                            startPos ++ ;
                        }
                        // ファイル名のコメントを付加.
                        buf.append( "/** libName << " ).append( libs[ i ] ).append( " >> **/\n" ) ;
                        startPos ++ ;
                        // ソースリストを追加.
                        int targetSize = SrcScript.getScriptList( s ) ;
                        SrcScriptList srcLst = new SrcScriptList( libs[ i ],startPos,targetSize ) ;
                        this.srcScript.get().setSrcScriptList( srcLst ) ;
                        buf.append( s ) ;
                        startPos += targetSize ;
                        cnt ++ ;
                    }
                }
                String script = buf.toString() ;
                buf = null ;
                this.srcScript.get().create( script ) ;
                
                // 処理実行.
                Bindings bindings = ExternalBindings.getInstance() ;
                // dummyデータ設定.
                bindings.put( ScriptDef.SCRIPT_BY_MODEL,new DummyBaseModel() ) ;
                
                // エンジン実行.
                ScriptEngine engine = ApiManager.getInstance().getScriptEngine() ;
                ApiManager.setScriptName( SCRIPT_NAME ) ;
                CompiledScript cs = ( ( Compilable )engine ).compile( script ) ;
                cs.eval( bindings ) ;
                cs = null ;
                
                // APIマネージャに登録.
                Map<String,Object> libManager = ApiManager.getLocal() ;
                bindings.clear() ;
                ApiManager.getInstance().setLibMap( libManager ) ;
                if( LOG.isDebugEnabled() ) {
                    LOG.debug( "** lib-cacheを読み込み" ) ;
                }
            }
            // 対象となるライブラリ内容が存在しない場合は、内容をクリアする.
            if( cnt <= 0 ) {
                this.libCode.set( null ) ;
                this.packageId.set( -1 ) ;
                ApiManager.getInstance().setLibMap( null ) ;
                this.beforeTime.set( System.currentTimeMillis() ) ;
            }
            this.reloadFlag.set( false ) ;
        } catch( Exception e ) {
            // 例外の場合は、ロールバック処理.
            this.libCode.set( beforeManagerCode ) ;
            this.packageId.set( beforePackageId ) ;
            this.beforeTime.set( beforeBeforeTime ) ;
            throw e ;
        } finally {
            ApiManager.removeLocal() ;
        }
    }
    
    /**
     * ライブラリ管理コードを生成.
     */
    private String createLibMonitor()
        throws Exception {
        if( LOG.isDebugEnabled() ) {
            LOG.debug( "## read LibCode" ) ;
        }
        String[] libs = ScriptDef.getUseScript( READ_DIR ) ;
        return ScriptDef.getScriptManagerCode( libs,READ_DIR ) ;
    }
}

