package org.maachang.comet.httpd.engine.script.cache;

import java.util.Map;

import javax.script.Bindings;
import javax.script.Compilable;
import javax.script.CompiledScript;
import javax.script.ScriptEngine;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.comet.MaachangDef;
import org.maachang.comet.ServiceDef;
import org.maachang.comet.httpd.engine.script.DummyBaseModel;
import org.maachang.comet.httpd.engine.script.ScriptDef;
import org.maachang.comet.httpd.engine.script.SrcScript;
import org.maachang.comet.httpd.engine.script.SrcScriptList;
import org.maachang.comet.httpd.engine.script.dao.DaoFactory;
import org.maachang.comet.httpd.engine.script.js.JsDef;
import org.maachang.comet.httpd.engine.script.scripts.ScriptManager;
import org.maachang.jsr.script.api.ApiManager;
import org.maachang.jsr.script.api.ExternalBindings;
import org.maachang.manager.GlobalManager;
import org.maachang.util.FileUtil;
import org.maachang.util.atomic.AtomicBOOL;
import org.maachang.util.atomic.AtomicINT;
import org.maachang.util.atomic.AtomicLONG;
import org.maachang.util.atomic.AtomicOBJECT;

/**
 * モデルAPIマネージャ.
 * 
 * @version 2008/08/11
 * @author masahito suzuki
 * @since MaachangComet 1.23
 */
class ModelApiManager {
    
    /**
     * LOG.
     */
    private static final Log LOG = LogFactory.getLog( ModelApiManager.class ) ;
    
    /**
     * スクリプトエンジン名.
     */
    protected static final String ENGINE_NAME = "js" ;
    
    /**
     * スクリプト実行名.
     */
    protected static final String SCRIPT_NAME = "[model_api]" ;
    
    /**
     * モデルファイルフッダ名.
     */
    protected static final String MODEL_NAME_FOODER = "Model" ;
    
    /**
     * カスタムモデルスクリプト.
     */
    protected static final String MODEL_CUSTOM = "Custom" ;
    
    /**
     * 読み込み対象ディレクトリ.
     */
    private static final String READ_DIR = baseDir() ;
    
    /**
     * 再読み込み確認時間.
     */
    private static final long CHECK_TIME = 30000L ;
    
    /**
     * 前回再読み込み時間.
     */
    private final AtomicLONG beforeTime = new AtomicLONG( -1L ) ;
    
    /**
     * モデル管理コード.
     */
    private final AtomicOBJECT<String> modelCode = new AtomicOBJECT<String>() ;
    
    /**
     * スクリプト内容.
     */
    private final AtomicOBJECT<SrcScript> srcScript = new AtomicOBJECT<SrcScript>() ;
    
    /**
     * デフォルトパッケージ更新ID.
     */
    private final AtomicINT packageId = new AtomicINT( -1 ) ;
    
    /**
     * 再読み込みチェックフラグ.
     */
    private final AtomicBOOL reloadFlag = new AtomicBOOL( false ) ;
    
    /**
     * コンストラクタ.
     */
    public ModelApiManager() throws Exception {
        try {
            if( GlobalManager.getValue( ServiceDef.MANAGER_BY_DBMS_POOL ) != null ) {
                loadScript() ;
            }
        } catch( Exception e ) {
            LOG.error( "## [modelApi] loadScriptError",e ) ;
            throw e ;
        }
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.clear() ;
    }
    
    /**
     * 基本ディレクトリを取得.
     */
    private static final String baseDir() {
        try {
            return FileUtil.getFullPath( MaachangDef.DIRECTORY_MODEL ) + FileUtil.FILE_SPACE ;
        } catch( Exception e ) {
        }
        return null ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    protected void clear() {
        beforeTime.set( -1L ) ;
        modelCode.set( null ) ;
        srcScript.set( null ) ;
        packageId.set( -1 ) ;
        reloadFlag.set( false ) ;
    }
    
    /**
     * 再読み込み処理.
     * @exception Exception 例外.
     */
    public void reload() throws Exception {
        loadScript() ;
    }
    
    /**
     * スクリプトキャッシュを有効にする.
     * この処理は、スクリプト呼び出しの前に必ず実行する必要があります.
     * @exception Exception 例外.
     */
    public void useScript() throws Exception {
        boolean reloadCheck = false ;
        long time = System.currentTimeMillis() ;
        try {
            if( reloadFlag.get() ) {
                reloadCheck = true ;
                reloadFlag.set( false ) ;
                String md = modelCode.get() ;
                if( md == null || md.equals( createModelMonitor() ) == false ) {
                    loadScript() ;
                    return ;
                }
            }
            if( GlobalManager.getValue( ServiceDef.MANAGER_BY_DBMS_POOL ) != null ) {
                int pid = this.packageId.get() ;
                if( pid != -1 && pid != JsDef.getDefaultPackageId() ) {
                    loadScript() ;
                    return ;
                }
                else if( beforeTime.get() + CHECK_TIME <= time ) {
                    String md = modelCode.get() ;
                    if( reloadCheck == false &&
                        ( md == null || md.equals( createModelMonitor() ) == false ) ) {
                        loadScript() ;
                    }
                    else {
                        beforeTime.set( System.currentTimeMillis() ) ;
                    }
                }
            }
        } catch( Exception e ) {
            throw e ;
        }
    }
    
    /**
     * スクリプト内容を取得.
     * <BR><BR>
     * スクリプト内容を取得します.
     * <BR>
     * @return SrcScript スクリプト内容が返されます.
     */
    public SrcScript getSrcScript() {
        return srcScript.get() ;
    }
    
    /**
     * 指定パスがこのキャッシュ内の条件であるかチェック.
     * <BR><BR>
     * 指定パスがこのキャッシュ内の条件であるかチェックします.
     * <BR>
     * @param path チェック対象のパスを設定します.
     * @return boolean [true]の場合はこのキャッシュ内の条件です.
     */
    public boolean isCache( String path ) {
        if( path == null || ( path = path.trim() ).length() <= 0 ||
            path.equals( SCRIPT_NAME ) == false ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * リロードフラグを設定.
     * @param reloadFlag [true]の場合、再読み込みチェックを行います.
     */
    public void setReloadFlag( boolean reloadFlag ) {
        this.reloadFlag.set( reloadFlag ) ;
    }
    
    /**
     * ディレクトリ名を取得.
     * @return String ディレクトリ名が返されます.
     */
    public String getDirectory() {
        return MaachangDef.DIRECTORY_MODEL ;
    }
    
    /**
     * スクリプト内容を読み込む.
     */
    private void loadScript() throws Exception {
        int cnt = 0 ;
        String beforeManagerCode = this.modelCode.get() ;
        int beforePackageId = this.packageId.get() ;
        long beforeBeforeTime = this.beforeTime.get() ;
        //ScriptContext ctx = null ;
        try {
            // Dao初期化.
            DaoFactory daoFactory = getDaoFactory() ;
            if( daoFactory != null ) {
                daoFactory.clear() ;
            }
            // モデル内容を読み込み.
            String[] models = ScriptDef.getUseScript( READ_DIR ) ;
            int startPos = 0 ;
            // 対象となるモデル内容が存在する場合.
            if( models != null && models.length > 0 ) {
                StringBuilder buf = new StringBuilder() ;
                // 更新された内容を一時保存.
                int[] outLine = new int[ 1 ] ;
                this.modelCode.set( ScriptDef.getScriptManagerCode( models,READ_DIR ) ) ;
                this.packageId.set( JsDef.pushDefaultPackage( outLine,buf ) ) ;
                this.beforeTime.set( System.currentTimeMillis() ) ;
                
                startPos = outLine[ 0 ] ;
                outLine = null ;
                
                // モデル内容を読み込む.
                int len = models.length ;
                if( this.srcScript.get() == null ) {
                    this.srcScript.set( new SrcScript() ) ;
                }
                else {
                    this.srcScript.get().clear() ;
                }
                // モデル群情報を確保.
                buf.append( "\nvar Models = {};\n" ) ;
                startPos += 2 ;
                for( int i = 0 ; i < len ; i ++ ) {
                    if( models[i].endsWith( ScriptManager.SCRIPT_PLUS ) ) {
                        models[i] = models[i].substring( 0,models[i].length()-ScriptManager.SCRIPT_PLUS.length() ) ;
                    }
                    if( models[i].endsWith( MODEL_CUSTOM ) ||
                        models[i].endsWith( MODEL_NAME_FOODER ) == false ) {
                        continue ;
                    }
                    else {
                        cnt ++ ;
                    }
                    // 本来のモデル名を取得して、付加処理.
                    String modelName = models[i].substring( 0,models[i].length()-MODEL_NAME_FOODER.length() ) ;
                    if( LOG.isDebugEnabled() ) {
                        LOG.debug( ">read[model] - " + modelName ) ;
                    }
                    startPos = putModel( modelName,buf,this.srcScript.get(),startPos ) ;
                }
                String script = buf.toString() ;
                buf = null ;
                this.srcScript.get().create( script ) ;
                
                // 処理実行.
                Bindings bindings = ExternalBindings.getInstance() ;
                // dummyデータ設定.
                bindings.put( ScriptDef.SCRIPT_BY_MODEL,new DummyBaseModel() ) ;
                
                // エンジン実行.
                ScriptEngine engine = ApiManager.getInstance().getScriptEngine() ;
                ApiManager.setScriptName( SCRIPT_NAME ) ;
                CompiledScript cs = ( ( Compilable )engine ).compile( script ) ;
                cs.eval( bindings ) ;
                cs = null ;
                
                // APIマネージャに登録.
                Map<String,Object> modelManager = ApiManager.getLocal() ;
                bindings.clear() ;
                ApiManager.getInstance().setModelMap( modelManager ) ;
                if( LOG.isDebugEnabled() ) {
                    LOG.debug( "** model-cacheを読み込み" ) ;
                }
            }
            // 対象となるモデル内容が存在しない場合は、内容をクリアする.
            if( cnt <= 0 ) {
                this.modelCode.set( null ) ;
                this.packageId.set( -1 ) ;
                ApiManager.getInstance().setModelMap( null ) ;
                this.beforeTime.set( System.currentTimeMillis() ) ;
            }
            this.reloadFlag.set( false ) ;
        } catch( Exception e ) {
            // 例外の場合は、ロールバック処理.
            this.modelCode.set( beforeManagerCode ) ;
            this.packageId.set( beforePackageId ) ;
            this.beforeTime.set( beforeBeforeTime ) ;
            throw e ;
        } finally {
            ApiManager.removeLocal() ;
        }
    }
    
    /**
     * Model管理コードを生成.
     */
    private String createModelMonitor()
        throws Exception {
        if( LOG.isDebugEnabled() ) {
            LOG.debug( "## read ModelCode" ) ;
        }
        String[] models = ScriptDef.getUseScript( READ_DIR ) ;
        return ScriptDef.getScriptManagerCode( models,READ_DIR ) ;
    }
    
    /**
     * Model内容を追加.
     */
    private int putModel( String model,StringBuilder buf,SrcScript srcScript,int startPos )
        throws Exception {
        buf.append( "\n\n" ) ;
        buf.append( "/** modelName << " ).append( model ).append( MODEL_NAME_FOODER ).append( ScriptManager.SCRIPT_PLUS ).append( " >> **/\n" ) ;
        buf.append( "var " ).append( model ).append( " = baseModel(\"" ).append( model ).append( "\");\n" ) ;
        buf.append( "Models[\"" ).append( model ).append( "\"] = " ).append( model ).append( ";\n" ) ;
        startPos += 5 ;
        String modelScript = FileUtil.getFileByString( READ_DIR+"/"+model+MODEL_NAME_FOODER+ScriptManager.SCRIPT_PLUS,"UTF8" ) ;
        if( modelScript != null && ( modelScript = modelScript.trim() ).length() > 0 ) {
            int targetSize = SrcScript.getScriptList( modelScript ) ;
            SrcScriptList srcLst = new SrcScriptList( model,startPos,targetSize ) ;
            this.srcScript.get().setSrcScriptList( srcLst ) ;
            buf.append( modelScript ).append( "\n" ) ;
            startPos += ( targetSize + 1 ) ;
        }
        modelScript = null ;
        // カスタムモデルスクリプトが存在する場合、それも含めて取り込む.
        String customName = new StringBuilder().append( model ).append( MODEL_CUSTOM ).append( ScriptManager.SCRIPT_PLUS ).toString() ;
        String custom = READ_DIR+"/"+customName ;
        if( FileUtil.isFileExists( custom ) == true ) {
            modelScript = FileUtil.getFileByString( custom,"UTF8" ) ;
            if( modelScript != null && ( modelScript = modelScript.trim() ).length() > 0 ) {
                buf.append( "/** modelName(Custom) << " ).append( customName ).append( " >> **/\n" ) ;
                startPos ++ ;
                int targetSize = SrcScript.getScriptList( modelScript ) ;
                SrcScriptList srcLst = new SrcScriptList( customName,startPos,targetSize ) ;
                this.srcScript.get().setSrcScriptList( srcLst ) ;
                buf.append( modelScript ).append( "\n" ) ;
                startPos += ( targetSize + 1 ) ;
            }
        }
        return startPos ;
    }
    
    /**
     * DaoFactoryを取得.
     */
    private DaoFactory getDaoFactory() {
        return ( DaoFactory )GlobalManager.getInstance().get( ServiceDef.DAO_FACTORY ) ;
    }
}
