package org.maachang.comet.net.nio ;

import java.io.IOException;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;

import org.maachang.comet.net.ssl.SslElement;
import org.maachang.comet.net.ssl.SslOption;
import org.maachang.util.atomic.AtomicBOOL;
import org.maachang.util.atomic.AtomicLONG;
import org.maachang.util.atomic.AtomicOBJECT;

/**
 * Nio受信データ要素.
 * 
 * @version 2008/05/28
 * @author  masahito suzuki
 * @since   MaachangComet 1.1B
 */
public class NioElement {
    
    /**
     * SelectionKey.
     */
    private final AtomicOBJECT<SelectionKey> key = new AtomicOBJECT<SelectionKey>() ;
    
    /**
     * SocketChannel.
     */
    private final AtomicOBJECT<SocketChannel> channel = new AtomicOBJECT<SocketChannel>() ;
    
    /**
     * 書き込み可能を示すバッファ.
     */
    private final AtomicOBJECT<IsWriteQueue> isWriteQueue = new AtomicOBJECT<IsWriteQueue>() ;
    
    /**
     * 最新時間.
     */
    private final AtomicLONG time = new AtomicLONG( -1L ) ;
    
    /**
     * 処理中フラグ.
     */
    private final AtomicBOOL executionFlag = new AtomicBOOL( false ) ;
    
    /**
     * クローズフラグ.
     */
    private final AtomicBOOL closeFlag = new AtomicBOOL( false ) ; ;
    
    /**
     * 付加オブジェクト.
     */
    private final AtomicOBJECT<Object> object = new AtomicOBJECT<Object>() ;
    
    /**
     * SSL要素.
     */
    private final AtomicOBJECT<SslElement> sslElement = new AtomicOBJECT<SslElement>() ;
    
    /**
     * SSLオプション.
     */
    private final AtomicOBJECT<SslOption> sslOption = new AtomicOBJECT<SslOption>() ;
    
    /**
     * 受信バッファ.
     */
    private final ReceiveBuffer recvBuffer = new ReceiveBuffer() ;
    
    /**
     * 送信バッファ.
     */
    private final SendBuffer sendBuffer = new SendBuffer() ;
    
    /**
     * コンストラクタ.
     */
    private NioElement() {
        
    }
    
    /**
     * コンストラクタ.
     * @param channel 対象のチャネル情報を設定します.
     * @exception Exception 例外.
     */
    public NioElement( SocketChannel channel ) throws Exception {
        this( channel,null ) ;
    }
    
    /**
     * コンストラクタ.
     * @param ch 対象のチャネル情報を設定します.
     * @param queue 書き込み通知キューを設定します.
     * @exception Exception 例外.
     */
    public NioElement( SocketChannel ch,IsWriteQueue queue ) throws Exception {
        if( ch == null) {
            throw new IOException( "ソケットチャネルは不正です" ) ;
        }
        this.channel.set( ch ) ;
        this.isWriteQueue.set( queue ) ;
        this.time.set( System.currentTimeMillis() ) ;
        this.executionFlag.set( false ) ;
        this.closeFlag.set( false ) ;
        this.object.set( null ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    protected void destroy() {
        if( sslElement.get() != null ) {
            sslElement.get().destroy() ;
        }
        sslElement.set( null ) ;
        if( key.get() != null ) {
            key.get().attach( null ) ;
            try {
                key.get().cancel() ;
            } catch( Exception e ) {
            }
        }
        key.set( null ) ;
        if( channel.get() != null ) {
            try {
                channel.get().socket().close() ;
            } catch( Exception e ) {
            }
            try {
                channel.get().close() ;
            } catch( Exception e ) {
            }
        }
        channel.set( null ) ;
        isWriteQueue.set( null ) ;
        object.set( null ) ;
        executionFlag.set( false ) ;
        closeFlag.set( false ) ;
        sslOption.set( null ) ;
        recvBuffer.destroy() ;
        sendBuffer.destroy() ;
    }
    
    /**
     * 書き込み開始キューを設定.
     * @param queue 書き込み開始キューを設定します.
     */
    protected void setIsWriteQueue( IsWriteQueue queue ) {
        isWriteQueue.set( queue ) ;
    }
    
    /**
     * SSL条件を追加.
     * @param sslOption 対象のSSLオプションを設定します.
     * @param sslElement 対象のSSL要素を追加します.
     */
    protected void setSsl( SslOption op ) {
        sslOption.set( op ) ;
        sslElement.set( null ) ;
    }
    
    /**
     * SSL要素を取得.
     * @return SslElement SSL要素が返されます.
     */
    public SslElement getSslElement() {
        return sslElement.get() ;
    }
    
    /**
     * SSL要素を取得.
     * @return SslElement SSL要素が返されます.
     */
    public SslOption getSslOption() {
        return sslOption.get() ;
    }
    
    /**
     * 新しいSSLオプションを生成.
     * @exception Exception 例外.
     */
    public void createSsl() throws Exception {
        if( sslOption.get() != null && sslElement.get() == null ) {
            try {
                sslElement.set( new SslElement(
                    this,sslOption.get().getSSLEngine() ) ) ;
            } catch( Exception e ) {
                throw e ;
            }
        }
    }
    
    /**
     * SSL要素をクリア.
     */
    public void clearSsl() {
        sslElement.set( null ) ;
    }
    
    /**
     * SelectionKeyを設定.
     * @param k SelectionKeyを設定します.
     */
    public void setKey( SelectionKey k ) {
        key.set( k ) ;
    }
    
    /**
     * SelectionKeyを取得.
     * @return SelectionKey SelectionKeyが返されます.
     */
    public SelectionKey getKey() {
        return key.get() ;
    }
    
    /**
     * SocketChannelを取得.
     * @return SocketChannel SocketChannelが返されます.
     */
    public SocketChannel getChannel() {
        return channel.get() ;
    }
    
    /**
     * 受信バッファを取得.
     * @return ReceiveBuffer 受信バッファを取得します.
     */
    public ReceiveBuffer getReceiveBuffer() {
        return recvBuffer ;
    }
    
    /**
     * 受信バッファが存在するかチェック.
     * @return boolean [true]の場合、存在します.
     */
    public boolean isReceiveBuffer() {
        return recvBuffer.isData() ;
    }
    
    /**
     * 書き込み開始/再開を通知.
     */
    public void restartWrite() {
        if( isWriteQueue.get() != null &&
            key.get() != null && ( key.get().interestOps() & SelectionKey.OP_WRITE ) == 0 ) {
            isWriteQueue.get().append( this ) ;
            key.get().selector().wakeup() ;
        }
    }
    
    /**
     * 書き込みバッファオブジェクトを取得.
     * @return SendBuffer 書き込みバッファオブジェクトが返されます.
     */
    public SendBuffer getSendBuffer() {
        return sendBuffer ;
    }
    
    /**
     * クローズフラグをセット.
     * @param mode [true]の場合、クローズとなります.
     */
    public void setCloseFlag( boolean mode ) {
        closeFlag.set( mode ) ;
    }
    
    /**
     * クローズフラグを取得.
     * @return boolean [true]の場合、クローズです.
     */
    public boolean isCloseFlag() {
        return closeFlag.get() ;
    }
    
    /**
     * 更新処理.
     */
    public void update() {
        time.set( System.currentTimeMillis() ) ;
    }
    
    /**
     * 最新時間を取得.
     * @return long 最新時間が返されます.
     */
    public long getTime() {
        return time.get() ;
    }
    
    /**
     * 処理中フラグを設定.
     * @param flg [true]の場合、処理中フラグをONにします.
     */
    protected void setExecutionFlag( boolean flg ) {
        executionFlag.set( flg ) ;
        if( flg == false ) {
            update() ;
        }
    }
    
    /**
     * 処理中フラグを取得.
     * @return boolean [true]の場合、処理中です.
     */
    protected boolean isExecutionFlag() {
        return executionFlag.get() ;
    }
    
    /**
     * 付加オブジェクトを設定.
     * @param o 付加オブジェクトを設定します.
     */
    public void setObject( Object o ) {
        object.set( o ) ;
    }
    
    /**
     * 付加オブジェクトを取得.
     * @return Object 付加オブジェクトが返されます.
     */
    public Object getObject() {
        return object.get() ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        boolean ret ;
        ret = ( channel.get() != null && channel.get().socket() != null ) ;
        if( ret == false ) {
            if( sslElement.get() != null ) {
                sslElement.get().destroy() ;
            }
            sslElement.set( null ) ;
            recvBuffer.destroy() ;
            sendBuffer.destroy() ;
        }
        return ret ;
    }
}
